!
! Copyright (C) 2016 Quantum ESPRESSO Foundation
! This file is distributed under the terms of the
! GNU General Public License. See the file `License'
! in the root directory of the present distribution,
! or http://www.gnu.org/copyleft/gpl.txt .
!
! Author: Mitsuaki Kawamura, U. Tokyo
!----------------------------------------------------------------------------
MODULE elph_scdft_mod
  !--------------------------------------------------------------------------
  !
  IMPLICIT NONE
  !
  PRIVATE
  !
  PUBLIC elph_scdft
  !
  CONTAINS
 !
!----------------------------------------------------------------------------
SUBROUTINE elph_scdft()
  !--------------------------------------------------------------------------
  !
  ! This routine computes the electron-phonon matrix
  ! in the irreducible Brillouin zone and
  ! expand that to whole BZ.
  !
  USE kinds, ONLY : dp
  USE mp_pools, ONLY : npool, inter_pool_comm, my_pool_id
  USE mp_images, ONLY : me_image
  USE io_global,   ONLY : stdout
  USE cell_base, ONLY : at
  USE ions_base, ONLY : nat
  USE start_k, ONLY: nk1, nk2, nk3
  USE qpoint, ONLY : xq, nksq
  USE dynmat, ONLY : dyn, w2
  USE el_phon, ONLY : el_ph_mat, elph_nbnd_min, elph_nbnd_max
  USE control_ph, ONLY : current_iq
  USE modes, ONLY : u
  USE noncollin_module, ONLY : nspin_lsda
  !
  INTEGER :: ik, ib, jb, ii, elph_unit, nrcv, ispin
  INTEGER :: nbnd_fs
  !
  COMPLEX(dp),ALLOCATABLE :: gep_col(:,:,:), gep(:,:,:,:)
  CHARACTER(100) :: elphname
  INTEGER, EXTERNAL :: find_free_unit
  !
  WRITE(stdout,*) "[elph_scdft]  write elph.dat with symmetries (only on Fermi surfaces)"
  !
  nbnd_fs = elph_nbnd_max - elph_nbnd_min + 1
  !
  WRITE(stdout,*) "[elph_scdft]   Lowest band which contains FS : ", elph_nbnd_min
  WRITE(stdout,*) "[elph_scdft]  Highest band which contains FS : ", elph_nbnd_max
  WRITE(stdout,*) "[elph_scdft]    # of bands which contains FS : ", nbnd_fs
  !
  ! Compute g in each pool
  !
  ALLOCATE(gep(3*nat, nbnd_fs, nbnd_fs, nksq))
  !
  gep(1:3 * nat, 1:nbnd_fs, 1:nbnd_fs, 1:nksq) = 0_dp
  !
  DO ik = 1, nksq
     !
     DO ib = 1, nbnd_fs
        DO jb = 1, nbnd_fs
           !
           gep(1:3*nat,jb,ib,ik) = MATMUL(CONJG(u(1:3*nat,1:3*nat)), &
           &   el_ph_mat(elph_nbnd_min - 1 + jb,elph_nbnd_min - 1 + ib,ik,1:3*nat))
           gep(1:3*nat,jb,ib,ik) = MATMUL(gep(1:3*nat,jb,ib,ik), dyn(1:3 * nat,1:3 * nat))
           !
        END DO ! jb
     END DO ! ib
     !
  END DO ! ik
  !
  DO ii = 1, 3 * nat
     IF(w2(ii) <= 0_dp) THEN
        gep(ii, 1:nbnd_fs, 1:nbnd_fs, 1:nksq) = 0_dp
     ELSE
        gep(ii,1:nbnd_fs,1:nbnd_fs,1:nksq) = gep(ii,1:nbnd_fs,1:nbnd_fs,1:nksq) &
        &                                  / SQRT(SQRT(w2(ii)) * 2.0_dp)
     END IF
  END DO
  !
  ! Gather El-Ph matrix inter pool
  !
  IF(my_pool_id == 0) THEN
     nrcv = 3 * nat * nbnd_fs * nbnd_fs * nk1*nk2*nk3 * nspin_lsda
     ALLOCATE(gep_col(3*nat*nbnd_fs*nbnd_fs, nk1*nk2*nk3, nspin_lsda))
  ELSE
     nrcv = 1
     ALLOCATE(gep_col(1,1,1))
  END IF
  !
  CALL elph_scdft_gather(gep, 3*nat*nbnd_fs*nbnd_fs*nksq, gep_col, nrcv, &
  &                      my_pool_id, npool, inter_pool_comm)
  !
  ! Write el-ph to file elph.dat
  !
  IF(me_image == 0) THEN
     !
     elph_unit = find_free_unit()
     !
     WRITE(elphname,'(a,i0,a)') "elph", current_iq, ".dat"
     !
     OPEN(elph_unit,file = TRIM(elphname))
     !
     !# of Monkhost-Pack grid
     !
     WRITE(elph_unit,*) nk1, nk2, nk3
     ! 
     !# of band
     !
     WRITE(elph_unit,*) elph_nbnd_min, elph_nbnd_max
     !
     ! q-vector(Crystal cordinate)
     !
     WRITE(elph_unit,*) MATMUL(xq(1:3), at(1:3, 1:3))
     !
     ! # of mode
     !
     WRITE(elph_unit,*) 3 * nat
     !
     ! Frequences[Ryd]
     !
     DO ii = 1, 3 * nat
        WRITE(elph_unit,*) SQRT(ABS(w2(ii)))
     END DO
     WRITE(elph_unit,*) ""
     !
     ! Electron-Phonon matrix
     !
     DO ispin = 1, nspin_lsda
        DO ik = 1, nk1*nk2*nk3
           WRITE(elph_unit,'(6e25.15)') &
           &  gep_col(1:3*nat*nbnd_fs*nbnd_fs, ik, ispin)
        END DO
     END DO
     !
     CLOSE(elph_unit)
     !
  END IF ! IF(ionode)
  !
  DEALLOCATE(gep, gep_col)
  !
  CALL elph_scdft_dmuxc()
  !
END SUBROUTINE elph_scdft
!
!------------------------------------------------------------------------
SUBROUTINE elph_scdft_gather(snd,nsnd,rcv,nrcv,mype,npe,comm)
  !----------------------------------------------------------------------
  !
  ! This routine gathers a real matrix to PE 0.
  !
  USE kinds, ONLY : dp
  USE mp, ONLY : mp_sum, mp_gather
  !
  INTEGER,INTENT(IN) :: nsnd, nrcv, mype, npe, comm
  COMPLEX(dp),INTENT(IN) :: snd(nsnd)
  COMPLEX(dp),INTENT(OUT) :: rcv(nrcv)
  !
  INTEGER :: cnt(0:npe - 1), dsp(0:npe - 1), ipe
  !
  cnt(0:npe - 1) = 0
  cnt(mype) = nsnd
  !
  CALL mp_sum(cnt, comm)
  !
  dsp(0) = 0
  DO ipe = 1, npe - 1
     dsp(ipe) = dsp(ipe - 1) + cnt(ipe - 1)
  END DO
  !
  CALL mp_gather(snd(1:nsnd), rcv(1:nrcv), cnt, dsp, 0, comm)
  !
END SUBROUTINE elph_scdft_gather
!
!--------------------------------------------------------
SUBROUTINE elph_scdft_dmuxc()
  !------------------------------------------------------
  !
  ! This routine output the f_{XC} for LDA in G space to a file.
  !
  USE mp,        ONLY : mp_sum, mp_max, mp_min
  USE kinds,     ONLY : dp
  USE gvect,     ONLY : mill
  USE wvfct,     ONLY : npwx
  USE klist,     ONLY : ngk, igk_k
  USE fft_base,  ONLY : dffts
  USE mp_images, ONLY : me_image
  USE mp_pools,  ONLY : intra_pool_comm
  USE eqv,       ONLY : dmuxc
  !
  INTEGER :: gmin(3), gmax(3), ig, dmxc_unit
  COMPLEX(dp) :: dmxc1(dffts%nnr), dmxc2(npwx)
  COMPLEX(dp),ALLOCATABLE :: dmxc3(:,:,:)
  !
  INTEGER, EXTERNAL :: find_free_unit
  !
  ! Define whole G-grid
  !
  gmin(1:3) =   100
  gmax(1:3) = - 100
  DO ig = 1, ngk(1)
     gmax(1:3) = max(gmax(1:3), mill(1:3, igk_k(ig,1)))
     gmin(1:3) = min(gmin(1:3), mill(1:3, igk_K(ig,1)))
  END DO
  !
  ! FW_FFT dmuxc_r -> dmuxc_G
  !
  dmxc1(1:dffts%nnr) = CMPLX(dmuxc(1:dffts%nnr,1,1), 0.0_dp, KIND=DP)
  CALL elph_scdft_fft(dmxc2(1:npwx), dmxc1(1:dffts%nnr),  ngk(1),  igk_k(:,1),  -1)
  !
  CALL mp_max(gmax, intra_pool_comm)
  CALL mp_min(gmin, intra_pool_comm)
  !
  ALLOCATE(dmxc3(gmin(1):gmax(1), gmin(2):gmax(2), gmin(3):gmax(3)))
  !
  dmxc3(gmin(1):gmax(1),gmin(2):gmax(2),gmin(3):gmax(3)) = cmplx(0_dp,0_dp)
  DO ig = 1, ngk(1)
     dmxc3(mill(1,igk_k(ig,1)), mill(2,igk_k(ig,1)), mill(3,igk_k(ig,1))) = dmxc2(ig)
  END DO
  !
  CALL mp_sum(dmxc3, intra_pool_comm)
  !
  IF(me_image == 0) THEN
     !
     dmxc_unit = find_free_unit()
     !
     OPEN(dmxc_unit, file = "dmuxc.dat")
     !
     WRITE(dmxc_unit,*) gmin(1:3)
     WRITE(dmxc_unit,*) gmax(1:3)
     !
     WRITE(dmxc_unit,*) ""
     WRITE(dmxc_unit,'(2e25.15)') dmxc3(gmin(1):gmax(1),gmin(2):gmax(2),gmin(3):gmax(3))
     !
     CLOSE(dmxc_unit)
     !
  END IF
  !
  DEALLOCATE(dmxc3)
  !
END SUBROUTINE elph_scdft_dmuxc
!
!--------------------------------------------------------------------
SUBROUTINE elph_scdft_fft(evc_g, evc_r, npw, igk, isw)
  !------------------------------------------------------------------
  !
  ! This routine perform inverse-FFT for f_{XC}
  !
  USE kinds, ONLY : dp
  USE wvfct, ONLY : npwx
  USE fft_base,   ONLY: dffts
  USE fft_interfaces, ONLY: fwfft, invfft
  !
  INTEGER,INTENT(IN) :: isw
  INTEGER,INTENT(IN) :: npw, igk(npw)
  COMPLEX(dp),INTENT(INOUT) :: evc_g(npwx), evc_r(dffts%nnr)
  !
  INTEGER :: ig
  !
  IF (isw.eq.1) THEN
     !
     evc_r = CMPLX(0.0_dp, 0.0_dp, KIND=dp)
     !     
     DO ig = 1, npw
        evc_r(dffts%nl(igk(ig))) = evc_g(ig)
     END DO
     !
     CALL invfft ('Wave', evc_r(1:dffts%nnr), dffts)
     !
  ELSE IF(isw.eq.-1) THEN
     !
     CALL fwfft ('Wave', evc_r(1:dffts%nnr), dffts)
     !
     DO ig = 1, npw
        evc_g(ig) = evc_r(dffts%nl(igk(ig)))
     END DO
     !
  END IF
  !
END SUBROUTINE elph_scdft_fft
!
END MODULE elph_scdft_mod
