﻿// <著作権>
// Copyright (c) 2008, NTT DATA Corporation.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Common.BLogic;
using System.Data;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.CommonUT.BLogic
{
    ///<summary>
    /// BLogicResult のテストクラスです。
    ///</summary>

    [TestClass()]
    public class BLogicResultTest
    {
        private TestContext testContextInstance;

        ///<summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        ///<summary>
        /// BLogicResult() のテスト
        ///</summary>
        ///<remarks>
        /// 引数無しのコンストラクタを呼び出した場合、
        /// インスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            //実行
            BLogicResult target = new BLogicResult();

            //検証
            Assert.IsNotNull(target);
            Assert.IsNull(target.ResultData);
            Assert.IsNull(target.ResultString);
            Assert.IsNotNull(target.Errors);
            Assert.AreEqual(0, target.Errors.Count);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        /// <summary>
        ///BLogicResult (string) のテスト
        ///</summary>
        ///<remarks>
        /// 引数ありのコンストラクタを呼び出した場合、
        /// インスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest02()
        {
            //準備
            string resultString = "TERASOLUNA";

            //実行
            BLogicResult target = new BLogicResult(resultString);

            //検証
            Assert.IsNotNull(target);
            Assert.AreEqual(resultString, target.ResultString);
            Assert.IsNull(target.ResultData);
            Assert.IsNotNull(target.Errors);
            Assert.AreEqual(0, target.Errors.Count);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        /// <summary>
        ///BLogicResult (string, DataSet) のテスト
        ///</summary>
        ///<remarks>
        /// 引数ありのコンストラクタを呼び出した場合、
        /// インスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest03()
        {
            //準備
            string resultString = "TERASOLUNA";
            DataSet resultData = new DataSet();

            //実行
            BLogicResult target = new BLogicResult(resultString, resultData);

            //検証
            Assert.IsNotNull(target);
            Assert.AreEqual(resultString, target.ResultString);
            Assert.AreSame(resultData, target.ResultData);
            Assert.IsNotNull(target.Errors);
            Assert.AreEqual(0, target.Errors.Count);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        ///<summary>
        /// AddErrors (IList) のテスト
        ///</summary>
        ///<remarks>
        /// 要素数1のIListを格納/取得した場合、
        /// 取得した要素数と要素が、格納したものと一致することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddErrorsTest01()
        {
            //準備
            BLogicResult target = new BLogicResult();
            IList<MessageInfo> expectedMessages = new List<MessageInfo>();
            expectedMessages.Add(new MessageInfo("error01", "01エラーが発生しました"));

            //実行
            target.AddErrors(expectedMessages);

            //検証
            IList<MessageInfo> actualMessages = target.Errors;
            Assert.IsNotNull(actualMessages);
            Assert.AreEqual(1, actualMessages.Count);
            Assert.AreSame(expectedMessages[0], actualMessages[0]);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        ///<summary>
        /// AddErrors (IList) のテスト
        ///</summary>
        ///<remarks>
        /// 要素数が複数のIListを格納/取得した場合、
        /// 取得した要素数と要素が、格納したものと一致することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddErrorsTest02()
        {
            //準備
            BLogicResult target = new BLogicResult();
            IList<MessageInfo> expectedMessages = new List<MessageInfo>();
            expectedMessages.Add(new MessageInfo("error01", "01エラーが発生しました"));
            expectedMessages.Add(new MessageInfo("error02", "02エラーが発生しました"));
            expectedMessages.Add(new MessageInfo("error03", "03エラーが発生しました"));

            //実行
            target.AddErrors(expectedMessages);

            //検証
            IList<MessageInfo> actualMessages = target.Errors;
            Assert.IsNotNull(actualMessages);
            Assert.AreEqual(3, actualMessages.Count);
            Assert.AreSame(expectedMessages[0], actualMessages[0]);
            Assert.AreSame(expectedMessages[1], actualMessages[1]);
            Assert.AreSame(expectedMessages[2], actualMessages[2]);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        ///<summary>
        /// AddErrors (IList) のテスト
        ///</summary>
        ///<remarks>
        /// 要素数0のIListを格納/取得した場合、
        /// 取得した要素数が格納したものと一致することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddErrorsTest03()
        {
            //準備
            BLogicResult target = new BLogicResult();
            IList<MessageInfo> expectedMessages = new List<MessageInfo>();
            int expectedCount = expectedMessages.Count;
            //実行
            target.AddErrors(expectedMessages);

            //検証
            IList<MessageInfo> actualMessages = target.Errors;
            Assert.IsNotNull(actualMessages);
            Assert.AreEqual(0, actualMessages.Count);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        ///<summary>
        /// AddErrors (IList) のテスト
        ///</summary>
        ///<remarks>
        /// nullを格納した場合、ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddErrorsTest04()
        {
            //準備
            BLogicResult target = new BLogicResult();
            IList<MessageInfo> expectedMessages = null;
            try
            {
                //実行
                target.AddErrors(expectedMessages);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // 成功
                Assert.AreEqual("messages", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            //ログ
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                typeof(BLogicResult), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "messages"));
        }

        ///<summary>
        /// Errors のテスト
        ///</summary>
        ///<remarks>
        /// Errorsに値を追加していない場合、例外がスローされず、要素数0のIListが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ErrorsTest01()
        {
            //準備
            BLogicResult target = new BLogicResult();

            //実行
            IList<MessageInfo> actualMessages = target.Errors;

            //検証
            Assert.IsNotNull(actualMessages);
            Assert.AreEqual(0, actualMessages.Count);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        ///<summary>
        /// Errors のテスト
        ///</summary>
        ///<remarks>
        /// Errorsに値を追加した場合、取得した値と一致することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ErrorsTest02()
        {
            //準備
            BLogicResult target = new BLogicResult();
            IList<MessageInfo> expectedMessages = new List<MessageInfo>();
            expectedMessages.Add(new MessageInfo("error01", "01エラーが発生しました"));
            target.AddErrors(expectedMessages);

            //検証
            IList<MessageInfo> actualMessages = target.Errors;

            //検証
            Assert.IsNotNull(actualMessages);
            Assert.AreEqual(1, actualMessages.Count);
            Assert.AreSame(expectedMessages[0], actualMessages[0]);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        ///<summary>
        /// ResultData のテスト
        ///</summary>
        ///<remarks>
        /// ResultDataにDataSetを格納し取得した場合、一致するかどうかを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ResultDataTest01()
        {
            //準備
            BLogicResult target = new BLogicResult();
            DataSet expectedDataSet = new DataSet();

            //実行
            target.ResultData = expectedDataSet;
            DataSet actualDataSet = target.ResultData;

            //検証
            Assert.AreSame(expectedDataSet, actualDataSet);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        ///<summary>
        /// ResultString のテスト
        ///</summary>
        ///<remarks>
        /// ResultStringに文字列を格納し取得した場合、一致するかどうかを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ResultStringTest01()
        {
            //準備
            BLogicResult target = new BLogicResult();
            string expectedString = "testtest";

            //実行
            target.ResultString = expectedString;
            string actualString = target.ResultString;

            //検証
            Assert.AreEqual(expectedString, actualString);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }

        ///<summary>
        /// Items のテスト
        ///</summary>
        ///<remarks>
        /// Itemsにインスタンスを格納し取得した場合、一致するかどうかを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemsTest01()
        {
            //準備
            BLogicResult target = new BLogicResult();

           
            //実行
            IDictionary<string, object> actualItems = target.Items;

            //検証
            Assert.IsInstanceOfType(actualItems, typeof(Dictionary<string, object>));
            Assert.AreEqual(0, actualItems.Count);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(BLogicResult), LogLevel.INFO));
        }
    }
}
