﻿// <著作権>
// Copyright (c) 2007, NTT DATA Corporation.
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Common.Validation.Validators;
using Microsoft.Practices.EnterpriseLibrary.Validation;
using System.Reflection;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.CommonUT.Validation.Validators
{
    /// <summary>
    /// ZenkakuStringValidator のテストクラスです。
    /// </summary>
    [TestClass()]
    public class ZenkakuStringValidatorTest
    {
        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        //ログの初期化
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion

        /// <summary>
        /// DefaultNegatedMessageTemplateプロパティのテスト
        /// </summary>
        /// <remarks>
        /// 正常な値を設定/取得した場合、一致することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DefaultNegatedMessageTemplateTest01()
        {
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = true;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);

            object[] placeHolder = { "{0}", "{1}", "{2}", "{3}", "{4}", "{5}" };
            string val = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NEGATED_MESSAGETEMPLATE", placeHolder);

            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            Assert.AreEqual<string>(val, accessor.DefaultNegatedMessageTemplate, "プロパティの値が不正です");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DefaultNonNegatedMessageTemplateプロパティのテスト
        /// </summary>
        /// <remarks>
        /// 正常な値を設定/取得した場合、一致することを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DefaultNonNegatedMessageTemplateTest01()
        {
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);

            object[] placeHolder = { "{0}", "{1}", "{2}", "{3}", "{4}", "{5}" };
            string val = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NONNEGATED_MESSAGETEMPLATE", placeHolder);

            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            Assert.AreEqual<string>(val, accessor.DefaultNonNegatedMessageTemplate, "プロパティの値が不正です");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        #region DoValidate
        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateがnullの場合、検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest01()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);


            object objectToValidate = null;  // 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsTrue(results.IsValid);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateが空文字の場合、検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest02()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);


            object objectToValidate = ""; // 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsTrue(results.IsValid);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateが全角文字のみの場合、
        /// 検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest03()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            // 検証対象データ
            object objectToValidate = "アイウエオあいうえお亜胃鶏獲尾ＡＢＣ０１２";

            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsTrue(results.IsValid);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれる場合、
        /// 検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest04()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(1, resultList.Count, "エラー数が不正です.");
            object[] placeHolder = { "{0}", "{1}", tag, "{3}", "{4}", "{5}" };
            string expectedMessage = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NONNEGATED_MESSAGETEMPLATE", placeHolder);
            ValidationResult r = resultList.FindFromMessage(expectedMessage);
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.AreEqual<string>(r.Key, "キー", "Keyが不正です.");
            Assert.AreEqual<string>(r.Tag, "検証対象項目", "Tagが不正です.");
            Assert.AreEqual(r.Target, "オブジェクト", "Targetが不正です.");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateが非文字列の場合、
        /// 検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest05()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);


            object objectToValidate = new DateTime();　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(1, resultList.Count, "エラー数が不正です.");
            object[] placeHolder = { "{0}", "{1}", tag, "{3}", "{4}", "{5}" };
            string expectedMessage = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NONNEGATED_MESSAGETEMPLATE", placeHolder);
            ValidationResult r = resultList.FindFromMessage(expectedMessage);
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.AreEqual<string>(r.Key, "キー", "Keyが不正です.");
            Assert.AreEqual<string>(r.Tag, "検証対象項目", "Tagが不正です.");
            Assert.AreEqual(r.Target, "オブジェクト", "Targetが不正です.");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      

        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// currentTargetがnullの場合、検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest06()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = null;
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(1, resultList.Count, "エラー数が不正です.");
            object[] placeHolder = { "{0}", "{1}", tag, "{3}", "{4}", "{5}" };
            string expectedMessage = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NONNEGATED_MESSAGETEMPLATE", placeHolder);
            ValidationResult r = resultList.FindFromMessage(expectedMessage);
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.AreEqual<string>(r.Key, "キー", "Keyが不正です.");
            Assert.AreEqual<string>(r.Tag, "検証対象項目", "Tagが不正です.");
            Assert.IsNull(r.Target, "Targetが不正です.");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      

        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateが全角文字のみであり、
        /// currentTargetがnullの場合、検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest07()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "アイウエオあいうえお亜胃鶏獲尾ＡＢＣ０１２";　// 検証対象データ
            object currentTarget = null;
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsTrue(results.IsValid);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// keyがnullの場合、検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest08()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = null; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(1, resultList.Count, "エラー数が不正です.");
            object[] placeHolder = { "{0}", "{1}", tag, "{3}", "{4}", "{5}" };
            string expectedMessage = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NONNEGATED_MESSAGETEMPLATE", placeHolder);
            ValidationResult r = resultList.FindFromMessage(expectedMessage);
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.IsNull(r.Key, "Keyが不正です.");
            Assert.AreEqual<string>(r.Tag, "検証対象項目", "Tagが不正です.");
            Assert.AreEqual(r.Target, "オブジェクト", "Targetが不正です.");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateが全角文字のみであり、
        /// keyがnullの場合、検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest09()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "アイウエオあいうえお亜胃鶏獲尾ＡＢＣ０１２";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = null; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsTrue(results.IsValid);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// ValidationResultsがnullの場合、ArgumentNullExceptionがスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest10()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = null;

            // 実施
            try
            {
                accessor.DoValidate(objectToValidate, currentTarget, key, results);
                Assert.Fail();
            }
            catch (ArgumentNullException e)
            {
                Assert.IsInstanceOfType(e, typeof(ArgumentNullException), "例外の型が不正です");
                // 成功
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            //ログ
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ZenkakuStringValidator), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "validationResults"));
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateが全角文字のみであり、
        /// ValidationResultsがnullの場合、ArgumentNullExceptionがスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest11()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "アイウエオあいうえお亜胃鶏獲尾ＡＢＣ０１２";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = null;

            // 実施
            try
            {
                accessor.DoValidate(objectToValidate, currentTarget, key, results);
                Assert.Fail();
            }
            catch (ArgumentNullException e)
            {
                Assert.IsInstanceOfType(e, typeof(ArgumentNullException), "例外の型が不正です");
                // 成功
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            //ログ
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ZenkakuStringValidator), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "validationResults"));

        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// ValidationResultsの検証エラーが1つの場合、検証結果オブジェクトの
        /// 値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest12()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();
            results.AddResult(new ValidationResult("result01", null, null, null, null));

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(2, resultList.Count, "エラー数が不正です.");
            object[] placeHolder = { "{0}", "{1}", tag, "{3}", "{4}", "{5}" };
            string expectedMessage = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NONNEGATED_MESSAGETEMPLATE", placeHolder);
            ValidationResult r = resultList.FindFromMessage(expectedMessage);
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.AreEqual<string>(r.Key, "キー", "Keyが不正です.");
            Assert.AreEqual<string>(r.Tag, "検証対象項目", "Tagが不正です.");
            Assert.AreEqual(r.Target, "オブジェクト", "Targetが不正です.");

            ValidationResult r1 = resultList.FindFromMessage("result01");
            Assert.IsNotNull(r1, "メッセージが不正です.");
            Assert.IsNull(r1.Key);
            Assert.IsNull(r1.Target);
            Assert.IsNull(r1.Tag);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateが全角文字のみであり、
        /// ValidationResultsの検証エラーが1つの場合、検証結果オブジェクトの
        /// 値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest13()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "アイウエオあいうえお亜胃鶏獲尾ＡＢＣ０１２";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();
            results.AddResult(new ValidationResult("result01", null, null, null, null));

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(1, resultList.Count, "エラー数が不正です.");

            ValidationResult r1 = resultList.FindFromMessage("result01");
            Assert.IsNotNull(r1, "メッセージが不正です.");
            Assert.IsNull(r1.Key);
            Assert.IsNull(r1.Target);
            Assert.IsNull(r1.Tag);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO)); 
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// validationResultsの検証エラーが3つの場合、検証結果オブジェクトの
        /// 値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest14()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();
            results.AddResult(new ValidationResult("result01", null, null, null, null));
            results.AddResult(new ValidationResult("result02", null, null, null, null));
            results.AddResult(new ValidationResult("result03", null, null, null, null));
            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(4, resultList.Count, "エラー数が不正です.");
            object[] placeHolder = { "{0}", "{1}", tag, "{3}", "{4}", "{5}" };
            string expectedMessage = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NONNEGATED_MESSAGETEMPLATE", placeHolder);
            ValidationResult r = resultList.FindFromMessage(expectedMessage);
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.AreEqual<string>(r.Key, "キー", "Keyが不正です.");
            Assert.AreEqual<string>(r.Tag, "検証対象項目", "Tagが不正です.");
            Assert.AreEqual(r.Target, "オブジェクト", "Targetが不正です.");

            ValidationResult r1 = resultList.FindFromMessage("result01");
            Assert.IsNotNull(r1, "メッセージが不正です.");
            Assert.IsNull(r1.Key);
            Assert.IsNull(r1.Target);
            Assert.IsNull(r1.Tag);

            ValidationResult r2 = resultList.FindFromMessage("result02");
            Assert.IsNotNull(r2, "メッセージが不正です.");
            Assert.IsNull(r2.Key);
            Assert.IsNull(r2.Target);
            Assert.IsNull(r2.Tag);

            ValidationResult r3 = resultList.FindFromMessage("result03");
            Assert.IsNotNull(r3, "メッセージが不正です.");
            Assert.IsNull(r3.Key);
            Assert.IsNull(r3.Target);
            Assert.IsNull(r3.Tag);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateが全角文字のみであり、
        /// validationResultsの検証エラーが3つの場合、検証結果オブジェクトの
        /// 値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest15()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "アイウエオあいうえお亜胃鶏獲尾ＡＢＣ０１２";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();
            results.AddResult(new ValidationResult("result01", null, null, null, null));
            results.AddResult(new ValidationResult("result02", null, null, null, null));
            results.AddResult(new ValidationResult("result03", null, null, null, null));
            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(3, resultList.Count, "エラー数が不正です.");

            ValidationResult r1 = resultList.FindFromMessage("result01");
            Assert.IsNotNull(r1, "メッセージが不正です.");
            Assert.IsNull(r1.Key);
            Assert.IsNull(r1.Target);
            Assert.IsNull(r1.Tag);

            ValidationResult r2 = resultList.FindFromMessage("result02");
            Assert.IsNotNull(r2, "メッセージが不正です.");
            Assert.IsNull(r2.Key);
            Assert.IsNull(r2.Target);
            Assert.IsNull(r2.Tag);

            ValidationResult r3 = resultList.FindFromMessage("result03");
            Assert.IsNotNull(r3, "メッセージが不正です.");
            Assert.IsNull(r3.Key);
            Assert.IsNull(r3.Target);
            Assert.IsNull(r3.Tag);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      

        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// プロパティのNegatedがtrueの場合、検証結果オブジェクトの
        /// 値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest16()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = true;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsTrue(results.IsValid);

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      

        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateが全角文字のみであり、
        /// プロパティのNegatedがtrueの場合、検証結果オブジェクトの
        /// 値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest17()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = true;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "アイウエオあいうえお亜胃鶏獲尾ＡＢＣ０１２";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(1, resultList.Count, "エラー数が不正です.");
            object[] placeHolder = { "{0}", "{1}", tag, "{3}", "{4}", "{5}" };
            string expectedMessage = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NEGATED_MESSAGETEMPLATE", placeHolder);
            ValidationResult r = resultList.FindFromMessage(expectedMessage);
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.AreEqual<string>(r.Key, "キー", "Keyが不正です.");
            Assert.AreEqual<string>(r.Tag, "検証対象項目", "Tagが不正です.");
            Assert.AreEqual(r.Target, "オブジェクト", "Targetが不正です.");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      

        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// プロパティのMessageTemplateが空文字の場合、検証結果オブジェクトの
        /// 値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest18()
        {
            // 定義
            string messageTemplate = "";
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(1, resultList.Count, "エラー数が不正です.");
            ValidationResult r = resultList.FindFromMessage("");
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.AreEqual<string>(r.Key, "キー", "Keyが不正です.");
            Assert.AreEqual<string>(r.Tag, "検証対象項目", "Tagが不正です.");
            Assert.AreEqual(r.Target, "オブジェクト", "Targetが不正です.");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// プロパティのMessageTemplateがプレースホルダ0～2の場合、
        /// 検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest19()
        {
            // 定義
            string messageTemplate = "{0},{1},{2}";
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(1, resultList.Count, "エラー数が不正です.");
            ValidationResult r = resultList.FindFromMessage("ABCabc123ｱｲｳｴｵアイウエオ,キー,検証対象項目");
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.AreEqual<string>(r.Key, "キー", "Keyが不正です.");
            Assert.AreEqual<string>(r.Tag, "検証対象項目", "Tagが不正です.");
            Assert.AreEqual(r.Target, "オブジェクト", "Targetが不正です.");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      

        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// プロパティのMessageTemplateがプレースホルダ0～3の場合、
        /// FormatExceptionがスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest20()
        {
            // 定義
            string messageTemplate = "{0},{1},{2},{3}";
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            try
            {
                accessor.DoValidate(objectToValidate, currentTarget, key, results);
                Assert.Fail();
            }
            catch (FormatException e)
            {
                Assert.IsInstanceOfType(e, typeof(FormatException), "例外の型が不正です");

                // 成功
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateに全角文字以外の文字が含まれ、
        /// プロパティのTagがnullの場合、検証結果オブジェクトの各プロパティの値が想定値であることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest21()
        {
            // 定義
            string messageTemplate = null;
            string tag = null;
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = "ABCabc123ｱｲｳｴｵアイウエオ";　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = new ValidationResults();

            // 実施
            accessor.DoValidate(objectToValidate, currentTarget, key, results);

            // 確認
            Assert.IsFalse(results.IsValid);

            ValidationResultList resultList = new ValidationResultList(results);

            Assert.AreEqual<int>(1, resultList.Count, "エラー数が不正です.");
            object[] placeHolder = { "{0}", "{1}", tag, "{3}", "{4}", "{5}" };
            string expectedMessage = MessageManager.GetMessage("ZENKAKU_STRING_VALIDATOR_NONNEGATED_MESSAGETEMPLATE", placeHolder);
            ValidationResult r = resultList.FindFromMessage(expectedMessage);
            Assert.IsNotNull(r, "メッセージが不正です.");
            Assert.AreEqual<string>(r.Key, "キー", "Keyが不正です.");
            Assert.IsNull(r.Tag, "Tagが不正です.");
            Assert.AreEqual(r.Target, "オブジェクト", "Targetが不正です.");

            //ログ
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));      
        }

        /// <summary>
        /// DoValidate (object, object, string, ValidationResults) のテスト
        /// </summary>
        /// <remarks>
        /// 引数のobjectToValidateがnullであり、ValidationResultsがnullの場合、
        /// ArgumentNullExceptionがスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void DoValidateTest22()
        {
            // 定義
            string messageTemplate = null;
            string tag = "検証対象項目";
            bool negated = false;

            ZenkakuStringValidator target =
                new ZenkakuStringValidator(messageTemplate, tag, negated);
            TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor accessor = new TERASOLUNA.Fw.CommonUT.TERASOLUNA_Fw_Common_Validation_Validators_ZenkakuStringValidatorAccessor(target);

            object objectToValidate = null;　// 検証対象データ
            object currentTarget = "オブジェクト";
            string key = "キー"; // 

            ValidationResults results = null;

            // 実施
            try
            {
                accessor.DoValidate(objectToValidate, currentTarget, key, results);
                Assert.Fail();
            }
            catch (ArgumentNullException e)
            {
                Assert.IsInstanceOfType(e, typeof(ArgumentNullException), "例外の型が不正です");
                // 成功
            }
            catch (Exception e)
            {
                Assert.Fail(e.Message);
            }

            //ログ
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ZenkakuStringValidator), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ZenkakuStringValidator), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "validationResults"));
        }
        #endregion
    }
}
