// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Web.Configuration.Sql;
using System.Configuration;
using TERASOLUNA.Fw.WebUT;
using TERASOLUNA.Fw.Web.Configuration.Page;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Reflection;
using TERASOLUNA.Fw.Common;
namespace TERASOLUNA.Fw.WebUT.Configuration.Sql
{
    /// <summary>
    /// TERASOLUNA.Fw.Common.Configuration.SqlsCollection のテスト クラスです。すべての
    /// TERASOLUNA.Fw.Common.Configuration.SqlsCollection 単体テストをここに含めます
    /// </summary>

    [TestClass()]
    public class SqlsCollectionTest
    {

        private TestContext testContextInstance;

        /// <summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        /// </summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        /// <summary>
        /// SqlsCollection () のテスト
        /// </summary>
        /// <TestTarget>Constructor()</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>インスタンスが生成されること</TestAssertion>
        [TestMethod()]
        public void TestConstructor01()
        {
            SqlsCollection target = new SqlsCollection();
            Assert.IsNotNull(target);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        /// CollectionType のテスト
        /// </summary>
        /// <TestTarget>CollectionType()</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>ConfigurationElementCollectionType.BasicMapが返却されること</TestAssertion>
        [TestMethod()]
        public void TestCollectionType01()
        {
            SqlsCollection target = new SqlsCollection();
            Assert.AreEqual(ConfigurationElementCollectionType.BasicMap, target.CollectionType);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        /// this[int index] のテスト
        /// </summary>
        /// <TestTarget>this[int index]</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>indexに指定したSqlElementが返却されること</TestAssertion>
        [TestMethod()]
        public void TestItem01()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement val1 = new SqlElement("SQL1", "SELECT * FROM TABLE");
            SqlElement val2 = new SqlElement("SQL2", "INSERT INTO TABLE VALUES (2)");
            target.Add(val1);
            target.Add(val2);

            Assert.AreEqual(val1, target[0]);
            Assert.AreEqual(val2, target[1]);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        /// this[string name] のテスト
        /// </summary>
        /// <TestTarget>this[string name]</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>nameに指定したSqlElementが返却されること</TestAssertion>
        [TestMethod()]
        public void TestItem02()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement val1 = new SqlElement("SQL1", "SELECT * FROM TABLE");
            SqlElement val2 = new SqlElement("SQL2", "INSERT INTO TABLE VALUES (2)");
            target.Add(val1);
            target.Add(val2);

            Assert.AreEqual(val1, target["SQL1"]);
            Assert.AreEqual(val2, target["SQL2"]);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        /// Count のテスト
        /// </summary>
        /// <TestTarget>Count</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>保持しているSql要素数が返却されること</TestAssertion>
        [TestMethod()]
        public void TestCount01()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement val1 = new SqlElement("SQL1", "SELECT * FROM TABLE");
            SqlElement val2 = new SqlElement("SQL2", "INSERT INTO TABLE VALUES (2)");
            target.Add(val1);
            target.Add(val2);

            Assert.AreEqual(2, target.Count);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        /// Add (SqlElement) のテスト
        /// </summary>
        /// <TestTarget>Add (SqlElement)</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>引数の<see cref="TERASOLUNA.Common.Configuration.SqlElement" />が要素コレクションに追加されること</TestAssertion>
        [TestMethod()]
        public void TestAdd01()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement element = new SqlElement("SQL1", "SELECT * FROM TABLE");
            target.Add(element);

            Assert.AreEqual(element, target[0]);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        /// IsReadOnly () のテスト
        /// </summary>
        /// <TestTarget>IsReadOnly()</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>falseが返却されること</TestAssertion>
        [TestMethod()]
        public void TestIsReadOnly01()
        {
            SqlsCollection target = new SqlsCollection();
            Assert.IsFalse(target.IsReadOnly());
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        /// Add (SqlElement) のテスト
        /// </summary>
        /// <TestTarget>Add (SqlElement)</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestAdd02()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement element = new SqlElement();
            element = null;

            try
            {
                target.Add(element);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                Assert.AreEqual("element", ex.ParamName);
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(SqlsCollection), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "element"));
        }

        /// <summary>
        ///CreateNewElement () のテスト
        ///</summary>
        /// <TestTarget>CreateNewElement</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>インスタンスが生成されること</TestAssertion>
        [DeploymentItem("TERASOLUNA.Fw.Web.dll")]
        [TestMethod()]
        public void TestCreateNewElement01()
        {
            SqlsCollection target = new SqlsCollection();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor(target);

            ConfigurationElement actual;

            actual = accessor.CreateNewElement();

            Assert.IsNotNull(actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        ///GetElementKey () のテスト
        ///</summary>
        /// <TestTarget>GetElementKey</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>Keyが取得できること</TestAssertion>
        [DeploymentItem("TERASOLUNA.Fw.Web.dll")]
        [TestMethod()]
        public void TestGetElementKey01()
        {
            SqlsCollection target = new SqlsCollection();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor(target);

            object GetKey;

            SqlElement element = new SqlElement("SQL1", "SELECT * FROM TABLE");

            GetKey = accessor.GetElementKey(element);

            Assert.AreEqual(GetKey.ToString(), "SQL1");
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));

        }

        /// <summary>
        ///GetElementKey () のテスト
        ///</summary>
        /// <TestTarget>GetElementKey</TestTarget>
        /// <TestCondition>引数のelementにnullを設定する</TestCondition>
        /// <TestAssertion>ArgumentNullExceptionがスローされること</TestAssertion>
        [DeploymentItem("TERASOLUNA.Fw.Web.dll")]
        [TestMethod()]
        public void TestGetElementKey02()
        {
            SqlsCollection target = new SqlsCollection();

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor(target);

            object GetKey;

            SqlElement element = new SqlElement();
            element = null;

            try
            {

                GetKey = accessor.GetElementKey(element);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException ex)
            {
                //ArgumentNullException ae = ex.InnerException as ArgumentNullException;
                Assert.AreEqual("element", ex.ParamName);
                Assert.IsInstanceOfType(ex, typeof(ArgumentNullException));
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(SqlsCollection), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "element"));

        }

        /// <summary>
        /// GetElementKey () のテスト
        /// </summary>
        /// <TestTarget>Constructor()</TestTarget>
        /// <TestCondition>引数にSqlElement型以外を指定すること</TestCondition>
        /// <TestAssertion>ArgumentExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestGetElementKey03()
        {
            try
            {
                SqlsCollection target = new SqlsCollection();

                TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor(target);

                object GetKey;

                PageElement element = new PageElement();

                GetKey = accessor.GetElementKey(element);
                Assert.Fail("到達してはいけないコードです。");

            }
            catch (ArgumentException ex)
            {
                //ArgumentException ae = ex.InnerException as ArgumentException;
                Assert.AreEqual(typeof(ArgumentException).Name, ex.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_TYPE","element"), ex.Message);
                
            }
            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(
                    typeof(SqlsCollection), typeof(ArgumentException), "E_INVALID_TYPE", "element"));
        }


        /// <summary>
        /// Count のテスト
        /// </summary>
        /// <TestTarget>Count</TestTarget>
        /// <TestCondition>一つも値を追加しない</TestCondition>
        /// <TestAssertion>０が返却されること</TestAssertion>
        [TestMethod()]
        public void TestCount02()
        {
            SqlsCollection target = new SqlsCollection();
            Assert.AreEqual(0, target.Count);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        ///ElementName のテスト
        ///</summary>
        /// <TestTarget>ElementName</TestTarget>
        /// <TestCondition></TestCondition>
        /// <TestAssertion>"sql"を取得すること</TestAssertion>
        [DeploymentItem("TERASOLUNA.Fw.Web.dll")]
        [TestMethod()]
        public void TestElementName01()
        {
            SqlsCollection target = new SqlsCollection();

            string val = "sql";

            TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor accessor = new TERASOLUNA.Fw.WebUT.TERASOLUNA_Fw_Web_Configuration_Sql_SqlsCollectionAccessor(target);


            Assert.AreEqual(val, accessor.ElementName, "TERASOLUNA.Fw.Configuration.SqlsCollection.ElementName は正しく設定されていませんでした。");

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        /// this[int index] のテスト
        /// </summary>
        /// <TestTarget>this[int index]</TestTarget>
        /// <TestCondition>indexに不正な値を設定すること</TestCondition>
        /// <TestAssertion>IndexOutOfRangeExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestItem03()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement val1 = new SqlElement("SQL1", "SELECT * FROM TABLE");
            SqlElement val2 = new SqlElement("SQL2", "INSERT INTO TABLE VALUES (2)");
            target.Add(val1);
            target.Add(val2);

            try
            {
                Assert.AreEqual(val1, target[-1]);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException e)
            {
                Assert.AreEqual("インデックス -1 が範囲を超えています。", e.Message);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }


        /// <summary>
        /// this[int index] のテスト
        /// </summary>
        /// <TestTarget>this[int index]</TestTarget>
        /// <TestCondition>indexに範囲外の値を設定すること</TestCondition>
        /// <TestAssertion>IndexOutOfRangeExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestItem04()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement val1 = new SqlElement("SQL1", "SELECT * FROM TABLE");
            SqlElement val2 = new SqlElement("SQL2", "INSERT INTO TABLE VALUES (2)");
            target.Add(val1);
            target.Add(val2);

            try
            {

                Assert.AreEqual(val1, target[2]);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException e)
            {
                Assert.AreEqual("インデックス 2 が範囲を超えています。", e.Message);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }

        /// <summary>
        /// this[string name] のテスト
        /// </summary>
        /// <TestTarget>this[string name]</TestTarget>
        /// <TestCondition>nameに不正な値を設定すること</TestCondition>
        /// <TestAssertion>nullを取得すること</TestAssertion>
        [TestMethod()]
        public void TestItem05()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement val1 = new SqlElement("SQL1", "SELECT * FROM TABLE");
            SqlElement val2 = new SqlElement("SQL2", "INSERT INTO TABLE VALUES (2)");
            target.Add(val1);
            target.Add(val2);

            string indexValue = "TEST";

            Assert.AreEqual(null, target[indexValue]);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }


        /// <summary>
        /// this[string name] のテスト
        /// </summary>
        /// <TestTarget>this[string name]</TestTarget>
        /// <TestCondition>nameに不正な値を設定すること</TestCondition>
        /// <TestAssertion>NullReferenceExceptionがスローされること</TestAssertion>
        [TestMethod()]
        public void TestItem06()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement val1 = new SqlElement("SQL1", "SELECT * FROM TABLE");
            SqlElement val2 = new SqlElement("SQL2", "INSERT INTO TABLE VALUES (2)");
            target.Add(val1);
            target.Add(val2);

            string indexValue = null;

            try
            {
                Assert.AreEqual(val1, target[indexValue]);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (System.NullReferenceException e)
            {
                Assert.AreEqual("オブジェクト参照がオブジェクト インスタンスに設定されていません。", e.Message);
            }
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }


        /// <summary>
        /// this[string name] のテスト
        /// </summary>
        /// <TestTarget>this[string name]</TestTarget>
        /// <TestCondition>nameに不正な値を設定すること</TestCondition>
        /// <TestAssertion>nullを取得すること</TestAssertion>
        [TestMethod()]
        public void TestItem07()
        {
            SqlsCollection target = new SqlsCollection();

            SqlElement val1 = new SqlElement("SQL1", "SELECT * FROM TABLE");
            SqlElement val2 = new SqlElement("SQL2", "INSERT INTO TABLE VALUES (2)");
            target.Add(val1);
            target.Add(val2);

            string indexValue = "";

            Assert.AreEqual(null, target[indexValue]);
            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(SqlsCollection), LogLevel.INFO));
        }
    }
}
