/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.ackedqueue;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.logstash.ackedqueue.io.FileCheckpointIO;

public final class PqRepair {
    private static final Logger LOGGER = LogManager.getLogger(PqRepair.class);

    private PqRepair() {
    }

    public static void main(String ... args) throws IOException {
        if (args.length == 0) {
            throw new IllegalArgumentException("No queue directory given.");
        }
        Path pqRoot = Paths.get(args[0], new String[0]);
        PqRepair.repair(pqRoot);
    }

    public static void repair(Path path) throws IOException {
        if (!path.toFile().isDirectory()) {
            throw new IllegalArgumentException(String.format("Given PQ path %s is not a directory.", path));
        }
        HashMap<Integer, Path> pageFiles = new HashMap<Integer, Path>();
        try (DirectoryStream<Path> pfs = Files.newDirectoryStream(path, "page.*");){
            pfs.forEach(p -> pageFiles.put(Integer.parseInt(p.getFileName().toString().substring("page.".length())), (Path)p));
        }
        HashMap<Integer, Path> checkpointFiles = new HashMap<Integer, Path>();
        try (DirectoryStream<Path> cpfs = Files.newDirectoryStream(path, "checkpoint.*");){
            cpfs.forEach(c -> {
                String cpFilename = c.getFileName().toString();
                if (!"checkpoint.head".equals(cpFilename)) {
                    checkpointFiles.put(Integer.parseInt(cpFilename.substring("checkpoint.".length())), (Path)c);
                }
            });
        }
        PqRepair.deleteFullyAcked(path, pageFiles, checkpointFiles);
        PqRepair.fixMissingPages(pageFiles, checkpointFiles);
        PqRepair.fixZeroSizePages(pageFiles, checkpointFiles);
        PqRepair.fixMissingCheckpoints(pageFiles, checkpointFiles);
    }

    private static void deleteFullyAcked(Path root, Map<Integer, Path> pages, Map<Integer, Path> checkpoints) throws IOException {
        String headCpName = "checkpoint.head";
        File headCheckpoint = root.resolve("checkpoint.head").toFile();
        if (headCheckpoint.exists()) {
            int lowestUnAcked = new FileCheckpointIO(root).read("checkpoint.head").getFirstUnackedPageNum();
            PqRepair.deleteFullyAcked(pages, lowestUnAcked, PqRepair.extractPagenums(pages));
            PqRepair.deleteFullyAcked(checkpoints, lowestUnAcked, PqRepair.extractPagenums(checkpoints));
        }
    }

    private static void deleteFullyAcked(Map<Integer, Path> files, int lowestUnAcked, int[] knownPagenums) throws IOException {
        for (int number : knownPagenums) {
            if (number >= lowestUnAcked) break;
            Path file = files.remove(number);
            if (file == null) continue;
            LOGGER.info("Deleting {} because it was fully acknowledged.", (Object)file);
            Files.delete(file);
        }
    }

    private static void fixMissingCheckpoints(Map<Integer, Path> pages, Map<Integer, Path> checkpoints) throws IOException {
        int[] knownPagenums = PqRepair.extractPagenums(pages);
        for (int i = 0; i < knownPagenums.length - 1; ++i) {
            int number = knownPagenums[i];
            Path cpPath = checkpoints.get(number);
            if (cpPath == null) {
                Path page = pages.get(number);
                PqRepair.recreateCheckpoint(page, number);
                continue;
            }
            if (cpPath.toFile().length() == 34L) continue;
            Files.delete(cpPath);
            PqRepair.recreateCheckpoint(pages.get(number), number);
        }
    }

    private static void recreateCheckpoint(Path pageFile, int number) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocateDirect(12);
        LOGGER.info("Recreating missing checkpoint for page {}", (Object)pageFile);
        try (FileChannel page = FileChannel.open(pageFile, new OpenOption[0]);){
            page.read(buffer);
            byte version = buffer.get(0);
            if (version != 1 && version != 2) {
                throw new IllegalStateException(String.format("Pagefile %s contains version byte %d, this tool only supports versions 1 and 2.", pageFile, version));
            }
            buffer.position(1);
            buffer.compact();
            page.read(buffer);
            long firstSeqNum = buffer.getLong(0);
            long maxSize = page.size();
            long position = page.position();
            position += (long)buffer.getInt(8) + 4L;
            int count = 1;
            while (position < maxSize - 18L) {
                page.position(position);
                buffer.clear();
                page.read(buffer);
                position += (long)buffer.getInt(8) + 4L;
                ++count;
            }
            new FileCheckpointIO(pageFile.getParent()).write(String.format("checkpoint.%d", number), number, 0, firstSeqNum, firstSeqNum, count);
        }
    }

    private static void fixMissingPages(Map<Integer, Path> pages, Map<Integer, Path> checkpoints) throws IOException {
        int[] knownCpNums;
        for (int number : knownCpNums = PqRepair.extractPagenums(checkpoints)) {
            if (pages.containsKey(number)) continue;
            Path cpPath = checkpoints.remove(number);
            Files.delete(cpPath);
            LOGGER.info("Deleting checkpoint {} because it has no associated page", (Object)cpPath);
        }
    }

    private static void fixZeroSizePages(Map<Integer, Path> pages, Map<Integer, Path> checkpoints) throws IOException {
        int[] knownPagenums;
        for (int number : knownPagenums = PqRepair.extractPagenums(pages)) {
            Path pagePath = pages.get(number);
            if (pagePath.toFile().length() >= 18L) continue;
            LOGGER.info("Deleting empty page found at {}", (Object)pagePath);
            Files.delete(pagePath);
            pages.remove(number);
            Path cpPath = checkpoints.remove(number);
            if (cpPath == null) continue;
            LOGGER.info("Deleting checkpoint {} because it has no associated page", (Object)cpPath);
            Files.delete(cpPath);
        }
    }

    private static int[] extractPagenums(Map<Integer, Path> fileMap) {
        int[] knownPagenums = fileMap.keySet().stream().mapToInt(Integer::intValue).toArray();
        Arrays.sort(knownPagenums);
        return knownPagenums;
    }
}

