(* vararg-constants.sml
 *
 * Constants shared between calling coded and the interpreter generator.
 *)

structure VarargConstants =
  struct

    structure W = Word32

  (* encodings for the kinds of argument locations *)
    local
	val [ GPR, FPR, STK, FSTK ] = List.tabulate(4,  W.fromInt)
    in
    fun kind CLocKind.GPR = GPR
      | kind CLocKind.FPR = FPR
      | kind CLocKind.STK = STK
      | kind CLocKind.FSTK = FSTK
    end

  (* 
   * the located argument structure:

      typedef void* Word_t;
      enum loc_kind {GPR=0, FPR, STK, FSTK};
      struct located_arg_s {
	loc_kind k;
	int width;
	int narrowing;
	int loc;
	int offset;
	union {
	   Word_t* p; long l; int i; char* s; double d; 
	} arg;
      };

   * The following offsets should mimic the way that C lays out the struct.
   *)
    val [ kindOffB, widthOffB, narrowingOffB, locOffB, offsetOffB, argOffB ] = 
	                 List.tabulate(6, fn i => W.fromInt i * 0w4)

  (* max byte width of arguments *)
    val maxArgSzB = 8

    val argSzB : W.word = 0w8

  (* number of bytes for a located argument  *)
    val locdArgSzB = argOffB + argSzB

    val varargInterpreter = "VarargInterp"

    val header = String.concatWith "\n" [
          "/* ",
	  " * ",
	  " * This file is GENERATED by the varargs library. Modify the",
	  " * code at your peril :).",
	  " * ",
	  " * This interpreter parses the arguments for a vararg C function,",
	  " * putting them in the correct locations, and then calls the function.",
	  " * ",
	  " * Mike Rainey (mrainey@cs.uchicago.edu)",
	  " */"
	]

  end
