/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.kie.dmn.feel.lang.ast;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.antlr.v4.runtime.ParserRuleContext;
import org.kie.dmn.api.feel.runtime.events.FEELEvent.Severity;
import org.kie.dmn.feel.lang.EvaluationContext;
import org.kie.dmn.feel.lang.Type;
import org.kie.dmn.feel.util.EvalHelper;
import org.kie.dmn.feel.util.Msg;
import org.kie.dmn.feel.util.StringEvalHelper;

public class QualifiedNameNode
        extends BaseNode {

    private List<NameRefNode> parts;
    private Type resultType;

    public QualifiedNameNode(ParserRuleContext ctx, List<NameRefNode> parts, Type type) {
        super( ctx );
        this.parts = parts;
        this.resultType = type;
    }

    public QualifiedNameNode(List<NameRefNode> parts, Type resultType, String text) {
        this.parts = parts;
        this.resultType = resultType;
        this.setText(text);
    }

    public List<NameRefNode> getParts() {
        return parts;
    }

    public void setParts(List<NameRefNode> parts) {
        this.parts = parts;
    }

    public String[] getPartsAsStringArray() {
        return parts.stream().map( p -> p.getText() ).toArray( String[]::new );
    }

    @Override
    public Object evaluate(EvaluationContext ctx) {
        Object current = parts.get( 0 ).evaluate( ctx );
        try {
            if ( current != null ) {
                for ( int i = 1; i < parts.size(); i++ ) {
                    String n = parts.get( i ).getText();
                    if ( current instanceof Collection ) {
                        // e.g.: FEEL: MyList.property1
                        // can't use Stream API as from EvalHelper.getValue I need to listen for checked exception
                        Collection<Object> result = new ArrayList<>();
                        for ( Object e : (Collection<?>) current ) {
                            result.add( EvalHelper.getValue(e, StringEvalHelper.normalizeVariableName(n ) ) );
                        }
                        current = result;
                    } else {
                        current = EvalHelper.getValue( current, StringEvalHelper.normalizeVariableName( n ) );
                    }
                }
                return current;
            }
        } catch ( Exception e ) {
            ctx.notifyEvt( astEvent(Severity.ERROR, Msg.createMessage(Msg.ERROR_ACCESSING_QUALIFIED_NAME, getText()), e) );
            return null;
        }
        return null;
    }

    @Override
    public Type getResultType() {
        return resultType;
    }

    @Override
    public ASTNode[] getChildrenNode() {
        return parts.toArray( new ASTNode[parts.size()] );
    }

    @Override
    public <T> T accept(Visitor<T> v) {
        return v.visit(this);
    }

}
