/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.imap.processor;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import java.util.Optional;
import org.apache.james.core.Username;
import org.apache.james.imap.api.ImapConfiguration;
import org.apache.james.imap.api.display.HumanReadableText;
import org.apache.james.imap.api.message.request.ImapRequest;
import org.apache.james.imap.api.message.response.StatusResponseFactory;
import org.apache.james.imap.api.process.ImapProcessor;
import org.apache.james.imap.api.process.ImapSession;
import org.apache.james.imap.main.PathConverter;
import org.apache.james.imap.processor.AbstractMailboxProcessor;
import org.apache.james.mailbox.DefaultMailboxes;
import org.apache.james.mailbox.MailboxManager;
import org.apache.james.mailbox.MailboxSession;
import org.apache.james.mailbox.exception.BadCredentialsException;
import org.apache.james.mailbox.exception.ForbiddenDelegationException;
import org.apache.james.mailbox.exception.MailboxException;
import org.apache.james.mailbox.exception.MailboxExistsException;
import org.apache.james.mailbox.exception.UserDoesNotExistException;
import org.apache.james.mailbox.model.MailboxPath;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.util.AuditTrail;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

public abstract class AbstractAuthProcessor<R extends ImapRequest>
extends AbstractMailboxProcessor<R> {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractAuthProcessor.class);
    private static final String ATTRIBUTE_NUMBER_OF_FAILURES = "org.apache.james.imap.processor.imap4rev1.NUMBER_OF_FAILURES";
    private static final int MAX_FAILURES = 3;
    private ImapConfiguration imapConfiguration;
    private final PathConverter.Factory pathConverterFactory;

    public AbstractAuthProcessor(Class<R> acceptableClass, MailboxManager mailboxManager, StatusResponseFactory factory, MetricFactory metricFactory, PathConverter.Factory pathConverterFactory) {
        super(acceptableClass, mailboxManager, factory, metricFactory);
        this.pathConverterFactory = pathConverterFactory;
    }

    @Override
    public void configure(ImapConfiguration imapConfiguration) {
        super.configure(imapConfiguration);
        this.imapConfiguration = imapConfiguration;
    }

    protected void doAuth(AuthenticationAttempt authenticationAttempt, ImapSession session, ImapRequest request, ImapProcessor.Responder responder, HumanReadableText failed) {
        Preconditions.checkArgument((!authenticationAttempt.isDelegation() ? 1 : 0) != 0);
        try {
            boolean authFailure = false;
            if (authenticationAttempt.getAuthenticationId() == null) {
                authFailure = true;
            }
            if (!authFailure) {
                try {
                    MailboxSession mailboxSession = this.getMailboxManager().authenticate(authenticationAttempt.getAuthenticationId(), authenticationAttempt.getPassword()).withoutDelegation();
                    session.authenticated();
                    session.setMailboxSession(mailboxSession);
                    this.provisionInbox(session, this.getMailboxManager(), mailboxSession);
                    AuditTrail.entry().username(() -> mailboxSession.getUser().asString()).sessionId(() -> session.sessionId().asString()).protocol("IMAP").action("AUTH").log("IMAP Authentication succeeded.");
                    this.okComplete(request, responder);
                    session.stopDetectingCommandInjection();
                }
                catch (BadCredentialsException e) {
                    authFailure = true;
                    AuditTrail.entry().username(() -> authenticationAttempt.getAuthenticationId().asString()).protocol("IMAP").action("AUTH").log("IMAP Authentication failed because of bad credentials.");
                }
            }
            if (authFailure) {
                this.manageFailureCount(session, request, responder, failed);
            }
        }
        catch (MailboxException e) {
            LOGGER.error("Error encountered while login", (Throwable)e);
            this.no(request, responder, HumanReadableText.GENERIC_FAILURE_DURING_PROCESSING);
        }
    }

    protected void doAuthWithDelegation(AuthenticationAttempt authenticationAttempt, ImapSession session, ImapRequest request, ImapProcessor.Responder responder) {
        Preconditions.checkArgument((boolean)authenticationAttempt.isDelegation());
        Username givenUser = authenticationAttempt.getAuthenticationId();
        if (givenUser == null) {
            this.manageFailureCount(session, request, responder);
            return;
        }
        Username otherUser = authenticationAttempt.getDelegateUserName().orElseThrow();
        this.doAuthWithDelegation(() -> this.getMailboxManager().authenticate(givenUser, authenticationAttempt.getPassword()).as(otherUser), session, request, responder, givenUser, otherUser);
    }

    protected void doAuthWithDelegation(MailboxSessionAuthWithDelegationSupplier mailboxSessionSupplier, ImapSession session, ImapRequest request, ImapProcessor.Responder responder, Username authenticateUser, Username delegatorUser) {
        try {
            MailboxManager mailboxManager = this.getMailboxManager();
            MailboxSession mailboxSession = mailboxSessionSupplier.get();
            session.authenticated();
            session.setMailboxSession(mailboxSession);
            AuditTrail.entry().username(() -> mailboxSession.getLoggedInUser().map(Username::asString).orElse("")).sessionId(() -> session.sessionId().asString()).protocol("IMAP").action("AUTH").remoteIP(() -> Optional.ofNullable(session.getRemoteAddress())).parameters(() -> ImmutableMap.of((Object)"delegatorUser", (Object)mailboxSession.getUser().asString())).log("IMAP Authentication with delegation succeeded.");
            this.okComplete(request, responder);
            this.provisionInbox(session, mailboxManager, mailboxSession);
        }
        catch (BadCredentialsException e) {
            AuditTrail.entry().username(() -> ((Username)authenticateUser).asString()).protocol("IMAP").action("AUTH").remoteIP(() -> Optional.ofNullable(session.getRemoteAddress())).parameters(() -> ImmutableMap.of((Object)"delegatorUser", (Object)delegatorUser.asString())).log("IMAP Authentication with delegation failed because of bad credentials.");
            this.manageFailureCount(session, request, responder);
        }
        catch (UserDoesNotExistException e) {
            LOGGER.info("User does not exist", (Throwable)e);
            this.no(request, responder, HumanReadableText.USER_DOES_NOT_EXIST);
        }
        catch (ForbiddenDelegationException e) {
            LOGGER.info("Delegate forbidden", (Throwable)e);
            AuditTrail.entry().username(() -> ((Username)authenticateUser).asString()).protocol("IMAP").action("AUTH").parameters(() -> ImmutableMap.of((Object)"delegatorUser", (Object)delegatorUser.asString())).log("IMAP Authentication with delegation failed because of non existing delegation.");
            this.no(request, responder, HumanReadableText.DELEGATION_FORBIDDEN);
        }
        catch (MailboxException e) {
            LOGGER.info("Login failed", (Throwable)e);
            this.no(request, responder, HumanReadableText.GENERIC_FAILURE_DURING_PROCESSING);
        }
    }

    protected void provisionInbox(ImapSession session, MailboxManager mailboxManager, MailboxSession mailboxSession) throws MailboxException {
        MailboxPath inboxPath = this.pathConverterFactory.forSession(session).buildFullPath("INBOX");
        if (((Boolean)Mono.from((Publisher)mailboxManager.mailboxExists(inboxPath, mailboxSession)).block()).booleanValue()) {
            LOGGER.debug("INBOX exists. No need to create it.");
        } else {
            this.provisionMailbox("INBOX", mailboxManager, mailboxSession);
            if (this.imapConfiguration.isProvisionDefaultMailboxes()) {
                for (String mailbox : DefaultMailboxes.DEFAULT_MAILBOXES) {
                    this.provisionMailbox(mailbox, mailboxManager, mailboxSession);
                }
            }
        }
    }

    private void provisionMailbox(String mailbox, MailboxManager mailboxManager, MailboxSession mailboxSession) throws MailboxException {
        MailboxPath mailboxPath = this.pathConverterFactory.forSession(mailboxSession).buildFullPath(mailbox);
        if (((Boolean)Mono.from((Publisher)mailboxManager.mailboxExists(mailboxPath, mailboxSession)).block()).booleanValue()) {
            LOGGER.debug("{} exists. No need to create it.", (Object)mailbox);
            return;
        }
        try {
            mailboxManager.createMailbox(mailboxPath, MailboxManager.CreateOption.CREATE_SUBSCRIPTION, mailboxSession).ifPresentOrElse(id -> LOGGER.info("Provisioning mailbox {}. {} created.", (Object)mailbox, id), () -> LOGGER.warn("Provisioning mailbox {} successful. But no MailboxId have been returned.", (Object)mailbox));
        }
        catch (MailboxExistsException e) {
            LOGGER.warn("Mailbox {} created by concurrent call. Safe to ignore this exception.", (Object)mailbox);
        }
    }

    protected void manageFailureCount(ImapSession session, ImapRequest request, ImapProcessor.Responder responder) {
        this.manageFailureCount(session, request, responder, HumanReadableText.AUTHENTICATION_FAILED);
    }

    protected void manageFailureCount(ImapSession session, ImapRequest request, ImapProcessor.Responder responder, HumanReadableText failed) {
        Integer currentNumberOfFailures = (Integer)session.getAttribute(ATTRIBUTE_NUMBER_OF_FAILURES);
        int failures = currentNumberOfFailures == null ? 1 : currentNumberOfFailures + 1;
        if (failures < 3) {
            session.setAttribute(ATTRIBUTE_NUMBER_OF_FAILURES, failures);
            this.no(request, responder, failed);
        } else {
            LOGGER.info("Too many authentication failures. Closing connection.");
            this.bye(responder, HumanReadableText.TOO_MANY_FAILURES);
            session.logout().block();
        }
    }

    protected static AuthenticationAttempt delegation(Username authorizeId, Username authenticationId, String password) {
        return new AuthenticationAttempt(Optional.of(authorizeId), authenticationId, password);
    }

    protected static AuthenticationAttempt noDelegation(Username authenticationId, String password) {
        return new AuthenticationAttempt(Optional.empty(), authenticationId, password);
    }

    protected void authSuccess(Username username, ImapSession session, ImapRequest request, ImapProcessor.Responder responder) {
        session.authenticated();
        session.setMailboxSession(this.getMailboxManager().createSystemSession(username));
        this.okComplete(request, responder);
    }

    protected static class AuthenticationAttempt {
        private final Optional<Username> delegateUserName;
        private final Username authenticationId;
        private final String password;

        public AuthenticationAttempt(Optional<Username> delegateUserName, Username authenticationId, String password) {
            this.delegateUserName = delegateUserName;
            this.authenticationId = authenticationId;
            this.password = password;
        }

        public boolean isDelegation() {
            return this.delegateUserName.isPresent() && !this.delegateUserName.get().equals((Object)this.authenticationId);
        }

        public Optional<Username> getDelegateUserName() {
            return this.delegateUserName;
        }

        public Username getAuthenticationId() {
            return this.authenticationId;
        }

        public String getPassword() {
            return this.password;
        }
    }

    @FunctionalInterface
    protected static interface MailboxSessionAuthWithDelegationSupplier {
        public MailboxSession get() throws MailboxException;
    }
}

