/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 *   SPDX-License-Identifier: Apache-2.0
 */

package org.apache.jena.http;

import java.net.URI;
import java.util.Objects;

import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeAll;

import org.apache.jena.atlas.logging.Log;
import org.apache.jena.atlas.web.AuthScheme;
import org.apache.jena.fuseki.auth.Auth;
import org.apache.jena.fuseki.main.FusekiServer;
import org.apache.jena.fuseki.main.JettySecurityLib;
import org.apache.jena.http.auth.AuthEnv;
import org.apache.jena.sparql.core.DatasetGraph;
import org.apache.jena.sparql.core.DatasetGraphFactory;
import org.eclipse.jetty.security.SecurityHandler;
import org.eclipse.jetty.security.UserStore;

public class TestAuthBasicRemote extends AbstractTestAuthRemote {
    private static String user = "user";
    private static String password = "password";

    private static FusekiServer server = null;
    private static String dsEndpoint;
    private static URI dsEndpointURI;

    @Override
    protected String endpoint() {
        return dsEndpoint;
    }

    @Override
    protected URI endpointURI() {
        return dsEndpointURI;
    }

    @Override
    protected String user() {
        return user;
    }

    @Override
    protected String password() {
        return password;
    }

    @BeforeAll public static void beforeClass() {
        server = server("/ds", DatasetGraphFactory.createTxnMem(), user, password);
    }

    @AfterEach
    public void afterTest() {
        AuthEnv.get().clearAuthEnv();
    }

    private static FusekiServer server(String dsName, DatasetGraph dsg, String user, String password) {
        Objects.requireNonNull(user);
        Objects.requireNonNull(password);
        FusekiServer.Builder builder = FusekiServer.create()
            .port(0)
            .enablePing(true)
            .auth(AuthScheme.BASIC)
            .add(dsName, dsg);
        // Instead of a password file.
        if ( user != null ) {
            UserStore userStore = JettySecurityLib.makeUserStore(user, password);
            SecurityHandler sh = JettySecurityLib.makeSecurityHandler("TripleStore",  userStore, AuthScheme.BASIC);
            builder.securityHandler(sh)
                   .serverAuthPolicy(Auth.policyAllowSpecific(user));
        }
        FusekiServer server = builder.build();
        server.start();
        dsEndpoint = "http://localhost:"+server.getHttpPort()+"/ds";
        dsEndpointURI = URI.create(dsEndpoint);
        return server;
    }

    @AfterAll public static void afterClass() {
        dsEndpoint = null;
        dsEndpointURI = null;
        if ( server == null )
            return;
        try {
            server.stop();
            server = null;
        } catch (Throwable th) {
            Log.warn(TestAuthDigestRemote.class, "Exception in test suite shutdown", th);
        }
    }
}
