/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.conversation.spring;

import org.springframework.beans.factory.xml.BeanDefinitionDecorator;
import org.springframework.beans.factory.xml.ParserContext;
import org.springframework.beans.factory.config.BeanDefinitionHolder;
import org.w3c.dom.Node;

/**
 * Invoked by Spring when a bean definition is found in its config file which has an
 * xml attribute called "conversationName" in the orchestra namespace.
 * <p>
 * The value of the xml attribute is simply copied onto the BeanDefinition object.
 * <p>
 * See also class OrchestraNamespaceHandler.
 */
public class BeanDefinitionConversationNameAttrDecorator implements BeanDefinitionDecorator
{
    /**
     * The name of the xml attribute in the spring bean definition that is used by
     * orchestra as the conversation name.
     */
    public final static String XSD_CONVERSATION_NAME_ATTRIBUTE = "conversationName"; // NON-NLS
    
    /**
     * A unique key used to store the orchestra conversationName within the attributes map
     * of a spring bean definition.
     */
    public final static String CONVERSATION_NAME_ATTRIBUTE =
        "org.apache.myfaces.orchestra.spring.conversationName"; // NON-NLS

    public BeanDefinitionHolder decorate(
        Node node,
        BeanDefinitionHolder definition,
        ParserContext parserContext)
    {
        String conversationName = node.getTextContent();
        if (conversationName != null && conversationName.length() > 0)
        {
            definition.getBeanDefinition().setAttribute(CONVERSATION_NAME_ATTRIBUTE, conversationName);
        }

        return definition;
    }
}
