﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/memorydb/MemoryDBRequest.h>
#include <aws/memorydb/MemoryDB_EXPORTS.h>
#include <aws/memorydb/model/IpDiscovery.h>
#include <aws/memorydb/model/ReplicaConfigurationRequest.h>
#include <aws/memorydb/model/ShardConfigurationRequest.h>

#include <utility>

namespace Aws {
namespace MemoryDB {
namespace Model {

/**
 */
class UpdateClusterRequest : public MemoryDBRequest {
 public:
  AWS_MEMORYDB_API UpdateClusterRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateCluster"; }

  AWS_MEMORYDB_API Aws::String SerializePayload() const override;

  AWS_MEMORYDB_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the cluster to update.</p>
   */
  inline const Aws::String& GetClusterName() const { return m_clusterName; }
  inline bool ClusterNameHasBeenSet() const { return m_clusterNameHasBeenSet; }
  template <typename ClusterNameT = Aws::String>
  void SetClusterName(ClusterNameT&& value) {
    m_clusterNameHasBeenSet = true;
    m_clusterName = std::forward<ClusterNameT>(value);
  }
  template <typename ClusterNameT = Aws::String>
  UpdateClusterRequest& WithClusterName(ClusterNameT&& value) {
    SetClusterName(std::forward<ClusterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the cluster to update.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateClusterRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The SecurityGroupIds to update.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroupIds() const { return m_securityGroupIds; }
  inline bool SecurityGroupIdsHasBeenSet() const { return m_securityGroupIdsHasBeenSet; }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds = std::forward<SecurityGroupIdsT>(value);
  }
  template <typename SecurityGroupIdsT = Aws::Vector<Aws::String>>
  UpdateClusterRequest& WithSecurityGroupIds(SecurityGroupIdsT&& value) {
    SetSecurityGroupIds(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  template <typename SecurityGroupIdsT = Aws::String>
  UpdateClusterRequest& AddSecurityGroupIds(SecurityGroupIdsT&& value) {
    m_securityGroupIdsHasBeenSet = true;
    m_securityGroupIds.emplace_back(std::forward<SecurityGroupIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the weekly time range during which maintenance on the cluster is
   * performed. It is specified as a range in the format ddd:hh24:mi-ddd:hh24:mi (24H
   * Clock UTC). The minimum maintenance window is a 60 minute period.</p> <p>Valid
   * values for <code>ddd</code> are:</p> <ul> <li> <p> <code>sun</code> </p> </li>
   * <li> <p> <code>mon</code> </p> </li> <li> <p> <code>tue</code> </p> </li> <li>
   * <p> <code>wed</code> </p> </li> <li> <p> <code>thu</code> </p> </li> <li> <p>
   * <code>fri</code> </p> </li> <li> <p> <code>sat</code> </p> </li> </ul>
   * <p>Example: <code>sun:23:00-mon:01:30</code> </p>
   */
  inline const Aws::String& GetMaintenanceWindow() const { return m_maintenanceWindow; }
  inline bool MaintenanceWindowHasBeenSet() const { return m_maintenanceWindowHasBeenSet; }
  template <typename MaintenanceWindowT = Aws::String>
  void SetMaintenanceWindow(MaintenanceWindowT&& value) {
    m_maintenanceWindowHasBeenSet = true;
    m_maintenanceWindow = std::forward<MaintenanceWindowT>(value);
  }
  template <typename MaintenanceWindowT = Aws::String>
  UpdateClusterRequest& WithMaintenanceWindow(MaintenanceWindowT&& value) {
    SetMaintenanceWindow(std::forward<MaintenanceWindowT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The SNS topic ARN to update.</p>
   */
  inline const Aws::String& GetSnsTopicArn() const { return m_snsTopicArn; }
  inline bool SnsTopicArnHasBeenSet() const { return m_snsTopicArnHasBeenSet; }
  template <typename SnsTopicArnT = Aws::String>
  void SetSnsTopicArn(SnsTopicArnT&& value) {
    m_snsTopicArnHasBeenSet = true;
    m_snsTopicArn = std::forward<SnsTopicArnT>(value);
  }
  template <typename SnsTopicArnT = Aws::String>
  UpdateClusterRequest& WithSnsTopicArn(SnsTopicArnT&& value) {
    SetSnsTopicArn(std::forward<SnsTopicArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the Amazon SNS notification topic. Notifications are sent only
   * if the status is active.</p>
   */
  inline const Aws::String& GetSnsTopicStatus() const { return m_snsTopicStatus; }
  inline bool SnsTopicStatusHasBeenSet() const { return m_snsTopicStatusHasBeenSet; }
  template <typename SnsTopicStatusT = Aws::String>
  void SetSnsTopicStatus(SnsTopicStatusT&& value) {
    m_snsTopicStatusHasBeenSet = true;
    m_snsTopicStatus = std::forward<SnsTopicStatusT>(value);
  }
  template <typename SnsTopicStatusT = Aws::String>
  UpdateClusterRequest& WithSnsTopicStatus(SnsTopicStatusT&& value) {
    SetSnsTopicStatus(std::forward<SnsTopicStatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the parameter group to update.</p>
   */
  inline const Aws::String& GetParameterGroupName() const { return m_parameterGroupName; }
  inline bool ParameterGroupNameHasBeenSet() const { return m_parameterGroupNameHasBeenSet; }
  template <typename ParameterGroupNameT = Aws::String>
  void SetParameterGroupName(ParameterGroupNameT&& value) {
    m_parameterGroupNameHasBeenSet = true;
    m_parameterGroupName = std::forward<ParameterGroupNameT>(value);
  }
  template <typename ParameterGroupNameT = Aws::String>
  UpdateClusterRequest& WithParameterGroupName(ParameterGroupNameT&& value) {
    SetParameterGroupName(std::forward<ParameterGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The daily time range (in UTC) during which MemoryDB begins taking a daily
   * snapshot of your cluster.</p>
   */
  inline const Aws::String& GetSnapshotWindow() const { return m_snapshotWindow; }
  inline bool SnapshotWindowHasBeenSet() const { return m_snapshotWindowHasBeenSet; }
  template <typename SnapshotWindowT = Aws::String>
  void SetSnapshotWindow(SnapshotWindowT&& value) {
    m_snapshotWindowHasBeenSet = true;
    m_snapshotWindow = std::forward<SnapshotWindowT>(value);
  }
  template <typename SnapshotWindowT = Aws::String>
  UpdateClusterRequest& WithSnapshotWindow(SnapshotWindowT&& value) {
    SetSnapshotWindow(std::forward<SnapshotWindowT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of days for which MemoryDB retains automatic cluster snapshots
   * before deleting them. For example, if you set SnapshotRetentionLimit to 5, a
   * snapshot that was taken today is retained for 5 days before being deleted.</p>
   */
  inline int GetSnapshotRetentionLimit() const { return m_snapshotRetentionLimit; }
  inline bool SnapshotRetentionLimitHasBeenSet() const { return m_snapshotRetentionLimitHasBeenSet; }
  inline void SetSnapshotRetentionLimit(int value) {
    m_snapshotRetentionLimitHasBeenSet = true;
    m_snapshotRetentionLimit = value;
  }
  inline UpdateClusterRequest& WithSnapshotRetentionLimit(int value) {
    SetSnapshotRetentionLimit(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A valid node type that you want to scale this cluster up or down to.</p>
   */
  inline const Aws::String& GetNodeType() const { return m_nodeType; }
  inline bool NodeTypeHasBeenSet() const { return m_nodeTypeHasBeenSet; }
  template <typename NodeTypeT = Aws::String>
  void SetNodeType(NodeTypeT&& value) {
    m_nodeTypeHasBeenSet = true;
    m_nodeType = std::forward<NodeTypeT>(value);
  }
  template <typename NodeTypeT = Aws::String>
  UpdateClusterRequest& WithNodeType(NodeTypeT&& value) {
    SetNodeType(std::forward<NodeTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the engine to be used for the cluster.</p>
   */
  inline const Aws::String& GetEngine() const { return m_engine; }
  inline bool EngineHasBeenSet() const { return m_engineHasBeenSet; }
  template <typename EngineT = Aws::String>
  void SetEngine(EngineT&& value) {
    m_engineHasBeenSet = true;
    m_engine = std::forward<EngineT>(value);
  }
  template <typename EngineT = Aws::String>
  UpdateClusterRequest& WithEngine(EngineT&& value) {
    SetEngine(std::forward<EngineT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The upgraded version of the engine to be run on the nodes. You can upgrade to
   * a newer engine version, but you cannot downgrade to an earlier engine version.
   * If you want to use an earlier engine version, you must delete the existing
   * cluster and create it anew with the earlier engine version.</p>
   */
  inline const Aws::String& GetEngineVersion() const { return m_engineVersion; }
  inline bool EngineVersionHasBeenSet() const { return m_engineVersionHasBeenSet; }
  template <typename EngineVersionT = Aws::String>
  void SetEngineVersion(EngineVersionT&& value) {
    m_engineVersionHasBeenSet = true;
    m_engineVersion = std::forward<EngineVersionT>(value);
  }
  template <typename EngineVersionT = Aws::String>
  UpdateClusterRequest& WithEngineVersion(EngineVersionT&& value) {
    SetEngineVersion(std::forward<EngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of replicas that will reside in each shard.</p>
   */
  inline const ReplicaConfigurationRequest& GetReplicaConfiguration() const { return m_replicaConfiguration; }
  inline bool ReplicaConfigurationHasBeenSet() const { return m_replicaConfigurationHasBeenSet; }
  template <typename ReplicaConfigurationT = ReplicaConfigurationRequest>
  void SetReplicaConfiguration(ReplicaConfigurationT&& value) {
    m_replicaConfigurationHasBeenSet = true;
    m_replicaConfiguration = std::forward<ReplicaConfigurationT>(value);
  }
  template <typename ReplicaConfigurationT = ReplicaConfigurationRequest>
  UpdateClusterRequest& WithReplicaConfiguration(ReplicaConfigurationT&& value) {
    SetReplicaConfiguration(std::forward<ReplicaConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of shards in the cluster.</p>
   */
  inline const ShardConfigurationRequest& GetShardConfiguration() const { return m_shardConfiguration; }
  inline bool ShardConfigurationHasBeenSet() const { return m_shardConfigurationHasBeenSet; }
  template <typename ShardConfigurationT = ShardConfigurationRequest>
  void SetShardConfiguration(ShardConfigurationT&& value) {
    m_shardConfigurationHasBeenSet = true;
    m_shardConfiguration = std::forward<ShardConfigurationT>(value);
  }
  template <typename ShardConfigurationT = ShardConfigurationRequest>
  UpdateClusterRequest& WithShardConfiguration(ShardConfigurationT&& value) {
    SetShardConfiguration(std::forward<ShardConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Access Control List that is associated with the cluster.</p>
   */
  inline const Aws::String& GetACLName() const { return m_aCLName; }
  inline bool ACLNameHasBeenSet() const { return m_aCLNameHasBeenSet; }
  template <typename ACLNameT = Aws::String>
  void SetACLName(ACLNameT&& value) {
    m_aCLNameHasBeenSet = true;
    m_aCLName = std::forward<ACLNameT>(value);
  }
  template <typename ACLNameT = Aws::String>
  UpdateClusterRequest& WithACLName(ACLNameT&& value) {
    SetACLName(std::forward<ACLNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The mechanism for discovering IP addresses for the cluster discovery
   * protocol. Valid values are 'ipv4' or 'ipv6'. When set to 'ipv4', cluster
   * discovery functions such as cluster slots, cluster shards, and cluster nodes
   * will return IPv4 addresses for cluster nodes. When set to 'ipv6', the cluster
   * discovery functions return IPv6 addresses for cluster nodes. The value must be
   * compatible with the NetworkType parameter. If not specified, the default is
   * 'ipv4'.</p>
   */
  inline IpDiscovery GetIpDiscovery() const { return m_ipDiscovery; }
  inline bool IpDiscoveryHasBeenSet() const { return m_ipDiscoveryHasBeenSet; }
  inline void SetIpDiscovery(IpDiscovery value) {
    m_ipDiscoveryHasBeenSet = true;
    m_ipDiscovery = value;
  }
  inline UpdateClusterRequest& WithIpDiscovery(IpDiscovery value) {
    SetIpDiscovery(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterName;

  Aws::String m_description;

  Aws::Vector<Aws::String> m_securityGroupIds;

  Aws::String m_maintenanceWindow;

  Aws::String m_snsTopicArn;

  Aws::String m_snsTopicStatus;

  Aws::String m_parameterGroupName;

  Aws::String m_snapshotWindow;

  int m_snapshotRetentionLimit{0};

  Aws::String m_nodeType;

  Aws::String m_engine;

  Aws::String m_engineVersion;

  ReplicaConfigurationRequest m_replicaConfiguration;

  ShardConfigurationRequest m_shardConfiguration;

  Aws::String m_aCLName;

  IpDiscovery m_ipDiscovery{IpDiscovery::NOT_SET};
  bool m_clusterNameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_securityGroupIdsHasBeenSet = false;
  bool m_maintenanceWindowHasBeenSet = false;
  bool m_snsTopicArnHasBeenSet = false;
  bool m_snsTopicStatusHasBeenSet = false;
  bool m_parameterGroupNameHasBeenSet = false;
  bool m_snapshotWindowHasBeenSet = false;
  bool m_snapshotRetentionLimitHasBeenSet = false;
  bool m_nodeTypeHasBeenSet = false;
  bool m_engineHasBeenSet = false;
  bool m_engineVersionHasBeenSet = false;
  bool m_replicaConfigurationHasBeenSet = false;
  bool m_shardConfigurationHasBeenSet = false;
  bool m_aCLNameHasBeenSet = false;
  bool m_ipDiscoveryHasBeenSet = false;
};

}  // namespace Model
}  // namespace MemoryDB
}  // namespace Aws
