﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-idp/CognitoIdentityProviderRequest.h>
#include <aws/cognito-idp/CognitoIdentityProvider_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CognitoIdentityProvider {
namespace Model {

/**
 * <p>Represents the request to list the devices.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cognito-idp-2016-04-18/ListDevicesRequest">AWS
 * API Reference</a></p>
 */
class ListDevicesRequest : public CognitoIdentityProviderRequest {
 public:
  AWS_COGNITOIDENTITYPROVIDER_API ListDevicesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListDevices"; }

  AWS_COGNITOIDENTITYPROVIDER_API Aws::String SerializePayload() const override;

  AWS_COGNITOIDENTITYPROVIDER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A valid access token that Amazon Cognito issued to the currently signed-in
   * user. Must include a scope claim for
   * <code>aws.cognito.signin.user.admin</code>.</p>
   */
  inline const Aws::String& GetAccessToken() const { return m_accessToken; }
  inline bool AccessTokenHasBeenSet() const { return m_accessTokenHasBeenSet; }
  template <typename AccessTokenT = Aws::String>
  void SetAccessToken(AccessTokenT&& value) {
    m_accessTokenHasBeenSet = true;
    m_accessToken = std::forward<AccessTokenT>(value);
  }
  template <typename AccessTokenT = Aws::String>
  ListDevicesRequest& WithAccessToken(AccessTokenT&& value) {
    SetAccessToken(std::forward<AccessTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of devices that you want Amazon Cognito to return in the
   * response.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline ListDevicesRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This API operation returns a limited number of results. The pagination token
   * is an identifier that you can present in an additional API request with the same
   * parameters. When you include the pagination token, Amazon Cognito returns the
   * next set of items after the current list. Subsequent requests return a new
   * pagination token. By use of this token, you can paginate through the full list
   * of items.</p>
   */
  inline const Aws::String& GetPaginationToken() const { return m_paginationToken; }
  inline bool PaginationTokenHasBeenSet() const { return m_paginationTokenHasBeenSet; }
  template <typename PaginationTokenT = Aws::String>
  void SetPaginationToken(PaginationTokenT&& value) {
    m_paginationTokenHasBeenSet = true;
    m_paginationToken = std::forward<PaginationTokenT>(value);
  }
  template <typename PaginationTokenT = Aws::String>
  ListDevicesRequest& WithPaginationToken(PaginationTokenT&& value) {
    SetPaginationToken(std::forward<PaginationTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accessToken;

  int m_limit{0};

  Aws::String m_paginationToken;
  bool m_accessTokenHasBeenSet = false;
  bool m_limitHasBeenSet = false;
  bool m_paginationTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoIdentityProvider
}  // namespace Aws
