/*********************************************************************************
 * Copyright (c) 2021 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */ 

package org.eclipse.app4mc.amalthea.model.editor.contribution.registry;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.eclipse.app4mc.amalthea.model.editor.contribution.service.CreationService;
import org.eclipse.app4mc.amalthea.model.provider.TransientItemProvider;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.osgi.service.component.annotations.ReferencePolicyOption;

@Component(service = CreationServiceRegistry.class)
public class CreationServiceRegistry extends ModelServiceRegistry<CreationService> {
	
	@Reference(
			cardinality = ReferenceCardinality.MULTIPLE,
			policy = ReferencePolicy.DYNAMIC,
			policyOption = ReferencePolicyOption.GREEDY)
	void bindCreationService(CreationService creationService, Map<String, Object> properties) {
		super.bindService(creationService, properties);
	}

	void unbindCreationService(CreationService creationService, Map<String, Object> properties) {
		super.unbindService(creationService, properties);
	}

	public List<RegistryServiceWrapper<CreationService>> getServices(Object selected) {
		if (selected == null) return Collections.emptyList();

		// defaults
		List<Class<?>> interfaces = Collections.singletonList(selected.getClass());
		List<EStructuralFeature> supportedFeatures = null;

		// in case of TransientItemProvider:
		//  - replace selected object with target object
		//  - get supported features
		if (selected instanceof TransientItemProvider) {
			TransientItemProvider provider = (TransientItemProvider) selected;
			selected = provider.getTarget();
			supportedFeatures = provider.myFeatures();
		}

		// in case of EObject:
		//  - replace selected object class with its interfaces
		if (selected instanceof EObject) {
			Class<? extends Object> class1 = selected.getClass();
			interfaces = Arrays.asList(class1.getInterfaces());			
		}

		return getServices(interfaces, supportedFeatures);
	}
			
	public List<RegistryServiceWrapper<CreationService>> getServices(List<Class<?>> classes, List<EStructuralFeature> supportedFeatures) {

		List<RegistryServiceWrapper<CreationService>> services = super.getServices(classes);

		if (supportedFeatures == null || supportedFeatures.isEmpty()) {
			return services;
		}
			
		return services.stream()
				.filter(s -> s.getServiceInstance().modelFeature() == null
							|| supportedFeatures.contains(s.getServiceInstance().modelFeature()))
				.collect(Collectors.toList());
	}

}
