/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.util;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.Locale;
import java.util.TimeZone;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.jgit.errors.ConfigInvalidException;
import org.eclipse.jgit.errors.CorruptObjectException;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.ObjectChecker;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.storage.file.FileBasedConfig;
import org.eclipse.jgit.storage.file.UserConfigFile;
import org.eclipse.jgit.util.FS;
import org.eclipse.jgit.util.StringUtils;
import org.eclipse.jgit.util.time.MonotonicClock;
import org.eclipse.jgit.util.time.MonotonicSystemClock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class SystemReader {
    private static final Logger LOG = LoggerFactory.getLogger(SystemReader.class);
    private static final SystemReader DEFAULT;
    private static volatile Boolean isMacOS;
    private static volatile Boolean isWindows;
    private static volatile Boolean isLinux;
    private static final String GIT_TRACE_PERFORMANCE = "GIT_TRACE_PERFORMANCE";
    private static final boolean performanceTrace;
    private static volatile SystemReader INSTANCE;
    private ObjectChecker platformChecker;
    private AtomicReference<FileBasedConfig> systemConfig = new AtomicReference();
    private AtomicReference<FileBasedConfig> userConfig = new AtomicReference();
    private AtomicReference<FileBasedConfig> jgitConfig = new AtomicReference();
    private volatile Charset defaultCharset;

    static {
        performanceTrace = SystemReader.initPerformanceTrace();
        Default r = new Default();
        r.init();
        INSTANCE = DEFAULT = r;
    }

    private static boolean initPerformanceTrace() {
        String val = System.getenv(GIT_TRACE_PERFORMANCE);
        if (val == null) {
            val = System.getenv(GIT_TRACE_PERFORMANCE);
        }
        if (val != null) {
            return Boolean.valueOf(val);
        }
        return false;
    }

    public static SystemReader getInstance() {
        return INSTANCE;
    }

    public static void setInstance(SystemReader newReader) {
        isMacOS = null;
        isWindows = null;
        isLinux = null;
        if (newReader == null) {
            INSTANCE = DEFAULT;
        } else {
            newReader.init();
            INSTANCE = newReader;
        }
    }

    private void init() {
        if (this.platformChecker == null) {
            this.setPlatformChecker();
        }
    }

    protected final void setPlatformChecker() {
        this.platformChecker = new ObjectChecker().setSafeForWindows(this.isWindows()).setSafeForMacOS(this.isMacOS());
    }

    public abstract String getHostname();

    public abstract String getenv(String var1);

    public abstract String getProperty(String var1);

    public abstract FileBasedConfig openUserConfig(Config var1, FS var2);

    public abstract FileBasedConfig openSystemConfig(Config var1, FS var2);

    public abstract FileBasedConfig openJGitConfig(Config var1, FS var2);

    public StoredConfig getUserConfig() throws ConfigInvalidException, IOException {
        FileBasedConfig c = this.userConfig.get();
        if (c == null) {
            this.userConfig.compareAndSet(null, this.openUserConfig(this.getSystemConfig(), FS.DETECTED));
            c = this.userConfig.get();
        }
        this.updateAll(c);
        return c;
    }

    public StoredConfig getJGitConfig() throws ConfigInvalidException, IOException {
        FileBasedConfig c = this.jgitConfig.get();
        if (c == null) {
            this.jgitConfig.compareAndSet(null, this.openJGitConfig(null, FS.DETECTED));
            c = this.jgitConfig.get();
        }
        this.updateAll(c);
        return c;
    }

    public StoredConfig getSystemConfig() throws ConfigInvalidException, IOException {
        FileBasedConfig c = this.systemConfig.get();
        if (c == null) {
            this.systemConfig.compareAndSet(null, this.openSystemConfig(this.getJGitConfig(), FS.DETECTED));
            c = this.systemConfig.get();
        }
        this.updateAll(c);
        return c;
    }

    public Path getXdgConfigDirectory(FS fileSystem) {
        String configHomePath = this.getenv("XDG_CONFIG_HOME");
        if (StringUtils.isEmptyOrNull(configHomePath)) {
            File home = fileSystem.userHome();
            if (home == null) {
                return null;
            }
            configHomePath = new File(home, ".config").getAbsolutePath();
        }
        try {
            return Paths.get(configHomePath, new String[0]);
        }
        catch (InvalidPathException e) {
            LOG.error(JGitText.get().logXDGConfigHomeInvalid, (Object)configHomePath, (Object)e);
            return null;
        }
    }

    public Path getXdgCacheDirectory(FS fileSystem) {
        String cacheHomePath = this.getenv("XDG_CACHE_HOME");
        if (StringUtils.isEmptyOrNull(cacheHomePath)) {
            File home = fileSystem.userHome();
            if (home == null) {
                return null;
            }
            cacheHomePath = new File(home, ".cache").getAbsolutePath();
        }
        try {
            return Paths.get(cacheHomePath, new String[0]);
        }
        catch (InvalidPathException e) {
            LOG.error(JGitText.get().logXDGCacheHomeInvalid, (Object)cacheHomePath, (Object)e);
            return null;
        }
    }

    private void updateAll(Config config) throws ConfigInvalidException, IOException {
        FileBasedConfig cfg;
        if (config == null) {
            return;
        }
        this.updateAll(config.getBaseConfig());
        if (config instanceof FileBasedConfig && (cfg = (FileBasedConfig)config).isOutdated()) {
            LOG.debug("loading config {}", (Object)cfg);
            cfg.load();
        }
    }

    @Deprecated(since="7.1")
    public abstract long getCurrentTime();

    public Instant now() {
        return Instant.ofEpochMilli(this.getCurrentTime());
    }

    public LocalDateTime civilNow() {
        return LocalDateTime.ofInstant(this.now(), this.getTimeZoneId());
    }

    public MonotonicClock getClock() {
        return new MonotonicSystemClock();
    }

    @Deprecated(since="7.1")
    public abstract int getTimezone(long var1);

    public ZoneOffset getTimeZoneAt(Instant when) {
        return this.getTimeZoneId().getRules().getOffset(when);
    }

    @Deprecated(since="7.1")
    public TimeZone getTimeZone() {
        return TimeZone.getDefault();
    }

    public ZoneId getTimeZoneId() {
        return ZoneId.systemDefault();
    }

    public Locale getLocale() {
        return Locale.getDefault();
    }

    public Charset getDefaultCharset() {
        Charset result = this.defaultCharset;
        if (result == null) {
            String encoding = this.getProperty("native.encoding");
            try {
                if (!StringUtils.isEmptyOrNull(encoding)) {
                    result = Charset.forName(encoding);
                }
            }
            catch (IllegalCharsetNameException | UnsupportedCharsetException e) {
                LOG.error(JGitText.get().logInvalidDefaultCharset, (Object)encoding);
            }
            if (result == null) {
                result = Charset.defaultCharset();
            }
            this.defaultCharset = result;
        }
        return result;
    }

    public SimpleDateFormat getSimpleDateFormat(String pattern) {
        return new SimpleDateFormat(pattern);
    }

    public SimpleDateFormat getSimpleDateFormat(String pattern, Locale locale) {
        return new SimpleDateFormat(pattern, locale);
    }

    public DateFormat getDateTimeInstance(int dateStyle, int timeStyle) {
        return DateFormat.getDateTimeInstance(dateStyle, timeStyle);
    }

    public boolean isWindows() {
        if (isWindows == null) {
            String osDotName = this.getOsName();
            isWindows = osDotName.startsWith("Windows");
        }
        return isWindows;
    }

    public boolean isMacOS() {
        if (isMacOS == null) {
            String osDotName = this.getOsName();
            isMacOS = "Mac OS X".equals(osDotName) || "Darwin".equals(osDotName);
        }
        return isMacOS;
    }

    public boolean isLinux() {
        if (isLinux == null) {
            String osname = this.getOsName();
            isLinux = osname.toLowerCase().startsWith("linux");
        }
        return isLinux;
    }

    public boolean isPerformanceTraceEnabled() {
        return performanceTrace;
    }

    private String getOsName() {
        return this.getProperty("os.name");
    }

    public void checkPath(String path) throws CorruptObjectException {
        this.platformChecker.checkPath(path);
    }

    public void checkPath(byte[] path) throws CorruptObjectException {
        this.platformChecker.checkPath(path, 0, path.length);
    }

    private static class Default
    extends SystemReader {
        private volatile String hostname;

        private Default() {
        }

        @Override
        public String getenv(String variable) {
            return System.getenv(variable);
        }

        @Override
        public String getProperty(String key) {
            return System.getProperty(key);
        }

        @Override
        public FileBasedConfig openSystemConfig(Config parent, FS fs) {
            File configFile;
            if (StringUtils.isEmptyOrNull(this.getenv("GIT_CONFIG_NOSYSTEM")) && (configFile = fs.getGitSystemConfig()) != null) {
                return new FileBasedConfig(parent, configFile, fs);
            }
            return new FileBasedConfig(parent, null, fs){

                @Override
                public void load() {
                }

                @Override
                public boolean isOutdated() {
                    return false;
                }
            };
        }

        @Override
        public FileBasedConfig openUserConfig(Config parent, FS fs) {
            File homeFile = new File(fs.userHome(), ".gitconfig");
            Path xdgPath = this.getXdgConfigDirectory(fs);
            if (xdgPath != null) {
                Path configPath = xdgPath.resolve("git").resolve("config");
                return new UserConfigFile(parent, homeFile, configPath.toFile(), fs);
            }
            return new FileBasedConfig(parent, homeFile, fs);
        }

        @Override
        public FileBasedConfig openJGitConfig(Config parent, FS fs) {
            Path xdgPath = this.getXdgConfigDirectory(fs);
            if (xdgPath != null) {
                Path configPath = xdgPath.resolve("jgit").resolve("config");
                return new FileBasedConfig(parent, configPath.toFile(), fs);
            }
            return new FileBasedConfig(parent, new File(fs.userHome(), ".jgitconfig"), fs);
        }

        @Override
        public String getHostname() {
            if (this.hostname == null) {
                try {
                    InetAddress localMachine = InetAddress.getLocalHost();
                    this.hostname = localMachine.getCanonicalHostName();
                }
                catch (UnknownHostException e) {
                    this.hostname = "localhost";
                }
                assert (this.hostname != null);
            }
            return this.hostname;
        }

        @Override
        public long getCurrentTime() {
            return System.currentTimeMillis();
        }

        @Override
        public Instant now() {
            return Instant.now();
        }

        @Override
        public int getTimezone(long when) {
            return this.getTimeZone().getOffset(when) / 60000;
        }
    }

    public static class Delegate
    extends SystemReader {
        private final SystemReader delegate;

        public Delegate(SystemReader delegate) {
            this.delegate = delegate;
        }

        @Override
        public String getHostname() {
            return this.delegate.getHostname();
        }

        @Override
        public String getenv(String variable) {
            return this.delegate.getenv(variable);
        }

        @Override
        public String getProperty(String key) {
            return this.delegate.getProperty(key);
        }

        @Override
        public FileBasedConfig openUserConfig(Config parent, FS fs) {
            return this.delegate.openUserConfig(parent, fs);
        }

        @Override
        public FileBasedConfig openSystemConfig(Config parent, FS fs) {
            return this.delegate.openSystemConfig(parent, fs);
        }

        @Override
        public FileBasedConfig openJGitConfig(Config parent, FS fs) {
            return this.delegate.openJGitConfig(parent, fs);
        }

        @Override
        public long getCurrentTime() {
            return this.delegate.getCurrentTime();
        }

        @Override
        public Instant now() {
            return this.delegate.now();
        }

        @Override
        public int getTimezone(long when) {
            return this.delegate.getTimezone(when);
        }

        @Override
        public ZoneOffset getTimeZoneAt(Instant when) {
            return this.delegate.getTimeZoneAt(when);
        }
    }
}

