//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2023, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.datasynth.options;

import static org.eclipse.escet.common.java.Sets.set;
import static org.eclipse.escet.common.java.Strings.fmt;

import java.util.Locale;
import java.util.stream.Collectors;

import org.eclipse.escet.cif.datasynth.settings.CifDataSynthesisSettingsDefaults;
import org.eclipse.escet.cif.datasynth.settings.FixedPointComputation;
import org.eclipse.escet.cif.datasynth.settings.FixedPointComputationsOrder;
import org.eclipse.escet.common.app.framework.options.Options;
import org.eclipse.escet.common.app.framework.options.StringOption;
import org.eclipse.escet.common.app.framework.output.OutputProvider;
import org.eclipse.escet.common.java.exceptions.InvalidOptionException;

/** Option to specify the order of the fixed-point computations. */
public class FixedPointComputationsOrderOption extends StringOption {
    /** The option description. */
    private static final String DESCRIPTION = "Specify the order in which the fixed-point computations are to be "
            + "performed during synthesis. Multiple fixed-point computations are performed (depending on other "
            + "settings), to compute the non-blocking states, states satisfying the reachability requirements, "
            + "controllable states and reachable states. Each computation must be specified exactly once. "
            + "Specify \"nonblock\", \"reach-reqs\", \"ctrl\", and \"reach\", "
            + "in the desired order, joined by commas. "
            + "For instance, specify \"ctrl,reach-reqs,reach,nonblock\" or \"reach,reach-reqs,ctrl,nonblock\".";

    /** The default value of the option. */
    private static final String DEFAULT_VALUE = toString(
            CifDataSynthesisSettingsDefaults.FIXED_POINT_COMPUTATIONS_ORDER_DEFAULT);

    /** Constructor for the {@link FixedPointComputationsOrderOption} class. */
    public FixedPointComputationsOrderOption() {
        super(
                // name
                "Fixed-point computations order",

                // description
                DESCRIPTION + fmt(" [DEFAULT=%s]", DEFAULT_VALUE),

                // cmdShort
                null,

                // cmdLong
                "fixed-point-order",

                // cmdValue
                "ORDER",

                // defaultValue
                DEFAULT_VALUE,

                // emptyAsNull
                false,

                // showInDialog
                true,

                // optDialogDescr
                DESCRIPTION,

                // optDialogLabelText
                "Order:"

        );
    }

    /**
     * Returns the value of the {@link FixedPointComputationsOrderOption} option.
     *
     * @return The value of the {@link FixedPointComputationsOrderOption} option.
     */
    public static FixedPointComputationsOrder getOrder() {
        // Get option text.
        String optionText = Options.get(FixedPointComputationsOrderOption.class);

        // Detect the old syntax and convert it to the new syntax. If the old syntax is used, warn the user.
        if (!optionText.contains(",")) {
            switch (optionText.trim()) {
                case "nonblock-ctrl-reach":
                case "nonblock-reach-ctrl":
                case "ctrl-nonblock-reach":
                case "ctrl-reach-nonblock":
                case "reach-nonblock-ctrl":
                case "reach-ctrl-nonblock":
                    OutputProvider.warn("Fixed-point computations order \"%s\" is deprecated. The old syntax with "
                            + "\"-\" characters between the computation names will be removed in a future version. "
                            + "Use the new syntax with commas between the computation names instead. Consult the "
                            + "option's help text and the tool's documentation for more information.", optionText);
                    optionText = optionText.replace('-', ','); // Convert to the new syntax.
                    optionText = optionText.replace("nonblock", "nonblock,reach-reqs"); // Add missing computation.
            }
        }

        // The option text is not using the old syntax. Proceed with parsing it using the new syntax.

        // Parse the option text.
        String[] parts = optionText.split(",");
        FixedPointComputation[] computations = new FixedPointComputation[parts.length];
        for (int i = 0; i < parts.length; i++) {
            String part = parts[i].trim();
            String partName = part.replace('-', '_').toUpperCase(Locale.US);
            try {
                computations[i] = FixedPointComputation.valueOf(partName);
            } catch (IllegalArgumentException e) {
                throw new InvalidOptionException(fmt("Invalid fixed-point computations order \"%s\": \"%s\" is not a "
                        + "valid fixed-point computation name.", optionText, part));
            }
        }

        // Check that the number of computations is correct.
        int expectedCount = FixedPointComputation.values().length;
        int actualCount = computations.length;
        if (actualCount != expectedCount) {
            throw new InvalidOptionException(fmt("Invalid fixed-point computations order \"%s\": expected %d "
                    + "fixed-point computations, but found %d.", optionText, expectedCount, actualCount));
        }

        // Check that all computations are present exactly once.
        if (!set(FixedPointComputation.values()).equals(set(computations))) {
            throw new InvalidOptionException(fmt("Invalid fixed-point computations order \"%s\": each fixed-point "
                    + "computation must be specified exactly once.", optionText));
        }

        // Return the fixed-point computations order.
        return new FixedPointComputationsOrder(computations);
    }

    /**
     * Converts a fixed-point computations order to the textual representation used by this option.
     *
     * @param order The fixed-point computations order.
     * @return The textual representation.
     */
    private static String toString(FixedPointComputationsOrder order) {
        return order.computations.stream().map(c -> c.toString().replace('_', '-').toLowerCase(Locale.US))
                .collect(Collectors.joining(","));
    }
}
