//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2024, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.cif2mcrl2;

import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newBinaryExpression;
import static org.eclipse.escet.cif.metamodel.java.CifConstructors.newBoolType;
import static org.eclipse.escet.common.emf.EMFHelper.deepclone;
import static org.eclipse.escet.common.java.Lists.first;
import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Lists.listc;
import static org.eclipse.escet.common.java.Lists.single;
import static org.eclipse.escet.common.java.Maps.mapc;
import static org.eclipse.escet.common.java.Strings.fmt;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.escet.cif.cif2cif.ElimAlgVariables;
import org.eclipse.escet.cif.cif2cif.ElimComponentDefInst;
import org.eclipse.escet.cif.cif2cif.ElimConsts;
import org.eclipse.escet.cif.cif2cif.ElimIfUpdates;
import org.eclipse.escet.cif.cif2cif.ElimStateEvtExclInvs;
import org.eclipse.escet.cif.cif2cif.ElimStateInvs;
import org.eclipse.escet.cif.cif2cif.ElimTypeDecls;
import org.eclipse.escet.cif.cif2cif.LinearizeProduct;
import org.eclipse.escet.cif.cif2cif.RemoveAnnotations;
import org.eclipse.escet.cif.cif2cif.RemoveIoDecls;
import org.eclipse.escet.cif.cif2cif.SimplifyValues;
import org.eclipse.escet.cif.cif2cif.SwitchesToIfs;
import org.eclipse.escet.cif.common.CifCollectUtils;
import org.eclipse.escet.cif.common.CifEnumUtils;
import org.eclipse.escet.cif.common.CifEvalException;
import org.eclipse.escet.cif.common.CifEvalUtils;
import org.eclipse.escet.cif.common.CifMarkedUtils;
import org.eclipse.escet.cif.common.CifTextUtils;
import org.eclipse.escet.cif.common.CifTypeUtils;
import org.eclipse.escet.cif.common.CifValueUtils;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.automata.Assignment;
import org.eclipse.escet.cif.metamodel.cif.automata.Automaton;
import org.eclipse.escet.cif.metamodel.cif.automata.Edge;
import org.eclipse.escet.cif.metamodel.cif.automata.EdgeEvent;
import org.eclipse.escet.cif.metamodel.cif.automata.Location;
import org.eclipse.escet.cif.metamodel.cif.automata.Update;
import org.eclipse.escet.cif.metamodel.cif.declarations.Declaration;
import org.eclipse.escet.cif.metamodel.cif.declarations.DiscVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.EnumDecl;
import org.eclipse.escet.cif.metamodel.cif.declarations.EnumLiteral;
import org.eclipse.escet.cif.metamodel.cif.declarations.Event;
import org.eclipse.escet.cif.metamodel.cif.declarations.InputVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.VariableValue;
import org.eclipse.escet.cif.metamodel.cif.expressions.BinaryExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.BinaryOperator;
import org.eclipse.escet.cif.metamodel.cif.expressions.BoolExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.CastExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.DiscVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ElifExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.EnumLiteralExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.EventExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.expressions.IfExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.InputVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.IntExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.TauExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.UnaryExpression;
import org.eclipse.escet.cif.metamodel.cif.types.BoolType;
import org.eclipse.escet.cif.metamodel.cif.types.CifType;
import org.eclipse.escet.cif.metamodel.cif.types.EnumType;
import org.eclipse.escet.cif.metamodel.cif.types.IntType;
import org.eclipse.escet.common.box.CodeBox;
import org.eclipse.escet.common.box.MemoryCodeBox;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.Lists;
import org.eclipse.escet.common.java.Sets;
import org.eclipse.escet.common.java.Termination;
import org.eclipse.escet.common.java.output.WarnOutput;
import org.eclipse.escet.common.position.metamodel.position.PositionObject;

/** CIF to mCRL2 transformer. */
public class CifToMcrl2Transformer {
    /** Cooperative termination query function. */
    private final Termination termination;

    /** Callback to send warnings to the user. */
    private final WarnOutput warnOutput;

    /** Per CIF enumeration, its representative. Is {@code null} if not yet initialized. */
    private Map<EnumDecl, EnumDecl> enums;

    /**
     * Per discrete variable of the original CIF specification, its original absolute name. Is {@code null} if not yet
     * initialized.
     */
    private Map<DiscVariable, String> origDiscVarNames;

    /**
     * Per location pointer variable introduced during linearization, its original absolute automaton name. Is
     * {@code null} if not yet initialized.
     */
    private Map<DiscVariable, String> origAutNames;

    /**
     * Constructor for the {@link CifToMcrl2Transformer} class.
     *
     * @param termination Cooperative termination query function.
     * @param warnOutput Callback to send warnings to the user.
     */
    public CifToMcrl2Transformer(Termination termination, WarnOutput warnOutput) {
        this.termination = termination;
        this.warnOutput = warnOutput;
    }

    /**
     * Transform a CIF specification to an mCRL2 model.
     *
     * @param spec The CIF specification. The specification is modified in-place.
     * @param absSpecPath The absolute local file system path to the CIF file.
     * @param valueActionPatterns The 'value' action patterns.
     * @param addMarked Whether to add a 'marked' action.
     * @return The mCRL2 model.
     */
    public CodeBox transform(Specification spec, String absSpecPath, String valueActionPatterns, boolean addMarked) {
        // Perform preprocessing to increase the supported subset.
        preprocess1(spec);

        // Check preconditions.
        CifToMcrl2PreChecker checker = new CifToMcrl2PreChecker(termination);
        checker.reportPreconditionViolations(spec, absSpecPath, "CIF to mCRL2 transformer");

        // Get original discrete variables and their original absolute names.
        List<DiscVariable> origVars = CifCollectUtils.collectDiscVariables(spec, list());
        origDiscVarNames = origVars.stream().collect(Collectors.toMap(v -> v, v -> CifTextUtils.getAbsName(v, false)));

        // Perform additional pre-processing to ease the transformation. Also collect location pointer variables and
        // their original absolute automata names.
        origAutNames = preprocess2(spec);

        // Collect relevant declarations from the specification.
        enums = CifEnumUtils.getEnumDeclReprs(CifCollectUtils.collectEnumDecls(spec, list()));
        List<Declaration> vars = CifCollectUtils.collectDiscAndInputVariables(spec, list());
        List<InputVariable> inputVars = vars.stream().filter(InputVariable.class::isInstance)
                .map(InputVariable.class::cast).toList();
        List<Event> events = CifCollectUtils.collectEvents(spec, list());

        // Collect linearized edges from the specification.
        List<Automaton> linearizedAuts = CifCollectUtils.collectAutomata(spec, list());
        Automaton aut = single(linearizedAuts);
        Location loc = single(aut.getLocations());
        List<Edge> edges = loc.getEdges();

        // Collect variables from linearized specification that require 'value' actions.
        Set<Declaration> valueActionVars = determineValueActionVars(vars, valueActionPatterns);

        // Collect simplified marker predicate, if applicable.
        Expression marked = addMarked ? getMarked(spec) : null;

        // Collect the initialization predicates from the components of the specification, simplify them, and keep only
        // those predicates that add restrictions.
        SimplifyValues simplifyValues = new SimplifyValues();
        List<Expression> initPreds = CifCollectUtils.getComplexComponentsStream(spec)
                .flatMap(comp -> comp.getInitials().stream()).toList();
        initPreds = initPreds.stream().map(pred -> simplifyValues.transform(pred))
                .filter(pred -> !CifValueUtils.isTriviallyTrue(pred, true, true)).toList();

        // Generate the mCRL2 code.
        MemoryCodeBox code = new MemoryCodeBox();
        code.add("% Generated by the CIF to mCRL2 transformer of the Eclipse ESCET toolkit.");
        code.add();
        addSortsForEnums(code);
        addActionsForEvents(events, code);
        addActionsForInputVars(inputVars, code);
        addVarValueActions(valueActionVars, code);
        if (addMarked) {
            addMarkedAction(code);
        }
        addProcess(vars, inputVars, valueActionVars, edges, marked, code);
        addInstantiationForInit(vars, initPreds, code);

        // Return the generated model.
        return code;
    }

    /**
     * Perform preprocessing to increase the supported subset of models.
     *
     * @param spec The CIF specification to preprocess. Is modified in-place.
     */
    private void preprocess1(Specification spec) {
        // Remove/ignore I/O declarations, to increase the supported subset.
        RemoveIoDecls removeIoDecls = new RemoveIoDecls();
        removeIoDecls.transform(spec);
        removeIoDecls.warnAboutIgnoredSvgInputDecsIfRemoved(warnOutput);

        // Perform preprocessing on the specification. The most expensive variant of value simplification is used, to
        // inline (and thus support) constants, and get the most simple result.
        new RemoveAnnotations().transform(spec);
        new ElimComponentDefInst().transform(spec);
        new SimplifyValues().transform(spec);
    }

    /**
     * Perform additional pre-processing to ease the transformation.
     *
     * @param spec The CIF specification to pre-process. Is modified in-place.
     * @return Per location pointer variable introduced during linearization, its original absolute automaton name.
     */
    private Map<DiscVariable, String> preprocess2(Specification spec) {
        // Inline type declarations.
        new ElimTypeDecls().transform(spec);

        // Inline constants and algebraic variables.
        new ElimConsts().transform(spec);
        new ElimAlgVariables().transform(spec);

        // Eliminate state/event exclusion invariants. This introduces new automata, so we do this before linearization.
        new ElimStateEvtExclInvs().transform(spec);

        // Linearize the specification.
        LinearizeProduct linearize = new LinearizeProduct(true);
        linearize.transform(spec);

        // Eliminate state invariants. This doesn't support synchronizing events, so we do it after linearization.
        new ElimStateInvs().transform(spec);

        // Convert 'switch' expressions to 'if' expressions.
        new SwitchesToIfs().transform(spec);

        // Convert 'if' updates to 'if' expressions.
        new ElimIfUpdates().transform(spec);

        // Return location pointer information.
        return linearize.getLpVarToAbsAutNameMap();
    }

    /**
     * Determine the variables of the linearized specification for which 'value' actions are to be generated.
     *
     * @param vars The variables of the linearized CIF specification.
     * @param valueActionPatterns Patterns for selecting variables and automata that get 'value' actions.
     * @return The variables of the linearized CIF specification that get 'value' actions.
     */
    private Set<Declaration> determineValueActionVars(List<Declaration> vars, String valueActionPatterns) {
        // Get name to variable mapping.
        Map<String, Declaration> namesMap = mapc(origDiscVarNames.size() + origAutNames.size());
        for (Entry<DiscVariable, String> entry: origDiscVarNames.entrySet()) {
            Declaration prev = namesMap.put(entry.getValue(), entry.getKey());
            Assert.check(prev == null); // No duplicates.
        }
        for (Entry<DiscVariable, String> entry: origAutNames.entrySet()) {
            Declaration prev = namesMap.put(entry.getValue(), entry.getKey());
            Assert.check(prev == null); // No duplicates.
        }
        for (Declaration var: vars) {
            if (var instanceof InputVariable) {
                Declaration prev = namesMap.put(CifTextUtils.getAbsName(var, false), var);
                Assert.check(prev == null); // No duplicates.
            }
        }

        // Get set of all names to match against.
        Set<String> names = Sets.list2set(Sets.sortedstrings(namesMap.keySet()));

        // Match names against the patterns to determine the names to include.
        Set<String> includeNames = ProcessValueActions.matchNames(valueActionPatterns, names, warnOutput);

        // Get the variables of the linearized specification that should get 'value' actions.
        Set<Declaration> result = includeNames.stream().map(n -> namesMap.get(n)).collect(Sets.toSet());
        Assert.check(vars.containsAll(result)); // Select a subset of the variables from the linearized specification.
        return result;
    }

    /**
     * Get the marker predicate of the specification, and simplify it.
     *
     * @param spec The CIF specification.
     * @return The simplified marker predicate.
     */
    private Expression getMarked(Specification spec) {
        // Get marker predicate of the specification.
        Expression marked = CifMarkedUtils.getMarked(spec);

        // Simplify the marker predicate.
        return new SimplifyValues().transform(marked);
    }

    /**
     * Add mCRL2 sorts for the representative CIF enumerations.
     *
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addSortsForEnums(MemoryCodeBox code) {
        // Skip if no enums.
        if (enums.isEmpty()) {
            return;
        }

        // Generate a sort for each unique representative enumeration.
        code.add("% Sorts for CIF enumerations.");
        for (EnumDecl enumDecl: enums.values().stream().collect(Sets.toSet())) {
            String litNames = enumDecl.getLiterals().stream().map(l -> getName(l)).collect(Collectors.joining(" | "));
            code.add("sort %s = struct %s;", getName(enumDecl), litNames);
        }
        code.add();
    }

    /**
     * Add mCRL2 actions for the CIF events.
     *
     * @param events The CIF events.
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addActionsForEvents(List<Event> events, MemoryCodeBox code) {
        // Skip if no events.
        if (events.isEmpty()) {
            return;
        }

        // Generate an action for each event.
        code.add("% Actions for CIF events.");
        for (Event event: events) {
            code.add("act %s;", getName(event));
        }
        code.add();
    }

    /**
     * Add mCRL2 actions for the input variables.
     *
     * @param inputVars The CIF input variables.
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addActionsForInputVars(List<InputVariable> inputVars, MemoryCodeBox code) {
        // Skip if no input vars.
        if (inputVars.isEmpty()) {
            return;
        }

        // Generate an action for each input variable.
        code.add("% Actions for CIF input variables.");
        for (InputVariable inputVar: inputVars) {
            code.add("act %s'input;", getName(inputVar));
        }
        code.add();
    }

    /**
     * Add mCRL2 actions for the variable 'value' actions.
     *
     * @param vars The CIF variables that should get a variable 'value' action.
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addVarValueActions(Set<Declaration> vars, MemoryCodeBox code) {
        // Skip if no variables.
        if (vars.isEmpty()) {
            return;
        }

        // Generate an action for each variable.
        code.add("% Actions for CIF variables/automata having certain values/locations.");
        for (Declaration var: vars) {
            code.add("act %s'varvalue: %s;", getName(var), generateSortExprForType(getVarType(var)));
        }
        code.add();
    }

    /**
     * Add an mCRL2 'marked' action.
     *
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addMarkedAction(MemoryCodeBox code) {
        code.add("% Action for CIF marker predicate.");
        code.add("act marked;");
        code.add();
    }

    /**
     * Add an mCRL2 process for the CIF variables and edges.
     *
     * @param vars The CIF variables. Includes {@code inputVars}.
     * @param inputVars The CIF input variables.
     * @param valueActVars The CIF variables that should get a variable 'value' action.
     * @param edges The CIF edges.
     * @param marked If non-{@code null}, the marker predicate of the specification for which to add a 'marked' action.
     *     If {@code null}, no such action is added.
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addProcess(List<Declaration> vars, List<InputVariable> inputVars, Set<Declaration> valueActVars,
            List<Edge> edges, Expression marked, MemoryCodeBox code)
    {
        // Generate header.
        code.add("% Process for behavior of the CIF specification.");
        if (vars.isEmpty()) {
            code.add("proc P =");
        } else {
            code.add("proc P(");
            code.indent();
            for (int i = 0; i < vars.size(); i++) {
                Declaration var = vars.get(i);
                boolean isLast = i == vars.size() - 1;
                code.add("%s: %s%s", getName(var), generateSortExprForType(getVarType(var)), isLast ? "" : ",");
            }
            code.dedent();
            code.add(") =");
        }

        // Generate process body.
        code.indent();
        boolean first = true;

        // Add process expressions to body, for edges.
        boolean firstEdge = true;
        for (Edge edge: edges) {
            if (!first) {
                code.add("+");
            }
            first = false;
            if (firstEdge) {
                code.add("% CIF linearized edges.");
                firstEdge = false;
            }
            addProcessExprForEdge(edge, vars, code);
        }

        // Add process expressions to body, for input variable actions.
        boolean firstInputVarAct = true;
        for (InputVariable inputVar: inputVars) {
            if (!first) {
                code.add("+");
            }
            first = false;
            if (firstInputVarAct) {
                code.add("% CIF input variable actions.");
                firstInputVarAct = false;
            }
            addProcessExprForInputVar(inputVar, code);
        }

        // Add process expressions to body, for variable 'value' actions.
        boolean firstValueAct = true;
        for (Declaration valueActVar: valueActVars) {
            if (!first) {
                code.add("+");
            }
            first = false;
            if (firstValueAct) {
                code.add("% CIF variable value actions.");
                firstValueAct = false;
            }
            addProcessExprForVarWithValueAct(valueActVar, code);
        }

        // Add process expression to body, for 'marked' action.
        if (marked != null) {
            if (!first) {
                code.add("+");
            }
            first = false;
            code.add("% CIF 'marked' action.");
            addProcessExprForMarked(marked, code);
        }

        // If the process body is empty, add a 'delta' process expression, to ensure a valid process without behavior.
        if (first) {
            code.add("delta");
        }

        // Done with body.
        code.dedent();
        code.add(";");
    }

    /**
     * Generate an mCRL2 sort expressions for a CIF type.
     *
     * @param type The CIF type.
     * @return The mCRL2 sort expression.
     */
    private String generateSortExprForType(CifType type) {
        if (type instanceof BoolType) {
            return "Bool";
        } else if (type instanceof IntType) {
            return "Int";
        } else if (type instanceof EnumType enumType) {
            EnumDecl representative = enums.get(enumType.getEnum());
            return getName(representative);
        } else {
            throw new AssertionError("Unexpected type: " + type);
        }
    }

    /**
     * Add an mCRL2 process expression for a CIF edge.
     *
     * @param edge The CIF edge.
     * @param vars The CIF variables of the linearized specification.
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addProcessExprForEdge(Edge edge, List<Declaration> vars, MemoryCodeBox code) {
        // Copy guards of the linearized edge.
        List<Expression> guards = list();
        guards.addAll(deepclone(edge.getGuards()));

        // Add additional guards to ensure that assigned integer variables don't go out of range. This prevents runtime
        // errors.
        Map<DiscVariable, Expression> assignments = getAssignments(edge.getUpdates());
        List<DiscVariable> intVars = vars.stream().filter(DiscVariable.class::isInstance).map(DiscVariable.class::cast)
                .filter(v -> v.getType() instanceof IntType).toList();
        for (DiscVariable var: intVars) {
            Expression newValue = assignments.get(var);
            if (newValue != null) {
                int lower = CifTypeUtils.getLowerBound((IntType)var.getType());
                int upper = CifTypeUtils.getUpperBound((IntType)var.getType());
                guards.addAll(createRangeGuards(newValue, lower, upper));
            }
        }

        // Get a single CIF guard predicate.
        Expression guardExpr = CifValueUtils.createConjunction(guards);

        // Simplify the guard predicate.
        guardExpr = new SimplifyValues().transform(guardExpr);

        // Get the mCRL2 guard expression.
        String guard = generateExpr(guardExpr, false);

        // Get event.
        EdgeEvent edgeEvent = single(edge.getEvents());
        Expression eventRef = edgeEvent.getEvent();
        String event;
        if (eventRef instanceof EventExpression eventExpr) {
            event = getName(eventExpr.getEvent());
        } else if (eventRef instanceof TauExpression) {
            event = "tau";
        } else {
            throw new AssertionError("Unexpected event reference: " + eventRef);
        }

        // Get updates.
        String updates = assignments.entrySet().stream().map(
                (Entry<DiscVariable, Expression> e) -> fmt("%s = %s", getName(e.getKey()),
                        generateExpr(e.getValue(), false)))
                .collect(Collectors.joining(", "));

        // Generate process expression.
        code.add("(%s) -> %s . P(%s)", guard, event, updates);
    }

    /**
     * Creates guards to ensure that an integer-typed variable doesn't go out of range.
     *
     * @param value The value of the variable.
     * @param lower The lower bound of the integer type.
     * @param upper The upper bound of the integer type.
     * @return The newly created guard predicates.
     */
    private List<Expression> createRangeGuards(Expression value, int lower, int upper) {
        // Create 'lower <= value'.
        Expression left1 = CifValueUtils.makeInt(lower);
        Expression right1 = deepclone(value);
        BinaryExpression bexpr1 = newBinaryExpression();
        bexpr1.setLeft(left1);
        bexpr1.setOperator(BinaryOperator.LESS_EQUAL);
        bexpr1.setRight(right1);
        bexpr1.setType(newBoolType());

        // Create 'value <= upper'.
        Expression left2 = deepclone(value);
        Expression right2 = CifValueUtils.makeInt(upper);
        BinaryExpression bexpr2 = newBinaryExpression();
        bexpr2.setLeft(left2);
        bexpr2.setOperator(BinaryOperator.LESS_EQUAL);
        bexpr2.setRight(right2);
        bexpr2.setType(newBoolType());

        // Return both.
        return List.of(bexpr1, bexpr2);
    }

    /**
     * Get a mapping from assigned variables to their assigned expressions.
     *
     * @param updates The CIF updates to consider.
     * @return The mapping.
     */
    private Map<DiscVariable, Expression> getAssignments(List<Update> updates) {
        Map<DiscVariable, Expression> result = mapc(updates.size());
        for (Update update: updates) {
            // Get assignment.
            Assert.check(update instanceof Assignment);
            Assignment asgn = (Assignment)update;

            // Get variable and value.
            Assert.check(asgn.getAddressable() instanceof DiscVariableExpression);
            DiscVariable var = ((DiscVariableExpression)asgn.getAddressable()).getVariable();
            Expression value = asgn.getValue();

            // Add to mapping.
            Expression prev = result.put(var, value);
            Assert.check(prev == null);
        }
        return result;
    }

    /**
     * Add an mCRL2 process expression for an input variable action.
     *
     * @param var The CIF input variable for which to add the process expression.
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addProcessExprForInputVar(InputVariable var, MemoryCodeBox code) {
        // Determine conditions. Each condition has parentheses around it.
        List<String> conditions = listc(2);

        if (var.getType() instanceof IntType intType) {
            // Add 'lower <= x' and 'x <= upper' for variable 'x' of type 'int[lower..upper]'. If the integer
            // type range is unbounded, the minimum and maximum values of the 'int' type are used.
            conditions.add(fmt("(%s <= i)", CifTypeUtils.getLowerBound(intType)));
            conditions.add(fmt("(i <= %s)", CifTypeUtils.getUpperBound(intType)));
        }

        conditions.add(fmt("(i != %s)", getName(var)));

        // Combine conditions into a single condition.
        String condition = (conditions.size() == 1) ? single(conditions)
                : fmt("(%s)", conditions.stream().collect(Collectors.joining(" && ")));

        // Add process expression.
        code.add("sum i: %s . %s -> %s'input . P(%s = i)", generateSortExprForType(var.getType()), condition,
                getName(var), getName(var));
    }

    /**
     * Add an mCRL2 process expression for a variable 'value' action.
     *
     * @param var The CIF variable for which to add the process expression.
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addProcessExprForVarWithValueAct(Declaration var, MemoryCodeBox code) {
        code.add("%s'varvalue(%s) . P()", getName(var), getName(var));
    }

    /**
     * Add an mCRL2 process expression for the 'marked' action.
     *
     * @param marked The CIF marker predicate.
     * @param code The mCRL2 code generated so far. Is extended in-place.
     */
    private void addProcessExprForMarked(Expression marked, MemoryCodeBox code) {
        code.add("(%s) -> marked . P()", generateExpr(marked, false));
    }

    /**
     * Generate an mCRL2 data expression for conjunction of some CIF predicates.
     *
     * @param preds The CIF predicates.
     * @param initial Whether the given expression is to be interpreted in the initial process ({@code true}) or within
     *     the regular process ({@code false}).
     * @return The mCRL2 data expression.
     */
    private String generatePreds(List<Expression> preds, boolean initial) {
        if (preds.isEmpty()) {
            return "true";
        } else if (preds.size() == 1) {
            return generateExpr(first(preds), initial);
        } else {
            return preds.stream().map(p -> generateExpr(p, initial)).collect(Collectors.joining(" && ", "(", ")"));
        }
    }

    /**
     * Generate an mCRL2 data expression for a CIF expression.
     *
     * @param expr The CIF expression.
     * @param initial Whether the given expression is to be interpreted in the initial process ({@code true}) or within
     *     the regular process ({@code false}).
     * @return The mCRL2 data expression.
     */
    private String generateExpr(Expression expr, boolean initial) {
        // Check for integer literals and include their value directly. This handles among others the integer number
        // '-2147483648' that is represented in CIF as '-2147483647 - 1'.
        Integer intValue = CifValueUtils.tryGetIntLiteralValue(expr);
        if (intValue != null) {
            return intValue.toString();
        }

        // Handle the different expressions.
        if (expr instanceof BoolExpression boolExpr) {
            return boolExpr.isValue() ? "true" : "false";
        } else if (expr instanceof BinaryExpression binExpr) {
            String left = generateExpr(binExpr.getLeft(), initial);
            String right = generateExpr(binExpr.getRight(), initial);
            String op = switch (binExpr.getOperator()) {
                case ADDITION -> "+";
                case BI_CONDITIONAL -> "==";
                case CONJUNCTION -> "&&";
                case DISJUNCTION -> "||";
                case EQUAL -> "==";
                case GREATER_EQUAL -> ">=";
                case GREATER_THAN -> ">";
                case IMPLICATION -> "=>";
                case INTEGER_DIVISION -> "div";
                case LESS_EQUAL -> "<=";
                case LESS_THAN -> "<";
                case MODULUS -> "mod";
                case MULTIPLICATION -> "*";
                case SUBTRACTION -> "-";
                case UNEQUAL -> "!=";
                default -> throw new AssertionError("Unexpected operator: " + binExpr.getOperator());
            };
            return fmt("(%s %s %s)", left, op, right);
        } else if (expr instanceof UnaryExpression unExpr) {
            String child = generateExpr(unExpr.getChild(), initial);
            String op = switch (unExpr.getOperator()) {
                case INVERSE -> "!";
                case NEGATE -> "-";
                case PLUS -> "";
                default -> throw new AssertionError("Unexpected operator: " + unExpr.getOperator());
            };
            return fmt("%s%s", op, child);
        } else if (expr instanceof DiscVariableExpression discRefExpr) {
            return getName(discRefExpr.getVariable()) + (initial ? "'init" : "");
        } else if (expr instanceof InputVariableExpression inputRefExpr) {
            return getName(inputRefExpr.getVariable()) + (initial ? "'init" : "");
        } else if (expr instanceof IntExpression intLitExpr) {
            return Integer.toString(intLitExpr.getValue());
        } else if (expr instanceof EnumLiteralExpression enumLitRefExpr) {
            EnumLiteral refEnumLit = enumLitRefExpr.getLiteral();
            EnumDecl refEnumDecl = (EnumDecl)refEnumLit.eContainer();
            int litIdx = refEnumDecl.getLiterals().indexOf(refEnumLit);
            EnumDecl representativeEnumDecl = enums.get(refEnumDecl);
            EnumLiteral representativeEnumLit = representativeEnumDecl.getLiterals().get(litIdx);
            return getName(representativeEnumLit);
        } else if (expr instanceof IfExpression ifExpr) {
            String result = generateExpr(ifExpr.getElse(), initial);
            for (ElifExpression elifExpr: Lists.reverse(ifExpr.getElifs())) {
                String elifGuard = generatePreds(elifExpr.getGuards(), initial);
                String elifThen = generateExpr(elifExpr.getThen(), initial);
                result = fmt("if(%s, %s, %s)", elifGuard, elifThen, result);
            }
            String ifGuard = generatePreds(ifExpr.getGuards(), initial);
            String ifThen = generateExpr(ifExpr.getThen(), initial);
            result = fmt("if(%s, %s, %s)", ifGuard, ifThen, result);
            return result;
        } else if (expr instanceof CastExpression castExpr) {
            return generateExpr(castExpr.getChild(), initial);
        }
        throw new AssertionError("Unexpected expression: " + expr);
    }

    /**
     * Statically evaluates the given CIF expression and generates an mCRL2 data expression for it.
     *
     * @param expr The CIF expression. Must be statically evaluable.
     * @param initial Whether the given expression is to be interpreted in the initial process ({@code true}) or within
     *     the regular process ({@code false}).
     * @return The mCRL2 data expression.
     */
    private String evalAndGenerateExpr(Expression expr, boolean initial) {
        // Evaluate statically. For negative integer values, this may not be a literal, but still an expression.
        Expression value;
        try {
            value = CifEvalUtils.evalAsExpr(expr, true);
        } catch (CifEvalException e) {
            throw new AssertionError("Precondition violation.", e);
        }

        // Generate an mCRL2 data expression for it.
        return generateExpr(value, initial);
    }

    /**
     * Add an mCRL2 instantiation for the initial values of the variables, taking into account initialization
     * predicates.
     *
     * @param vars The variables.
     * @param initPreds The initialization predicates.
     * @param code The code generated so far. Is extended in-place.
     */
    private void addInstantiationForInit(List<Declaration> vars, List<Expression> initPreds, MemoryCodeBox code) {
        // Handle case of not having any variables and no initialization predicates.
        if (vars.isEmpty() && initPreds.isEmpty()) {
            code.add();
            code.add("% Initialization.");
            code.add("init P();");
            return;
        }

        // Per variable, get its possible initial values. In case of 'any value in its domain' and multiple values in
        // the domain, the value is 'null'.
        List<List<String>> initsPerVar = listc(vars.size());
        boolean hasSingleInitialState = true;
        for (Declaration var: vars) {
            // Get initial values of the variable.
            if (var instanceof DiscVariable dvar) {
                VariableValue varValue = dvar.getValue();
                if (varValue == null) {
                    // Default initial value.
                    initsPerVar.add(
                            List.of(evalAndGenerateExpr(CifValueUtils.getDefaultValue(dvar.getType(), null), true)));
                } else if (varValue.getValues().isEmpty()) {
                    // Any value in its domain.
                    if (CifValueUtils.hasSingleValue(dvar.getType())) {
                        // Single value in the domain.
                        List<Expression> values = CifValueUtils.getPossibleValues(dvar.getType());
                        initsPerVar.add(List.of(evalAndGenerateExpr(single(values), true)));
                    } else {
                        // Multiple values in the domain.
                        initsPerVar.add(null);
                        hasSingleInitialState = false;
                    }
                } else {
                    // One or more explicit initial values.
                    initsPerVar.add(
                            varValue.getValues().stream().map(expr -> evalAndGenerateExpr(expr, true)).toList());
                    hasSingleInitialState &= varValue.getValues().size() == 1;
                }
            } else if (var instanceof InputVariable ivar) {
                // Any value in its domain.
                if (CifValueUtils.hasSingleValue(ivar.getType())) {
                    // Single value in the domain.
                    List<Expression> values = CifValueUtils.getPossibleValues(ivar.getType());
                    initsPerVar.add(List.of(evalAndGenerateExpr(single(values), true)));
                } else {
                    // Multiple values in the domain.
                    initsPerVar.add(null);
                    hasSingleInitialState = false;
                }
            } else {
                throw new AssertionError("Unexpected variable: " + var);
            }
        }

        // Optimization for a single initial state and no initialization predicates.
        if (hasSingleInitialState && initPreds.isEmpty()) {
            code.add();
            code.add("% Initialization.");
            code.add("init P(%s);", initsPerVar.stream().map(inits -> single(inits)).collect(Collectors.joining(", ")));
            return;
        }

        // Define an explicit initialization action.
        code.add();
        code.add("% Action for initialization, due to having multiple initial states.");
        code.add("act initialize;");

        // Add the start of the initial mCRL2 process.
        code.add();
        code.add("% Initialization.");
        code.add("init");
        code.indent();

        // Add per variable a summation over its type, to allow the variable to be initialized to any value supported by
        // its type. Add extra conditions if needed, to ensure the variable stays within its range, and can only be
        // initialized to its possible initial values. Where possible, we leave out summations for variables for which
        // we don't need them.
        List<String> args = listc(vars.size());
        for (int varIdx = 0; varIdx < vars.size(); varIdx++) {
            Declaration var = vars.get(varIdx);
            List<String> initsOfVar = initsPerVar.get(varIdx);
            if (initsOfVar != null && initsOfVar.size() == 1 && initPreds.isEmpty()) {
                // Use the single initial value as argument for the process.
                args.add(initsOfVar.get(0));
            } else {
                // Use a summation over all the possible initial values, providing the summation variable as argument
                // for the process.
                String sumVarName = getName(var) + "'init";
                args.add(sumVarName);

                // In principle, the summation is over the type. In certain cases, the summation is guarded with extra
                // conditions to ensure that we can only start with initial values. Each condition has parentheses
                // around it.
                List<String> conditions = list();

                if (getVarType(var) instanceof IntType intType) {
                    // Add 'lower <= x' and 'x <= upper' for variable 'x' of type 'int[lower..upper]'. If the integer
                    // type range is unbounded, the minimum and maximum values of the 'int' type are used.
                    conditions.add(fmt("(%s <= %s)", CifTypeUtils.getLowerBound(intType), sumVarName));
                    conditions.add(fmt("(%s <= %s)", sumVarName, CifTypeUtils.getUpperBound(intType)));
                }

                if (initsOfVar != null) {
                    // Add '((x == value_1) || (x == value_2) || ... || (x == value_n))' for variable 'x' with 'n'
                    // values.
                    conditions.add(initsOfVar.stream().map(v -> fmt("(%s == %s)", sumVarName, v))
                            .collect(Collectors.joining(" || ", "(", ")")));
                }

                // Combine conditions into a single condition.
                String condition = switch (conditions.size()) {
                    case 0 -> "";
                    case 1 -> single(conditions) + " ->";
                    default -> fmt("(%s) ->", conditions.stream().collect(Collectors.joining(" && ")));
                };

                // Add the summation.
                code.add("sum %s: %s . %s", sumVarName, generateSortExprForType(getVarType(var)), condition);
            }
        }

        // Add extra conditions for the initialization predicates.
        for (Expression initPred: initPreds) {
            code.add("(%s) ->", generateExpr(initPred, true));
        }

        // Add the remainder of the initial mCRL2 process.
        code.add("initialize . P(%s);", String.join(", ", args));
        code.dedent();
    }

    /**
     * Returns the type of the given variable.
     *
     * @param var The discrete or input variable.
     * @return The variable's type.
     */
    private static CifType getVarType(Declaration var) {
        return switch (var) {
            case DiscVariable dvar -> dvar.getType();
            case InputVariable ivar -> ivar.getType();
            default -> throw new AssertionError("Unexpected variable: " + var);
        };
    }

    /**
     * Get the mCRL2 name for a CIF object.
     *
     * @param cifObject The CIF object.
     * @return The mCRL2 name.
     */
    private String getName(PositionObject cifObject) {
        // Get absolute names. Since discrete variables are moved to the single linearized automaton during
        // linearization, use their original variable/automaton names.
        String name;
        if (cifObject instanceof DiscVariable dvar) {
            if (origDiscVarNames.containsKey(dvar)) {
                name = origDiscVarNames.get(dvar);
            } else if (origAutNames.containsKey(dvar)) {
                name = origAutNames.get(dvar);
            } else {
                throw new AssertionError("Unexpected discrete variable: " + dvar);
            }
        } else {
            name = CifTextUtils.getAbsName(cifObject, false);
        }

        // Make an mCRL2 name. Replace dots by apostrophes. Add one at the end to avoid clashes with mCRL2 keywords.
        name = name.replace('.', '\'');
        name += "'";
        return name;
    }
}
