//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2024, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.cif2cif;

import static org.eclipse.escet.common.java.Lists.copy;
import static org.eclipse.escet.common.java.Maps.map;
import static org.eclipse.escet.common.java.Sets.set;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.escet.cif.common.CifLocationUtils;
import org.eclipse.escet.cif.common.CifTypeUtils;
import org.eclipse.escet.cif.metamodel.cif.ComplexComponent;
import org.eclipse.escet.cif.metamodel.cif.ComponentDef;
import org.eclipse.escet.cif.metamodel.cif.ComponentInst;
import org.eclipse.escet.cif.metamodel.cif.ComponentParameter;
import org.eclipse.escet.cif.metamodel.cif.Group;
import org.eclipse.escet.cif.metamodel.cif.Parameter;
import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.cif.annotations.AnnotatedObject;
import org.eclipse.escet.cif.metamodel.cif.automata.Automaton;
import org.eclipse.escet.cif.metamodel.cif.automata.Location;
import org.eclipse.escet.cif.metamodel.cif.declarations.AlgVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.ContVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.DiscVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.InputVariable;
import org.eclipse.escet.cif.metamodel.cif.functions.Function;
import org.eclipse.escet.cif.metamodel.cif.types.CifType;
import org.eclipse.escet.cif.metamodel.cif.types.ComponentDefType;
import org.eclipse.escet.cif.metamodel.java.CifConstructors;
import org.eclipse.escet.cif.metamodel.java.CifWalker;
import org.eclipse.escet.common.emf.EMFHelper;
import org.eclipse.escet.common.java.Assert;

/**
 * In-place transformation that converts the specification to its interface (also known as skeleton).
 *
 * <p>
 * The interface contains the minimal elements to allow another specification to refer to its constants, types, events,
 * functions and state. The interface can then for instance be imported in other specifications, where it is undesired
 * to import the full original specification.
 * </p>
 *
 * <p>
 * The interface generated by the transformation has the same component structure as the original specification, but
 * elements that can't be referred are removed, and other elements are replaced (such as algebraic variables that are
 * replaced by input variables).
 * </p>
 *
 * @see ConvertToInterfaceComplete
 * @see ConvertToInterfaceReduced
 */
public class ConvertToInterface extends CifWalker implements CifToCifTransformation {
    /** The kind of interface to generate. */
    private final InterfaceKind interfaceKind;

    /** The input variables added to the specification during the transformation. */
    private final Set<InputVariable> addedInputVars = set();

    /** Per component definition, its original parameters, if parameters have been removed already. */
    private final Map<ComponentDef, List<Parameter>> origCompDefParams = map();

    /**
     * Constructor for the {@link ConvertToInterface} class.
     *
     * @param interfaceKind The kind of interface to generate.
     */
    public ConvertToInterface(InterfaceKind interfaceKind) {
        this.interfaceKind = interfaceKind;
    }

    @Override
    public void transform(Specification spec) {
        walkSpecification(spec);
    }

    @Override
    protected void preprocessAnnotatedObject(AnnotatedObject annotatedObj) {
        // Remove all annotations, as you can't refer to them. Merging the original model with the interface will
        // restore the annotations.
        annotatedObj.getAnnotations().clear();
    }

    @Override
    protected void preprocessComplexComponent(ComplexComponent comp) {
        // Remove things you can't refer to.
        comp.getIoDecls().clear();
        comp.getInitials().clear();
        comp.getMarkeds().clear();
        comp.getInvariants().clear();
        comp.getEquations().clear();

        // Remove functions and original input variables of the specification, if requested. Never remove input
        // variables added during this transformation.
        if (interfaceKind == InterfaceKind.REDUCED) {
            comp.getDeclarations().removeIf(d -> d instanceof Function);
            comp.getDeclarations().removeIf(d -> d instanceof InputVariable i && !addedInputVars.contains(i));
        }
    }

    @Override
    protected void preprocessComponentDef(ComponentDef compDef) {
        // Store original parameters, for later use.
        origCompDefParams.put(compDef, copy(compDef.getParameters()));

        // Remove some of the parameters, as you can't refer to them from the outside anyway. Preserve component
        // parameters, as we may be referring to types via them.
        compDef.getParameters().removeIf(p -> !(p instanceof ComponentParameter));
    }

    @Override
    protected void preprocessComponentInst(ComponentInst compInst) {
        // Remove the arguments for parameters that have been removed. This prevents referring to variables etc in the
        // arguments, while these variables etc may have been removed or replaced.

        // Get component definition.
        CifType compDefType = CifTypeUtils.normalizeType(compInst.getDefinition());
        Assert.check(compDefType instanceof ComponentDefType);
        ComponentDef compDef = ((ComponentDefType)compDefType).getDefinition();

        // Get (original) parameters of the component definition.
        List<Parameter> params = origCompDefParams.get(compDef);
        if (params == null) {
            params = compDef.getParameters();
        }

        // Remove arguments for non-component parameters.
        for (int i = params.size() - 1; i >= 0; i--) {
            if (params.get(i) instanceof ComponentParameter) {
                continue;
            }
            compInst.getArguments().remove(i);
        }
    }

    @Override
    protected void postprocessAlgVariable(AlgVariable algVar) {
        // Replace algebraic variable by an input variable. We do this as post-processing for the algebraic variable,
        // as it loses its type.
        InputVariable inputVar = CifConstructors.newInputVariable();
        inputVar.setName(algVar.getName());
        inputVar.setType(algVar.getType());

        EMFHelper.updateParentContainment(algVar, inputVar);

        addedInputVars.add(inputVar);
    }

    @Override
    protected void postprocessContVariable(ContVariable contVar) {
        // Replace continuous variable by an input variable. We do this as post-processing for the continuous variable,
        // for consistency with the other kinds of variables.
        InputVariable inputVar = CifConstructors.newInputVariable();
        inputVar.setName(contVar.getName());
        inputVar.setType(CifConstructors.newRealType());

        EMFHelper.updateParentContainment(contVar, inputVar);

        addedInputVars.add(inputVar);
    }

    @Override
    protected void postprocessDiscVariable(DiscVariable discVar) {
        // Replace discrete variable by an input variable. Don't replace function parameters/variables. We do this as
        // post-processing for the discrete variable, as it loses its type.
        if (discVar.eContainer() instanceof ComplexComponent) {
            InputVariable inputVar = CifConstructors.newInputVariable();
            inputVar.setName(discVar.getName());
            inputVar.setType(discVar.getType());

            EMFHelper.updateParentContainment(discVar, inputVar);

            addedInputVars.add(inputVar);
        }
    }

    @Override
    protected void preprocessLocation(Location loc) {
        // Create input variable in the automaton for the location, if it has a name. The location itself gets removed
        // when converting the automaton to a group.
        if (loc.getName() != null) {
            InputVariable inputVar = CifConstructors.newInputVariable();
            inputVar.setName(loc.getName());
            inputVar.setType(CifConstructors.newBoolType());

            Automaton aut = CifLocationUtils.getAutomaton(loc);
            aut.getDeclarations().add(inputVar);

            addedInputVars.add(inputVar);
        }
    }

    @Override
    protected void postprocessAutomaton(Automaton aut) {
        // Convert automaton to a group. We do this as post-processing for the automaton, to ensure that all its
        // locations have already been converted.
        Group group = CifConstructors.newGroup();
        group.setName(aut.getName());
        group.getDeclarations().addAll(aut.getDeclarations());

        EMFHelper.updateParentContainment(aut, group);
    }

    /** Kind of interface to generate. */
    public static enum InterfaceKind {
        /** Complete interface. */
        COMPLETE,

        /** Reduced interface, where the input variables (of the input specification) and functions are removed. */
        REDUCED;
    }
}
