/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.kura.core.keystore;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.UnrecoverableEntryException;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.eclipse.kura.KuraErrorCode;
import org.eclipse.kura.KuraException;
import org.eclipse.kura.KuraRuntimeException;
import org.eclipse.kura.configuration.ConfigurationService;
import org.eclipse.kura.configuration.Password;
import org.eclipse.kura.core.keystore.BaseKeystoreService;
import org.eclipse.kura.core.keystore.FilesystemKeystoreServiceOptions;
import org.eclipse.kura.core.keystore.KeystoreInstance;
import org.eclipse.kura.crypto.CryptoService;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FilesystemKeystoreServiceImpl
extends BaseKeystoreService {
    private static final Logger logger = LoggerFactory.getLogger(FilesystemKeystoreServiceImpl.class);
    private CryptoService cryptoService;
    private ConfigurationService configurationService;
    private FilesystemKeystoreServiceOptions keystoreServiceOptions;
    private ScheduledExecutorService selfUpdaterExecutor;
    private ScheduledFuture<?> selfUpdaterFuture;

    public void setCryptoService(CryptoService cryptoService) {
        this.cryptoService = cryptoService;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    @Override
    public void activate(ComponentContext context, Map<String, Object> properties) {
        logger.info("Bundle {} is starting!", properties.get("kura.service.pid"));
        this.componentContext = context;
        this.keystoreServiceOptions = new FilesystemKeystoreServiceOptions(properties, this.cryptoService);
        this.selfUpdaterExecutor = Executors.newSingleThreadScheduledExecutor();
        if (!this.keystoreExists(this.keystoreServiceOptions.getKeystorePath())) {
            try {
                this.createKeystore(this.keystoreServiceOptions);
            }
            catch (Exception e) {
                logger.error("Keystore file creation failed", (Throwable)e);
            }
        }
        if (this.keystoreServiceOptions.needsRandomPassword()) {
            this.setRandomPassword();
        }
        super.activate(context, properties);
        logger.info("Bundle {} has started!", properties.get("kura.service.pid"));
    }

    @Override
    public void updated(Map<String, Object> properties) {
        logger.info("Bundle {} is updating!", properties.get("kura.service.pid"));
        FilesystemKeystoreServiceOptions newOptions = new FilesystemKeystoreServiceOptions(properties, this.cryptoService);
        if (!this.keystoreServiceOptions.equals(newOptions)) {
            logger.info("Perform update...");
            if (!this.keystoreServiceOptions.getKeystorePath().equals(newOptions.getKeystorePath())) {
                this.updateKeystorePath(newOptions);
            } else {
                this.checkAndUpdateKeystorePassword(newOptions);
            }
            this.keystoreServiceOptions = new FilesystemKeystoreServiceOptions(properties, this.cryptoService);
        }
        super.updated(properties);
        logger.info("Bundle {} has updated!", properties.get("kura.service.pid"));
    }

    @Override
    public void deactivate() {
        logger.info("Bundle {} is deactivating!", this.keystoreServiceOptions.getProperties().get("kura.service.pid"));
        if (this.selfUpdaterFuture != null && !this.selfUpdaterFuture.isDone()) {
            logger.info("Self updater task running. Stopping it");
            this.selfUpdaterFuture.cancel(true);
        }
        super.deactivate();
    }

    @Override
    protected void saveKeystore(KeystoreInstance ks) throws IOException, KeyStoreException, NoSuchAlgorithmException, CertificateException {
        Throwable throwable = null;
        Object var3_4 = null;
        try (FileOutputStream tsOutStream = new FileOutputStream(this.keystoreServiceOptions.getKeystorePath());){
            ks.getKeystore().store(tsOutStream, ks.getPassword());
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    @Override
    protected KeystoreInstance loadKeystore() throws KuraException {
        return this.loadKeystore(this.keystoreServiceOptions);
    }

    @Override
    protected String getCrlStorePath() {
        return String.valueOf(this.keystoreServiceOptions.getKeystorePath()) + ".crl";
    }

    private void checkAndUpdateKeystorePassword(FilesystemKeystoreServiceOptions options) {
        try {
            KeystoreInstance ks = this.loadKeystore(this.keystoreServiceOptions);
            char[] configPassword = options.getKeystorePassword(this.cryptoService);
            if (!Arrays.equals(ks.getPassword(), configPassword)) {
                this.setKeystorePassword(ks, configPassword);
            }
        }
        catch (Exception e) {
            logger.warn("failed to load or update keystore password", (Throwable)e);
        }
    }

    private boolean keystoreExists(String keystorePath) {
        return keystorePath != null && new File(keystorePath).isFile();
    }

    private void createKeystore(FilesystemKeystoreServiceOptions options) throws Exception {
        String keystorePath = options.getKeystorePath();
        char[] passwordChar = options.getKeystorePassword(this.cryptoService);
        if (keystorePath == null) {
            return;
        }
        File fKeyStore = new File(keystorePath);
        if (!fKeyStore.createNewFile()) {
            logger.error("Keystore file already exists at location {}", (Object)keystorePath);
            throw new KuraException(KuraErrorCode.CONFIGURATION_ATTRIBUTE_INVALID, new Object[]{"keystore.path", keystorePath, "file already exists"});
        }
        try {
            Throwable throwable = null;
            Object var6_8 = null;
            try (FileOutputStream os = new FileOutputStream(fKeyStore);){
                KeyStore newKeystore = KeyStore.getInstance(KeyStore.getDefaultType());
                newKeystore.load(null, passwordChar);
                newKeystore.store(os, passwordChar);
                os.flush();
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (Exception e) {
            logger.error("Unable to load and store the keystore", (Throwable)e);
            throw e;
        }
        this.setKeystorePassword(this.loadKeystore(options), passwordChar);
    }

    private void updateKeystorePath(FilesystemKeystoreServiceOptions newOptions) {
        if (!this.keystoreExists(newOptions.getKeystorePath())) {
            try {
                this.createKeystore(newOptions);
            }
            catch (Exception e) {
                logger.error("Keystore file creation failed", (Throwable)e);
            }
        }
        try {
            this.loadKeystore(newOptions);
        }
        catch (Exception exception) {
            logger.warn("Keystore {} not accessible!", (Object)newOptions.getKeystorePath());
        }
    }

    private void setRandomPassword() {
        try {
            KeystoreInstance keystore = this.loadKeystore(this.keystoreServiceOptions);
            char[] newPassword = new BigInteger(160, new SecureRandom()).toString(32).toCharArray();
            this.setKeystorePassword(keystore, newPassword);
            HashMap<String, Object> props = new HashMap<String, Object>(this.keystoreServiceOptions.getProperties());
            props.put("keystore.password", new String(this.cryptoService.encryptAes(newPassword)));
            this.keystoreServiceOptions = new FilesystemKeystoreServiceOptions(props, this.cryptoService);
            this.updatePasswordInConfigService(newPassword);
        }
        catch (Exception e) {
            logger.warn("Keystore password change failed", (Throwable)e);
        }
    }

    private synchronized void saveKeystore(KeystoreInstance ks, char[] keyStorePassword) throws KeyStoreException, IOException, NoSuchAlgorithmException, CertificateException {
        Throwable throwable = null;
        Object var4_5 = null;
        try (FileOutputStream tsOutStream = new FileOutputStream(((KeystoreInstanceImpl)ks).path);){
            ks.getKeystore().store(tsOutStream, keyStorePassword);
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    private void updatePasswordInConfigService(char[] newPassword) {
        String pid = this.keystoreServiceOptions.getPid();
        HashMap<String, Object> props = new HashMap<String, Object>();
        props.putAll(this.keystoreServiceOptions.getProperties());
        props.put("keystore.path", this.keystoreServiceOptions.getKeystorePath());
        props.put("keystore.password", new Password(newPassword));
        props.put("randomize.password", false);
        this.selfUpdaterFuture = this.selfUpdaterExecutor.scheduleAtFixedRate(() -> {
            try {
                if (this.componentContext.getServiceReference() != null && this.configurationService.getComponentConfiguration(pid) != null && this.configurationService.getComponentConfiguration(pid).getDefinition() != null) {
                    this.configurationService.updateConfiguration(pid, props);
                    throw new KuraRuntimeException(KuraErrorCode.CONFIGURATION_SNAPSHOT_TAKING, new Object[]{"Updated. The task will be terminated."});
                }
                logger.info("No service or configuration available yet.");
            }
            catch (KuraException e) {
                logger.warn("Cannot get/update configuration for pid: {}", (Object)pid, (Object)e);
            }
        }, 1000L, 1000L, TimeUnit.MILLISECONDS);
    }

    private synchronized void setKeystorePassword(KeystoreInstance ks, char[] password) {
        try {
            FilesystemKeystoreServiceImpl.updateKeyEntriesPasswords(ks, password);
            this.saveKeystore(ks, password);
            this.cryptoService.setKeyStorePassword(((KeystoreInstanceImpl)ks).path, password);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException | CertificateException exception) {
            logger.warn("Failed to change keystore password");
        }
        catch (KuraException kuraException) {
            logger.warn("Failed to persist keystore password");
        }
    }

    private static void updateKeyEntriesPasswords(KeystoreInstance ks, char[] newPassword) throws KeyStoreException, NoSuchAlgorithmException, UnrecoverableEntryException {
        Enumeration<String> aliases = ks.getKeystore().aliases();
        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            if (!ks.getKeystore().isKeyEntry(alias)) continue;
            KeyStore.PasswordProtection oldPP = new KeyStore.PasswordProtection(ks.getPassword());
            KeyStore.Entry entry = ks.getKeystore().getEntry(alias, oldPP);
            KeyStore.PasswordProtection newPP = new KeyStore.PasswordProtection(newPassword);
            ks.getKeystore().setEntry(alias, entry, newPP);
        }
    }

    private synchronized KeystoreInstance loadKeystore(FilesystemKeystoreServiceOptions options) throws KuraException {
        KeystoreInstance result;
        ArrayList<char[]> passwords = new ArrayList<char[]>(2);
        passwords.add(options.getKeystorePassword(this.cryptoService));
        char[] passwordInCrypto = null;
        try {
            passwordInCrypto = this.cryptoService.getKeyStorePassword(options.getKeystorePath());
            if (passwordInCrypto != null) {
                passwords.add(passwordInCrypto);
            }
        }
        catch (Exception e) {
            logger.debug("failed to retrieve password", (Throwable)e);
        }
        if (!Arrays.equals(passwordInCrypto, (result = new KeystoreLoader(options.getKeystorePath(), passwords).loadKeystore()).getPassword())) {
            this.cryptoService.setKeyStorePassword(((KeystoreInstanceImpl)result).path, result.getPassword());
        }
        return result;
    }

    private static class KeystoreInstanceImpl
    implements KeystoreInstance {
        private final KeyStore keystore;
        private final char[] password;
        private final String path;

        public KeystoreInstanceImpl(KeyStore keystore, char[] password, String path) {
            this.keystore = keystore;
            this.password = password;
            this.path = path;
        }

        @Override
        public KeyStore getKeystore() {
            return this.keystore;
        }

        @Override
        public char[] getPassword() {
            return this.password;
        }
    }

    private static class KeystoreLoader {
        private final String path;
        private final List<char[]> passwords;

        KeystoreLoader(String path, List<char[]> passwords) {
            this.path = path;
            this.passwords = passwords;
        }

        private KeyStore loadKeystore(String path, char[] password) throws NoSuchAlgorithmException, CertificateException, IOException, KeyStoreException {
            KeyStore ks = null;
            Throwable throwable = null;
            Object var5_6 = null;
            try (FileInputStream tsReadStream = new FileInputStream(path);){
                ks = KeyStore.getInstance(KeyStore.getDefaultType());
                char[] keystorePassword = password;
                ks.load(tsReadStream, keystorePassword);
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            return ks;
        }

        KeystoreInstance loadKeystore() throws KuraException {
            for (char[] password : this.passwords) {
                try {
                    KeyStore keyStore = this.loadKeystore(this.path, password);
                    return new KeystoreInstanceImpl(keyStore, password, this.path);
                }
                catch (Exception e) {
                    logger.debug("failed to load keystore", (Throwable)e);
                }
            }
            throw new KuraException(KuraErrorCode.BAD_REQUEST, new Object[]{"Failed to get the KeyStore"});
        }
    }
}

