"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
const ee = require("events");
const messages_1 = require("./messages");
class Disposable0 {
    dispose() {
    }
}
class Emitter {
    get event() {
        if (!this._event) {
            this._event = (listener, thisArg) => {
                this._listener = listener;
                this._this = thisArg;
                let result;
                result = {
                    dispose: () => {
                        this._listener = undefined;
                        this._this = undefined;
                    }
                };
                return result;
            };
        }
        return this._event;
    }
    fire(event) {
        if (this._listener) {
            try {
                this._listener.call(this._this, event);
            }
            catch (e) {
            }
        }
    }
    hasListener() {
        return !!this._listener;
    }
    dispose() {
        this._listener = undefined;
        this._this = undefined;
    }
}
class ProtocolServer extends ee.EventEmitter {
    constructor() {
        super();
        this._sendMessage = new Emitter();
        this._pendingRequests = new Map();
        this.onDidSendMessage = this._sendMessage.event;
    }
    // ---- implements vscode.Debugadapter interface ---------------------------
    dispose() {
    }
    handleMessage(msg) {
        if (msg.type === 'request') {
            this.dispatchRequest(msg);
        }
        else if (msg.type === 'response') {
            const response = msg;
            const clb = this._pendingRequests.get(response.request_seq);
            if (clb) {
                this._pendingRequests.delete(response.request_seq);
                clb(response);
            }
        }
    }
    _isRunningInline() {
        return this._sendMessage && this._sendMessage.hasListener();
    }
    //--------------------------------------------------------------------------
    start(inStream, outStream) {
        this._sequence = 1;
        this._writableStream = outStream;
        this._rawData = new Buffer(0);
        inStream.on('data', (data) => this._handleData(data));
        inStream.on('close', () => {
            this._emitEvent(new messages_1.Event('close'));
        });
        inStream.on('error', (error) => {
            this._emitEvent(new messages_1.Event('error', 'inStream error: ' + (error && error.message)));
        });
        outStream.on('error', (error) => {
            this._emitEvent(new messages_1.Event('error', 'outStream error: ' + (error && error.message)));
        });
        inStream.resume();
    }
    stop() {
        if (this._writableStream) {
            this._writableStream.end();
        }
    }
    sendEvent(event) {
        this._send('event', event);
    }
    sendResponse(response) {
        if (response.seq > 0) {
            console.error(`attempt to send more than one response for command ${response.command}`);
        }
        else {
            this._send('response', response);
        }
    }
    sendRequest(command, args, timeout, cb) {
        const request = {
            command: command
        };
        if (args && Object.keys(args).length > 0) {
            request.arguments = args;
        }
        if (!this._writableStream) {
            this._emitEvent(new messages_1.Event('error', 'sendRequest: No writableStream'));
            return;
        }
        this._send('request', request);
        if (cb) {
            this._pendingRequests.set(request.seq, cb);
            const timer = setTimeout(() => {
                clearTimeout(timer);
                const clb = this._pendingRequests.get(request.seq);
                if (clb) {
                    this._pendingRequests.delete(request.seq);
                    clb(new messages_1.Response(request, 'timeout'));
                }
            }, timeout);
        }
    }
    // ---- protected ----------------------------------------------------------
    dispatchRequest(request) {
    }
    // ---- private ------------------------------------------------------------
    _emitEvent(event) {
        this.emit(event.event, event);
    }
    _send(typ, message) {
        message.type = typ;
        message.seq = this._sequence++;
        if (this._writableStream) {
            const json = JSON.stringify(message);
            this._writableStream.write(`Content-Length: ${Buffer.byteLength(json, 'utf8')}\r\n\r\n${json}`, 'utf8');
        }
        this._sendMessage.fire(message);
    }
    _handleData(data) {
        this._rawData = Buffer.concat([this._rawData, data]);
        while (true) {
            if (this._contentLength >= 0) {
                if (this._rawData.length >= this._contentLength) {
                    const message = this._rawData.toString('utf8', 0, this._contentLength);
                    this._rawData = this._rawData.slice(this._contentLength);
                    this._contentLength = -1;
                    if (message.length > 0) {
                        try {
                            let msg = JSON.parse(message);
                            this.handleMessage(msg);
                        }
                        catch (e) {
                            this._emitEvent(new messages_1.Event('error', 'Error handling data: ' + (e && e.message)));
                        }
                    }
                    continue; // there may be more complete messages to process
                }
            }
            else {
                const idx = this._rawData.indexOf(ProtocolServer.TWO_CRLF);
                if (idx !== -1) {
                    const header = this._rawData.toString('utf8', 0, idx);
                    const lines = header.split('\r\n');
                    for (let i = 0; i < lines.length; i++) {
                        const pair = lines[i].split(/: +/);
                        if (pair[0] == 'Content-Length') {
                            this._contentLength = +pair[1];
                        }
                    }
                    this._rawData = this._rawData.slice(idx + ProtocolServer.TWO_CRLF.length);
                    continue;
                }
            }
            break;
        }
    }
}
ProtocolServer.TWO_CRLF = '\r\n\r\n';
exports.ProtocolServer = ProtocolServer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvdG9jb2wuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvcHJvdG9jb2wudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Z0dBR2dHOztBQUVoRyw2QkFBNkI7QUFFN0IseUNBQTZDO0FBUzdDLE1BQU0sV0FBVztJQUNoQixPQUFPO0lBQ1AsQ0FBQztDQUNEO0FBTUQsTUFBTSxPQUFPO0lBTVosSUFBSSxLQUFLO1FBQ1IsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDakIsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLFFBQXVCLEVBQUUsT0FBYSxFQUFFLEVBQUU7Z0JBRXhELElBQUksQ0FBQyxTQUFTLEdBQUcsUUFBUSxDQUFDO2dCQUMxQixJQUFJLENBQUMsS0FBSyxHQUFHLE9BQU8sQ0FBQztnQkFFckIsSUFBSSxNQUFtQixDQUFDO2dCQUN4QixNQUFNLEdBQUc7b0JBQ1IsT0FBTyxFQUFFLEdBQUcsRUFBRTt3QkFDYixJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQzt3QkFDM0IsSUFBSSxDQUFDLEtBQUssR0FBRyxTQUFTLENBQUM7b0JBQ3hCLENBQUM7aUJBQ0QsQ0FBQztnQkFDRixPQUFPLE1BQU0sQ0FBQztZQUNmLENBQUMsQ0FBQztTQUNGO1FBQ0QsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDO0lBQ3BCLENBQUM7SUFFRCxJQUFJLENBQUMsS0FBUTtRQUNaLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNuQixJQUFJO2dCQUNILElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7YUFDdkM7WUFBQyxPQUFPLENBQUMsRUFBRTthQUNYO1NBQ0Q7SUFDRixDQUFDO0lBRUQsV0FBVztRQUNWLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7SUFDekIsQ0FBQztJQUVELE9BQU87UUFDTixJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztRQUMzQixJQUFJLENBQUMsS0FBSyxHQUFHLFNBQVMsQ0FBQztJQUN4QixDQUFDO0NBQ0Q7QUFZRCxNQUFhLGNBQWUsU0FBUSxFQUFFLENBQUMsWUFBWTtJQVlsRDtRQUNDLEtBQUssRUFBRSxDQUFDO1FBVEQsaUJBQVksR0FBRyxJQUFJLE9BQU8sRUFBd0IsQ0FBQztRQU1uRCxxQkFBZ0IsR0FBRyxJQUFJLEdBQUcsRUFBc0QsQ0FBQztRQVdsRixxQkFBZ0IsR0FBaUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUM7SUFQaEYsQ0FBQztJQUVELDRFQUE0RTtJQUVyRSxPQUFPO0lBQ2QsQ0FBQztJQUlNLGFBQWEsQ0FBQyxHQUFrQztRQUN0RCxJQUFJLEdBQUcsQ0FBQyxJQUFJLEtBQUssU0FBUyxFQUFFO1lBQzNCLElBQUksQ0FBQyxlQUFlLENBQXdCLEdBQUcsQ0FBQyxDQUFDO1NBQ2pEO2FBQU0sSUFBSSxHQUFHLENBQUMsSUFBSSxLQUFLLFVBQVUsRUFBRTtZQUNuQyxNQUFNLFFBQVEsR0FBMkIsR0FBRyxDQUFDO1lBQzdDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQzVELElBQUksR0FBRyxFQUFFO2dCQUNSLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDO2dCQUNuRCxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUM7YUFDZDtTQUNEO0lBQ0YsQ0FBQztJQUVTLGdCQUFnQjtRQUN6QixPQUFPLElBQUksQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUM3RCxDQUFDO0lBRUQsNEVBQTRFO0lBRXJFLEtBQUssQ0FBQyxRQUErQixFQUFFLFNBQWdDO1FBQzdFLElBQUksQ0FBQyxTQUFTLEdBQUcsQ0FBQyxDQUFDO1FBQ25CLElBQUksQ0FBQyxlQUFlLEdBQUcsU0FBUyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFOUIsUUFBUSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxJQUFZLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUU5RCxRQUFRLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxHQUFHLEVBQUU7WUFDekIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLGdCQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUNyQyxDQUFDLENBQUMsQ0FBQztRQUNILFFBQVEsQ0FBQyxFQUFFLENBQUMsT0FBTyxFQUFFLENBQUMsS0FBSyxFQUFFLEVBQUU7WUFDOUIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLGdCQUFLLENBQUMsT0FBTyxFQUFFLGtCQUFrQixHQUFHLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEYsQ0FBQyxDQUFDLENBQUM7UUFFSCxTQUFTLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxDQUFDLEtBQUssRUFBRSxFQUFFO1lBQy9CLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxnQkFBSyxDQUFDLE9BQU8sRUFBRSxtQkFBbUIsR0FBRyxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3JGLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDO0lBQ25CLENBQUM7SUFFTSxJQUFJO1FBQ1YsSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3pCLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxFQUFFLENBQUM7U0FDM0I7SUFDRixDQUFDO0lBRU0sU0FBUyxDQUFDLEtBQTBCO1FBQzFDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFFTSxZQUFZLENBQUMsUUFBZ0M7UUFDbkQsSUFBSSxRQUFRLENBQUMsR0FBRyxHQUFHLENBQUMsRUFBRTtZQUNyQixPQUFPLENBQUMsS0FBSyxDQUFDLHNEQUFzRCxRQUFRLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUN4RjthQUFNO1lBQ04sSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLEVBQUUsUUFBUSxDQUFDLENBQUM7U0FDakM7SUFDRixDQUFDO0lBRU0sV0FBVyxDQUFDLE9BQWUsRUFBRSxJQUFTLEVBQUUsT0FBZSxFQUFFLEVBQThDO1FBRTdHLE1BQU0sT0FBTyxHQUFRO1lBQ3BCLE9BQU8sRUFBRSxPQUFPO1NBQ2hCLENBQUM7UUFDRixJQUFJLElBQUksSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDekMsT0FBTyxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7U0FDekI7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUMxQixJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksZ0JBQUssQ0FBQyxPQUFPLEVBQUUsZ0NBQWdDLENBQUMsQ0FBQyxDQUFDO1lBQ3RFLE9BQU87U0FDUDtRQUVELElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRS9CLElBQUksRUFBRSxFQUFFO1lBQ1AsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBRTNDLE1BQU0sS0FBSyxHQUFHLFVBQVUsQ0FBQyxHQUFHLEVBQUU7Z0JBQzdCLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDcEIsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ25ELElBQUksR0FBRyxFQUFFO29CQUNSLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO29CQUMxQyxHQUFHLENBQUMsSUFBSSxtQkFBUSxDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDO2lCQUN0QztZQUNGLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztTQUNaO0lBQ0YsQ0FBQztJQUVELDRFQUE0RTtJQUVsRSxlQUFlLENBQUMsT0FBOEI7SUFDeEQsQ0FBQztJQUVELDRFQUE0RTtJQUVwRSxVQUFVLENBQUMsS0FBMEI7UUFDNUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFTyxLQUFLLENBQUMsR0FBcUMsRUFBRSxPQUFzQztRQUUxRixPQUFPLENBQUMsSUFBSSxHQUFHLEdBQUcsQ0FBQztRQUNuQixPQUFPLENBQUMsR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUUvQixJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDekIsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUNyQyxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsTUFBTSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLFdBQVcsSUFBSSxFQUFFLEVBQUUsTUFBTSxDQUFDLENBQUM7U0FDeEc7UUFDRCxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRU8sV0FBVyxDQUFDLElBQVk7UUFFL0IsSUFBSSxDQUFDLFFBQVEsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBRXJELE9BQU8sSUFBSSxFQUFFO1lBQ1osSUFBSSxJQUFJLENBQUMsY0FBYyxJQUFJLENBQUMsRUFBRTtnQkFDN0IsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sSUFBSSxJQUFJLENBQUMsY0FBYyxFQUFFO29CQUNoRCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztvQkFDdkUsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7b0JBQ3pELElBQUksQ0FBQyxjQUFjLEdBQUcsQ0FBQyxDQUFDLENBQUM7b0JBQ3pCLElBQUksT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7d0JBQ3ZCLElBQUk7NEJBQ0gsSUFBSSxHQUFHLEdBQWtDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7NEJBQzdELElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7eUJBQ3hCO3dCQUNELE9BQU8sQ0FBQyxFQUFFOzRCQUNULElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxnQkFBSyxDQUFDLE9BQU8sRUFBRSx1QkFBdUIsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO3lCQUNoRjtxQkFDRDtvQkFDRCxTQUFTLENBQUMsaURBQWlEO2lCQUMzRDthQUNEO2lCQUFNO2dCQUNOLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDM0QsSUFBSSxHQUFHLEtBQUssQ0FBQyxDQUFDLEVBQUU7b0JBQ2YsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQztvQkFDdEQsTUFBTSxLQUFLLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztvQkFDbkMsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUU7d0JBQ3RDLE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7d0JBQ25DLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLGdCQUFnQixFQUFFOzRCQUNoQyxJQUFJLENBQUMsY0FBYyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO3lCQUMvQjtxQkFDRDtvQkFDRCxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEdBQUcsR0FBRyxjQUFjLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDO29CQUMxRSxTQUFTO2lCQUNUO2FBQ0Q7WUFDRCxNQUFNO1NBQ047SUFDRixDQUFDOztBQTFLYyx1QkFBUSxHQUFHLFVBQVUsQ0FBQztBQUZ0Qyx3Q0E2S0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKi0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICogIENvcHlyaWdodCAoYykgTWljcm9zb2Z0IENvcnBvcmF0aW9uLiBBbGwgcmlnaHRzIHJlc2VydmVkLlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBNSVQgTGljZW5zZS4gU2VlIExpY2Vuc2UudHh0IGluIHRoZSBwcm9qZWN0IHJvb3QgZm9yIGxpY2Vuc2UgaW5mb3JtYXRpb24uXG4gKi0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tKi9cblxuaW1wb3J0ICogYXMgZWUgZnJvbSAnZXZlbnRzJztcbmltcG9ydCB7IERlYnVnUHJvdG9jb2wgfSBmcm9tICd2c2NvZGUtZGVidWdwcm90b2NvbCc7XG5pbXBvcnQgeyBSZXNwb25zZSwgRXZlbnQgfSBmcm9tICcuL21lc3NhZ2VzJztcblxuaW50ZXJmYWNlIERlYnVnUHJvdG9jb2xNZXNzYWdlIHtcbn1cblxuaW50ZXJmYWNlIElEaXNwb3NhYmxlIHtcblx0ZGlzcG9zZSgpOiB2b2lkO1xufVxuXG5jbGFzcyBEaXNwb3NhYmxlMCBpbXBsZW1lbnRzIElEaXNwb3NhYmxlIHtcblx0ZGlzcG9zZSgpOiBhbnkge1xuXHR9XG59XG5cbmludGVyZmFjZSBFdmVudDA8VD4ge1xuXHQobGlzdGVuZXI6IChlOiBUKSA9PiBhbnksIHRoaXNBcmc/OiBhbnkpOiBEaXNwb3NhYmxlMDtcbn1cblxuY2xhc3MgRW1pdHRlcjxUPiB7XG5cblx0cHJpdmF0ZSBfZXZlbnQ/OiBFdmVudDA8VD47XG5cdHByaXZhdGUgX2xpc3RlbmVyPzogKGU6IFQpID0+IHZvaWQ7XG5cdHByaXZhdGUgX3RoaXM/OiBhbnk7XG5cblx0Z2V0IGV2ZW50KCk6IEV2ZW50MDxUPiB7XG5cdFx0aWYgKCF0aGlzLl9ldmVudCkge1xuXHRcdFx0dGhpcy5fZXZlbnQgPSAobGlzdGVuZXI6IChlOiBUKSA9PiBhbnksIHRoaXNBcmc/OiBhbnkpID0+IHtcblxuXHRcdFx0XHR0aGlzLl9saXN0ZW5lciA9IGxpc3RlbmVyO1xuXHRcdFx0XHR0aGlzLl90aGlzID0gdGhpc0FyZztcblxuXHRcdFx0XHRsZXQgcmVzdWx0OiBJRGlzcG9zYWJsZTtcblx0XHRcdFx0cmVzdWx0ID0ge1xuXHRcdFx0XHRcdGRpc3Bvc2U6ICgpID0+IHtcblx0XHRcdFx0XHRcdHRoaXMuX2xpc3RlbmVyID0gdW5kZWZpbmVkO1xuXHRcdFx0XHRcdFx0dGhpcy5fdGhpcyA9IHVuZGVmaW5lZDtcblx0XHRcdFx0XHR9XG5cdFx0XHRcdH07XG5cdFx0XHRcdHJldHVybiByZXN1bHQ7XG5cdFx0XHR9O1xuXHRcdH1cblx0XHRyZXR1cm4gdGhpcy5fZXZlbnQ7XG5cdH1cblxuXHRmaXJlKGV2ZW50OiBUKTogdm9pZCB7XG5cdFx0aWYgKHRoaXMuX2xpc3RlbmVyKSB7XG5cdFx0XHR0cnkge1xuXHRcdFx0XHR0aGlzLl9saXN0ZW5lci5jYWxsKHRoaXMuX3RoaXMsIGV2ZW50KTtcblx0XHRcdH0gY2F0Y2ggKGUpIHtcblx0XHRcdH1cblx0XHR9XG5cdH1cblxuXHRoYXNMaXN0ZW5lcigpIDogYm9vbGVhbiB7XG5cdFx0cmV0dXJuICEhdGhpcy5fbGlzdGVuZXI7XG5cdH1cblxuXHRkaXNwb3NlKCkge1xuXHRcdHRoaXMuX2xpc3RlbmVyID0gdW5kZWZpbmVkO1xuXHRcdHRoaXMuX3RoaXMgPSB1bmRlZmluZWQ7XG5cdH1cbn1cblxuLyoqXG4gKiBBIHN0cnVjdHVyYWxseSBlcXVpdmFsZW50IGNvcHkgb2YgdnNjb2RlLkRlYnVnQWRhcHRlclxuICovXG5pbnRlcmZhY2UgVlNDb2RlRGVidWdBZGFwdGVyIGV4dGVuZHMgRGlzcG9zYWJsZTAge1xuXG5cdHJlYWRvbmx5IG9uRGlkU2VuZE1lc3NhZ2U6IEV2ZW50MDxEZWJ1Z1Byb3RvY29sTWVzc2FnZT47XG5cblx0aGFuZGxlTWVzc2FnZShtZXNzYWdlOiBEZWJ1Z1Byb3RvY29sLlByb3RvY29sTWVzc2FnZSk6IHZvaWQ7XG59XG5cbmV4cG9ydCBjbGFzcyBQcm90b2NvbFNlcnZlciBleHRlbmRzIGVlLkV2ZW50RW1pdHRlciBpbXBsZW1lbnRzIFZTQ29kZURlYnVnQWRhcHRlciB7XG5cblx0cHJpdmF0ZSBzdGF0aWMgVFdPX0NSTEYgPSAnXFxyXFxuXFxyXFxuJztcblxuXHRwcml2YXRlIF9zZW5kTWVzc2FnZSA9IG5ldyBFbWl0dGVyPERlYnVnUHJvdG9jb2xNZXNzYWdlPigpO1xuXG5cdHByaXZhdGUgX3Jhd0RhdGE6IEJ1ZmZlcjtcblx0cHJpdmF0ZSBfY29udGVudExlbmd0aDogbnVtYmVyO1xuXHRwcml2YXRlIF9zZXF1ZW5jZTogbnVtYmVyO1xuXHRwcml2YXRlIF93cml0YWJsZVN0cmVhbTogTm9kZUpTLldyaXRhYmxlU3RyZWFtO1xuXHRwcml2YXRlIF9wZW5kaW5nUmVxdWVzdHMgPSBuZXcgTWFwPG51bWJlciwgKHJlc3BvbnNlOiBEZWJ1Z1Byb3RvY29sLlJlc3BvbnNlKSA9PiB2b2lkPigpO1xuXG5cdGNvbnN0cnVjdG9yKCkge1xuXHRcdHN1cGVyKCk7XG5cdH1cblxuXHQvLyAtLS0tIGltcGxlbWVudHMgdnNjb2RlLkRlYnVnYWRhcHRlciBpbnRlcmZhY2UgLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG5cblx0cHVibGljIGRpc3Bvc2UoKTogYW55IHtcblx0fVxuXG5cdHB1YmxpYyBvbkRpZFNlbmRNZXNzYWdlOiBFdmVudDA8RGVidWdQcm90b2NvbE1lc3NhZ2U+ID0gdGhpcy5fc2VuZE1lc3NhZ2UuZXZlbnQ7XG5cblx0cHVibGljIGhhbmRsZU1lc3NhZ2UobXNnOiBEZWJ1Z1Byb3RvY29sLlByb3RvY29sTWVzc2FnZSk6IHZvaWQge1xuXHRcdGlmIChtc2cudHlwZSA9PT0gJ3JlcXVlc3QnKSB7XG5cdFx0XHR0aGlzLmRpc3BhdGNoUmVxdWVzdCg8RGVidWdQcm90b2NvbC5SZXF1ZXN0Pm1zZyk7XG5cdFx0fSBlbHNlIGlmIChtc2cudHlwZSA9PT0gJ3Jlc3BvbnNlJykge1xuXHRcdFx0Y29uc3QgcmVzcG9uc2UgPSA8RGVidWdQcm90b2NvbC5SZXNwb25zZT5tc2c7XG5cdFx0XHRjb25zdCBjbGIgPSB0aGlzLl9wZW5kaW5nUmVxdWVzdHMuZ2V0KHJlc3BvbnNlLnJlcXVlc3Rfc2VxKTtcblx0XHRcdGlmIChjbGIpIHtcblx0XHRcdFx0dGhpcy5fcGVuZGluZ1JlcXVlc3RzLmRlbGV0ZShyZXNwb25zZS5yZXF1ZXN0X3NlcSk7XG5cdFx0XHRcdGNsYihyZXNwb25zZSk7XG5cdFx0XHR9XG5cdFx0fVxuXHR9XG5cblx0cHJvdGVjdGVkIF9pc1J1bm5pbmdJbmxpbmUoKSB7XG5cdFx0cmV0dXJuIHRoaXMuX3NlbmRNZXNzYWdlICYmIHRoaXMuX3NlbmRNZXNzYWdlLmhhc0xpc3RlbmVyKCk7XG5cdH1cblxuXHQvLy0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG5cblx0cHVibGljIHN0YXJ0KGluU3RyZWFtOiBOb2RlSlMuUmVhZGFibGVTdHJlYW0sIG91dFN0cmVhbTogTm9kZUpTLldyaXRhYmxlU3RyZWFtKTogdm9pZCB7XG5cdFx0dGhpcy5fc2VxdWVuY2UgPSAxO1xuXHRcdHRoaXMuX3dyaXRhYmxlU3RyZWFtID0gb3V0U3RyZWFtO1xuXHRcdHRoaXMuX3Jhd0RhdGEgPSBuZXcgQnVmZmVyKDApO1xuXG5cdFx0aW5TdHJlYW0ub24oJ2RhdGEnLCAoZGF0YTogQnVmZmVyKSA9PiB0aGlzLl9oYW5kbGVEYXRhKGRhdGEpKTtcblxuXHRcdGluU3RyZWFtLm9uKCdjbG9zZScsICgpID0+IHtcblx0XHRcdHRoaXMuX2VtaXRFdmVudChuZXcgRXZlbnQoJ2Nsb3NlJykpO1xuXHRcdH0pO1xuXHRcdGluU3RyZWFtLm9uKCdlcnJvcicsIChlcnJvcikgPT4ge1xuXHRcdFx0dGhpcy5fZW1pdEV2ZW50KG5ldyBFdmVudCgnZXJyb3InLCAnaW5TdHJlYW0gZXJyb3I6ICcgKyAoZXJyb3IgJiYgZXJyb3IubWVzc2FnZSkpKTtcblx0XHR9KTtcblxuXHRcdG91dFN0cmVhbS5vbignZXJyb3InLCAoZXJyb3IpID0+IHtcblx0XHRcdHRoaXMuX2VtaXRFdmVudChuZXcgRXZlbnQoJ2Vycm9yJywgJ291dFN0cmVhbSBlcnJvcjogJyArIChlcnJvciAmJiBlcnJvci5tZXNzYWdlKSkpO1xuXHRcdH0pO1xuXG5cdFx0aW5TdHJlYW0ucmVzdW1lKCk7XG5cdH1cblxuXHRwdWJsaWMgc3RvcCgpOiB2b2lkIHtcblx0XHRpZiAodGhpcy5fd3JpdGFibGVTdHJlYW0pIHtcblx0XHRcdHRoaXMuX3dyaXRhYmxlU3RyZWFtLmVuZCgpO1xuXHRcdH1cblx0fVxuXG5cdHB1YmxpYyBzZW5kRXZlbnQoZXZlbnQ6IERlYnVnUHJvdG9jb2wuRXZlbnQpOiB2b2lkIHtcblx0XHR0aGlzLl9zZW5kKCdldmVudCcsIGV2ZW50KTtcblx0fVxuXG5cdHB1YmxpYyBzZW5kUmVzcG9uc2UocmVzcG9uc2U6IERlYnVnUHJvdG9jb2wuUmVzcG9uc2UpOiB2b2lkIHtcblx0XHRpZiAocmVzcG9uc2Uuc2VxID4gMCkge1xuXHRcdFx0Y29uc29sZS5lcnJvcihgYXR0ZW1wdCB0byBzZW5kIG1vcmUgdGhhbiBvbmUgcmVzcG9uc2UgZm9yIGNvbW1hbmQgJHtyZXNwb25zZS5jb21tYW5kfWApO1xuXHRcdH0gZWxzZSB7XG5cdFx0XHR0aGlzLl9zZW5kKCdyZXNwb25zZScsIHJlc3BvbnNlKTtcblx0XHR9XG5cdH1cblxuXHRwdWJsaWMgc2VuZFJlcXVlc3QoY29tbWFuZDogc3RyaW5nLCBhcmdzOiBhbnksIHRpbWVvdXQ6IG51bWJlciwgY2I6IChyZXNwb25zZTogRGVidWdQcm90b2NvbC5SZXNwb25zZSkgPT4gdm9pZCkgOiB2b2lkIHtcblxuXHRcdGNvbnN0IHJlcXVlc3Q6IGFueSA9IHtcblx0XHRcdGNvbW1hbmQ6IGNvbW1hbmRcblx0XHR9O1xuXHRcdGlmIChhcmdzICYmIE9iamVjdC5rZXlzKGFyZ3MpLmxlbmd0aCA+IDApIHtcblx0XHRcdHJlcXVlc3QuYXJndW1lbnRzID0gYXJncztcblx0XHR9XG5cblx0XHRpZiAoIXRoaXMuX3dyaXRhYmxlU3RyZWFtKSB7XG5cdFx0XHR0aGlzLl9lbWl0RXZlbnQobmV3IEV2ZW50KCdlcnJvcicsICdzZW5kUmVxdWVzdDogTm8gd3JpdGFibGVTdHJlYW0nKSk7XG5cdFx0XHRyZXR1cm47XG5cdFx0fVxuXG5cdFx0dGhpcy5fc2VuZCgncmVxdWVzdCcsIHJlcXVlc3QpO1xuXG5cdFx0aWYgKGNiKSB7XG5cdFx0XHR0aGlzLl9wZW5kaW5nUmVxdWVzdHMuc2V0KHJlcXVlc3Quc2VxLCBjYik7XG5cblx0XHRcdGNvbnN0IHRpbWVyID0gc2V0VGltZW91dCgoKSA9PiB7XG5cdFx0XHRcdGNsZWFyVGltZW91dCh0aW1lcik7XG5cdFx0XHRcdGNvbnN0IGNsYiA9IHRoaXMuX3BlbmRpbmdSZXF1ZXN0cy5nZXQocmVxdWVzdC5zZXEpO1xuXHRcdFx0XHRpZiAoY2xiKSB7XG5cdFx0XHRcdFx0dGhpcy5fcGVuZGluZ1JlcXVlc3RzLmRlbGV0ZShyZXF1ZXN0LnNlcSk7XG5cdFx0XHRcdFx0Y2xiKG5ldyBSZXNwb25zZShyZXF1ZXN0LCAndGltZW91dCcpKTtcblx0XHRcdFx0fVxuXHRcdFx0fSwgdGltZW91dCk7XG5cdFx0fVxuXHR9XG5cblx0Ly8gLS0tLSBwcm90ZWN0ZWQgLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuXG5cdHByb3RlY3RlZCBkaXNwYXRjaFJlcXVlc3QocmVxdWVzdDogRGVidWdQcm90b2NvbC5SZXF1ZXN0KTogdm9pZCB7XG5cdH1cblxuXHQvLyAtLS0tIHByaXZhdGUgLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG5cblx0cHJpdmF0ZSBfZW1pdEV2ZW50KGV2ZW50OiBEZWJ1Z1Byb3RvY29sLkV2ZW50KSB7XG5cdFx0dGhpcy5lbWl0KGV2ZW50LmV2ZW50LCBldmVudCk7XG5cdH1cblxuXHRwcml2YXRlIF9zZW5kKHR5cDogJ3JlcXVlc3QnIHwgJ3Jlc3BvbnNlJyB8ICdldmVudCcsIG1lc3NhZ2U6IERlYnVnUHJvdG9jb2wuUHJvdG9jb2xNZXNzYWdlKTogdm9pZCB7XG5cblx0XHRtZXNzYWdlLnR5cGUgPSB0eXA7XG5cdFx0bWVzc2FnZS5zZXEgPSB0aGlzLl9zZXF1ZW5jZSsrO1xuXG5cdFx0aWYgKHRoaXMuX3dyaXRhYmxlU3RyZWFtKSB7XG5cdFx0XHRjb25zdCBqc29uID0gSlNPTi5zdHJpbmdpZnkobWVzc2FnZSk7XG5cdFx0XHR0aGlzLl93cml0YWJsZVN0cmVhbS53cml0ZShgQ29udGVudC1MZW5ndGg6ICR7QnVmZmVyLmJ5dGVMZW5ndGgoanNvbiwgJ3V0ZjgnKX1cXHJcXG5cXHJcXG4ke2pzb259YCwgJ3V0ZjgnKTtcblx0XHR9XG5cdFx0dGhpcy5fc2VuZE1lc3NhZ2UuZmlyZShtZXNzYWdlKTtcblx0fVxuXG5cdHByaXZhdGUgX2hhbmRsZURhdGEoZGF0YTogQnVmZmVyKTogdm9pZCB7XG5cblx0XHR0aGlzLl9yYXdEYXRhID0gQnVmZmVyLmNvbmNhdChbdGhpcy5fcmF3RGF0YSwgZGF0YV0pO1xuXG5cdFx0d2hpbGUgKHRydWUpIHtcblx0XHRcdGlmICh0aGlzLl9jb250ZW50TGVuZ3RoID49IDApIHtcblx0XHRcdFx0aWYgKHRoaXMuX3Jhd0RhdGEubGVuZ3RoID49IHRoaXMuX2NvbnRlbnRMZW5ndGgpIHtcblx0XHRcdFx0XHRjb25zdCBtZXNzYWdlID0gdGhpcy5fcmF3RGF0YS50b1N0cmluZygndXRmOCcsIDAsIHRoaXMuX2NvbnRlbnRMZW5ndGgpO1xuXHRcdFx0XHRcdHRoaXMuX3Jhd0RhdGEgPSB0aGlzLl9yYXdEYXRhLnNsaWNlKHRoaXMuX2NvbnRlbnRMZW5ndGgpO1xuXHRcdFx0XHRcdHRoaXMuX2NvbnRlbnRMZW5ndGggPSAtMTtcblx0XHRcdFx0XHRpZiAobWVzc2FnZS5sZW5ndGggPiAwKSB7XG5cdFx0XHRcdFx0XHR0cnkge1xuXHRcdFx0XHRcdFx0XHRsZXQgbXNnOiBEZWJ1Z1Byb3RvY29sLlByb3RvY29sTWVzc2FnZSA9IEpTT04ucGFyc2UobWVzc2FnZSk7XG5cdFx0XHRcdFx0XHRcdHRoaXMuaGFuZGxlTWVzc2FnZShtc2cpO1xuXHRcdFx0XHRcdFx0fVxuXHRcdFx0XHRcdFx0Y2F0Y2ggKGUpIHtcblx0XHRcdFx0XHRcdFx0dGhpcy5fZW1pdEV2ZW50KG5ldyBFdmVudCgnZXJyb3InLCAnRXJyb3IgaGFuZGxpbmcgZGF0YTogJyArIChlICYmIGUubWVzc2FnZSkpKTtcblx0XHRcdFx0XHRcdH1cblx0XHRcdFx0XHR9XG5cdFx0XHRcdFx0Y29udGludWU7XHQvLyB0aGVyZSBtYXkgYmUgbW9yZSBjb21wbGV0ZSBtZXNzYWdlcyB0byBwcm9jZXNzXG5cdFx0XHRcdH1cblx0XHRcdH0gZWxzZSB7XG5cdFx0XHRcdGNvbnN0IGlkeCA9IHRoaXMuX3Jhd0RhdGEuaW5kZXhPZihQcm90b2NvbFNlcnZlci5UV09fQ1JMRik7XG5cdFx0XHRcdGlmIChpZHggIT09IC0xKSB7XG5cdFx0XHRcdFx0Y29uc3QgaGVhZGVyID0gdGhpcy5fcmF3RGF0YS50b1N0cmluZygndXRmOCcsIDAsIGlkeCk7XG5cdFx0XHRcdFx0Y29uc3QgbGluZXMgPSBoZWFkZXIuc3BsaXQoJ1xcclxcbicpO1xuXHRcdFx0XHRcdGZvciAobGV0IGkgPSAwOyBpIDwgbGluZXMubGVuZ3RoOyBpKyspIHtcblx0XHRcdFx0XHRcdGNvbnN0IHBhaXIgPSBsaW5lc1tpXS5zcGxpdCgvOiArLyk7XG5cdFx0XHRcdFx0XHRpZiAocGFpclswXSA9PSAnQ29udGVudC1MZW5ndGgnKSB7XG5cdFx0XHRcdFx0XHRcdHRoaXMuX2NvbnRlbnRMZW5ndGggPSArcGFpclsxXTtcblx0XHRcdFx0XHRcdH1cblx0XHRcdFx0XHR9XG5cdFx0XHRcdFx0dGhpcy5fcmF3RGF0YSA9IHRoaXMuX3Jhd0RhdGEuc2xpY2UoaWR4ICsgUHJvdG9jb2xTZXJ2ZXIuVFdPX0NSTEYubGVuZ3RoKTtcblx0XHRcdFx0XHRjb250aW51ZTtcblx0XHRcdFx0fVxuXHRcdFx0fVxuXHRcdFx0YnJlYWs7XG5cdFx0fVxuXHR9XG59XG4iXX0=