"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.InternalLogger = void 0;
const fs = require("fs");
const path = require("path");
const mkdirp = require("mkdirp");
function mkdirpPromise(folder) {
    return new Promise((resolve, reject) => {
        mkdirp(folder, (err, made) => {
            if (err) {
                reject(err);
            }
            else {
                resolve(made);
            }
        });
    });
}
const logger_1 = require("./logger");
/**
 * Manages logging, whether to console.log, file, or VS Code console.
 * Encapsulates the state specific to each logging session
 */
class InternalLogger {
    constructor(logCallback, isServer) {
        /** Dispose and allow exit to continue normally */
        this.beforeExitCallback = () => this.dispose();
        this._logCallback = logCallback;
        this._logToConsole = isServer;
        this._minLogLevel = logger_1.LogLevel.Warn;
        this.disposeCallback = (signal, code) => {
            this.dispose();
            // Exit with 128 + value of the signal code.
            // https://nodejs.org/api/process.html#process_exit_codes
            code = code || 2; // SIGINT
            code += 128;
            process.exit(code);
        };
    }
    setup(options) {
        return __awaiter(this, void 0, void 0, function* () {
            this._minLogLevel = options.consoleMinLogLevel;
            this._prependTimestamp = options.prependTimestamp;
            // Open a log file in the specified location. Overwritten on each run.
            if (options.logFilePath) {
                if (!path.isAbsolute(options.logFilePath)) {
                    this.log(`logFilePath must be an absolute path: ${options.logFilePath}`, logger_1.LogLevel.Error);
                }
                else {
                    const handleError = err => this.sendLog(`Error creating log file at path: ${options.logFilePath}. Error: ${err.toString()}\n`, logger_1.LogLevel.Error);
                    try {
                        yield mkdirpPromise(path.dirname(options.logFilePath));
                        this.log(`Verbose logs are written to:\n`, logger_1.LogLevel.Warn);
                        this.log(options.logFilePath + '\n', logger_1.LogLevel.Warn);
                        this._logFileStream = fs.createWriteStream(options.logFilePath);
                        this.logDateTime();
                        this.setupShutdownListeners();
                        this._logFileStream.on('error', err => {
                            handleError(err);
                        });
                    }
                    catch (err) {
                        handleError(err);
                    }
                }
            }
        });
    }
    logDateTime() {
        let d = new Date();
        let dateString = d.getUTCFullYear() + '-' + `${d.getUTCMonth() + 1}` + '-' + d.getUTCDate();
        const timeAndDateStamp = dateString + ', ' + getFormattedTimeString();
        this.log(timeAndDateStamp + '\n', logger_1.LogLevel.Verbose, false);
    }
    setupShutdownListeners() {
        process.addListener('beforeExit', this.beforeExitCallback);
        process.addListener('SIGTERM', this.disposeCallback);
        process.addListener('SIGINT', this.disposeCallback);
    }
    removeShutdownListeners() {
        process.removeListener('beforeExit', this.beforeExitCallback);
        process.removeListener('SIGTERM', this.disposeCallback);
        process.removeListener('SIGINT', this.disposeCallback);
    }
    dispose() {
        return new Promise(resolve => {
            this.removeShutdownListeners();
            if (this._logFileStream) {
                this._logFileStream.end(resolve);
                this._logFileStream = null;
            }
            else {
                resolve();
            }
        });
    }
    log(msg, level, prependTimestamp = true) {
        if (this._minLogLevel === logger_1.LogLevel.Stop) {
            return;
        }
        if (level >= this._minLogLevel) {
            this.sendLog(msg, level);
        }
        if (this._logToConsole) {
            const logFn = level === logger_1.LogLevel.Error ? console.error :
                level === logger_1.LogLevel.Warn ? console.warn :
                    null;
            if (logFn) {
                logFn(logger_1.trimLastNewline(msg));
            }
        }
        // If an error, prepend with '[Error]'
        if (level === logger_1.LogLevel.Error) {
            msg = `[${logger_1.LogLevel[level]}] ${msg}`;
        }
        if (this._prependTimestamp && prependTimestamp) {
            msg = '[' + getFormattedTimeString() + '] ' + msg;
        }
        if (this._logFileStream) {
            this._logFileStream.write(msg);
        }
    }
    sendLog(msg, level) {
        // Truncate long messages, they can hang VS Code
        if (msg.length > 1500) {
            const endsInNewline = !!msg.match(/(\n|\r\n)$/);
            msg = msg.substr(0, 1500) + '[...]';
            if (endsInNewline) {
                msg = msg + '\n';
            }
        }
        if (this._logCallback) {
            const event = new logger_1.LogOutputEvent(msg, level);
            this._logCallback(event);
        }
    }
}
exports.InternalLogger = InternalLogger;
function getFormattedTimeString() {
    let d = new Date();
    let hourString = _padZeroes(2, String(d.getUTCHours()));
    let minuteString = _padZeroes(2, String(d.getUTCMinutes()));
    let secondString = _padZeroes(2, String(d.getUTCSeconds()));
    let millisecondString = _padZeroes(3, String(d.getUTCMilliseconds()));
    return hourString + ':' + minuteString + ':' + secondString + '.' + millisecondString + ' UTC';
}
function _padZeroes(minDesiredLength, numberToPad) {
    if (numberToPad.length >= minDesiredLength) {
        return numberToPad;
    }
    else {
        return String('0'.repeat(minDesiredLength) + numberToPad).slice(-minDesiredLength);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZXJuYWxMb2dnZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW50ZXJuYWxMb2dnZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Z0dBR2dHOzs7Ozs7Ozs7Ozs7QUFFaEcseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3QixpQ0FBaUM7QUFFakMsU0FBUyxhQUFhLENBQUMsTUFBYztJQUNwQyxPQUFPLElBQUksT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLE1BQU0sRUFBRSxFQUFFO1FBQ3RDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxHQUFHLEVBQUUsSUFBSSxFQUFFLEVBQUU7WUFDNUIsSUFBSSxHQUFHLEVBQUU7Z0JBQ1IsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO2FBQ1o7aUJBQU07Z0JBQ04sT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQ2Q7UUFDRixDQUFDLENBQUMsQ0FBQztJQUNKLENBQUMsQ0FBQyxDQUFDO0FBQ0osQ0FBQztBQUVELHFDQUE0SDtBQUU1SDs7O0dBR0c7QUFDSCxNQUFhLGNBQWM7SUFtQjFCLFlBQVksV0FBeUIsRUFBRSxRQUFrQjtRQVR6RCxrREFBa0Q7UUFDMUMsdUJBQWtCLEdBQUcsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBU2pELElBQUksQ0FBQyxZQUFZLEdBQUcsV0FBVyxDQUFDO1FBQ2hDLElBQUksQ0FBQyxhQUFhLEdBQUcsUUFBUSxDQUFDO1FBRTlCLElBQUksQ0FBQyxZQUFZLEdBQUcsaUJBQVEsQ0FBQyxJQUFJLENBQUM7UUFFbEMsSUFBSSxDQUFDLGVBQWUsR0FBRyxDQUFDLE1BQWMsRUFBRSxJQUFZLEVBQUUsRUFBRTtZQUN2RCxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7WUFFZiw0Q0FBNEM7WUFDNUMseURBQXlEO1lBQ3pELElBQUksR0FBRyxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMzQixJQUFJLElBQUksR0FBRyxDQUFDO1lBRVosT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNwQixDQUFDLENBQUM7SUFDSCxDQUFDO0lBRVksS0FBSyxDQUFDLE9BQStCOztZQUNqRCxJQUFJLENBQUMsWUFBWSxHQUFHLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQztZQUMvQyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsT0FBTyxDQUFDLGdCQUFnQixDQUFDO1lBRWxELHNFQUFzRTtZQUN0RSxJQUFJLE9BQU8sQ0FBQyxXQUFXLEVBQUU7Z0JBQ3hCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsRUFBRTtvQkFDMUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5Q0FBeUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUFFLGlCQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7aUJBQ3pGO3FCQUFNO29CQUNOLE1BQU0sV0FBVyxHQUFHLEdBQUcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxvQ0FBb0MsT0FBTyxDQUFDLFdBQVcsWUFBWSxHQUFHLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxpQkFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO29CQUUvSSxJQUFJO3dCQUNILE1BQU0sYUFBYSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUM7d0JBQ3ZELElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLEVBQUUsaUJBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQzt3QkFDMUQsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxHQUFHLElBQUksRUFBRSxpQkFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO3dCQUVwRCxJQUFJLENBQUMsY0FBYyxHQUFHLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7d0JBQ2hFLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQzt3QkFDbkIsSUFBSSxDQUFDLHNCQUFzQixFQUFFLENBQUM7d0JBQzlCLElBQUksQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsRUFBRTs0QkFDckMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDO3dCQUNsQixDQUFDLENBQUMsQ0FBQztxQkFDSDtvQkFBQyxPQUFPLEdBQUcsRUFBRTt3QkFDYixXQUFXLENBQUMsR0FBRyxDQUFDLENBQUM7cUJBQ2pCO2lCQUNEO2FBQ0Q7UUFDRixDQUFDO0tBQUE7SUFFTyxXQUFXO1FBQ2xCLElBQUksQ0FBQyxHQUFHLElBQUksSUFBSSxFQUFFLENBQUM7UUFDbkIsSUFBSSxVQUFVLEdBQUcsQ0FBQyxDQUFDLGNBQWMsRUFBRSxHQUFHLEdBQUcsR0FBRyxHQUFHLENBQUMsQ0FBQyxXQUFXLEVBQUUsR0FBRyxDQUFDLEVBQUUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQzVGLE1BQU0sZ0JBQWdCLEdBQUcsVUFBVSxHQUFHLElBQUksR0FBRyxzQkFBc0IsRUFBRSxDQUFDO1FBQ3RFLElBQUksQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxFQUFFLGlCQUFRLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFTyxzQkFBc0I7UUFDN0IsT0FBTyxDQUFDLFdBQVcsQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDM0QsT0FBTyxDQUFDLFdBQVcsQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQ3JELE9BQU8sQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRU8sdUJBQXVCO1FBQzlCLE9BQU8sQ0FBQyxjQUFjLENBQUMsWUFBWSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzlELE9BQU8sQ0FBQyxjQUFjLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUN4RCxPQUFPLENBQUMsY0FBYyxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDeEQsQ0FBQztJQUVNLE9BQU87UUFDYixPQUFPLElBQUksT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQzVCLElBQUksQ0FBQyx1QkFBdUIsRUFBRSxDQUFDO1lBQy9CLElBQUksSUFBSSxDQUFDLGNBQWMsRUFBRTtnQkFDeEIsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBQ2pDLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDO2FBQzNCO2lCQUFNO2dCQUNOLE9BQU8sRUFBRSxDQUFDO2FBQ1Y7UUFDRixDQUFDLENBQUMsQ0FBQztJQUNKLENBQUM7SUFFTSxHQUFHLENBQUMsR0FBVyxFQUFFLEtBQWUsRUFBRSxnQkFBZ0IsR0FBRyxJQUFJO1FBQy9ELElBQUksSUFBSSxDQUFDLFlBQVksS0FBSyxpQkFBUSxDQUFDLElBQUksRUFBRTtZQUN4QyxPQUFPO1NBQ1A7UUFFRCxJQUFJLEtBQUssSUFBSSxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQy9CLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDO1NBQ3pCO1FBRUQsSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ3ZCLE1BQU0sS0FBSyxHQUNWLEtBQUssS0FBSyxpQkFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUMxQyxLQUFLLEtBQUssaUJBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztvQkFDeEMsSUFBSSxDQUFDO1lBRU4sSUFBSSxLQUFLLEVBQUU7Z0JBQ1YsS0FBSyxDQUFDLHdCQUFlLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQzthQUM1QjtTQUNEO1FBRUQsc0NBQXNDO1FBQ3RDLElBQUksS0FBSyxLQUFLLGlCQUFRLENBQUMsS0FBSyxFQUFFO1lBQzdCLEdBQUcsR0FBRyxJQUFJLGlCQUFRLENBQUMsS0FBSyxDQUFDLEtBQUssR0FBRyxFQUFFLENBQUM7U0FDcEM7UUFFRCxJQUFJLElBQUksQ0FBQyxpQkFBaUIsSUFBSSxnQkFBZ0IsRUFBRTtZQUMvQyxHQUFHLEdBQUcsR0FBRyxHQUFHLHNCQUFzQixFQUFFLEdBQUcsSUFBSSxHQUFHLEdBQUcsQ0FBQztTQUNsRDtRQUVELElBQUksSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUN4QixJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUMvQjtJQUNGLENBQUM7SUFFTyxPQUFPLENBQUMsR0FBVyxFQUFFLEtBQWU7UUFDM0MsZ0RBQWdEO1FBQ2hELElBQUksR0FBRyxDQUFDLE1BQU0sR0FBRyxJQUFJLEVBQUU7WUFDdEIsTUFBTSxhQUFhLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUM7WUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxHQUFHLE9BQU8sQ0FBQztZQUNwQyxJQUFJLGFBQWEsRUFBRTtnQkFDbEIsR0FBRyxHQUFHLEdBQUcsR0FBRyxJQUFJLENBQUM7YUFDakI7U0FDRDtRQUVELElBQUksSUFBSSxDQUFDLFlBQVksRUFBRTtZQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLHVCQUFjLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDO1lBQzdDLElBQUksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDekI7SUFDRixDQUFDO0NBQ0Q7QUFsSkQsd0NBa0pDO0FBRUQsU0FBUyxzQkFBc0I7SUFDOUIsSUFBSSxDQUFDLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztJQUNuQixJQUFJLFVBQVUsR0FBRyxVQUFVLENBQUMsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3hELElBQUksWUFBWSxHQUFHLFVBQVUsQ0FBQyxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUMsQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDNUQsSUFBSSxZQUFZLEdBQUcsVUFBVSxDQUFDLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUM1RCxJQUFJLGlCQUFpQixHQUFHLFVBQVUsQ0FBQyxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUMsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN0RSxPQUFPLFVBQVUsR0FBRyxHQUFHLEdBQUcsWUFBWSxHQUFHLEdBQUcsR0FBRyxZQUFZLEdBQUcsR0FBRyxHQUFHLGlCQUFpQixHQUFHLE1BQU0sQ0FBQztBQUNoRyxDQUFDO0FBRUQsU0FBUyxVQUFVLENBQUMsZ0JBQXdCLEVBQUUsV0FBbUI7SUFDaEUsSUFBSSxXQUFXLENBQUMsTUFBTSxJQUFJLGdCQUFnQixFQUFFO1FBQzNDLE9BQU8sV0FBVyxDQUFDO0tBQ25CO1NBQU07UUFDTixPQUFPLE1BQU0sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsV0FBVyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztLQUNuRjtBQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKi0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuICogIENvcHlyaWdodCAoYykgTWljcm9zb2Z0IENvcnBvcmF0aW9uLiBBbGwgcmlnaHRzIHJlc2VydmVkLlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBNSVQgTGljZW5zZS4gU2VlIExpY2Vuc2UudHh0IGluIHRoZSBwcm9qZWN0IHJvb3QgZm9yIGxpY2Vuc2UgaW5mb3JtYXRpb24uXG4gKi0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tKi9cblxuaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIG1rZGlycCBmcm9tICdta2RpcnAnO1xuXG5mdW5jdGlvbiBta2RpcnBQcm9taXNlKGZvbGRlcjogc3RyaW5nKTogUHJvbWlzZTxzdHJpbmc+IHtcblx0cmV0dXJuIG5ldyBQcm9taXNlKChyZXNvbHZlLCByZWplY3QpID0+IHtcblx0XHRta2RpcnAoZm9sZGVyLCAoZXJyLCBtYWRlKSA9PiB7XG5cdFx0XHRpZiAoZXJyKSB7XG5cdFx0XHRcdHJlamVjdChlcnIpO1xuXHRcdFx0fSBlbHNlIHtcblx0XHRcdFx0cmVzb2x2ZShtYWRlKTtcblx0XHRcdH1cblx0XHR9KTtcblx0fSk7XG59XG5cbmltcG9ydCB7IExvZ0xldmVsLCBJTG9nQ2FsbGJhY2ssIHRyaW1MYXN0TmV3bGluZSwgTG9nT3V0cHV0RXZlbnQsIElJbnRlcm5hbExvZ2dlck9wdGlvbnMsIElJbnRlcm5hbExvZ2dlciB9IGZyb20gJy4vbG9nZ2VyJztcblxuLyoqXG4gKiBNYW5hZ2VzIGxvZ2dpbmcsIHdoZXRoZXIgdG8gY29uc29sZS5sb2csIGZpbGUsIG9yIFZTIENvZGUgY29uc29sZS5cbiAqIEVuY2Fwc3VsYXRlcyB0aGUgc3RhdGUgc3BlY2lmaWMgdG8gZWFjaCBsb2dnaW5nIHNlc3Npb25cbiAqL1xuZXhwb3J0IGNsYXNzIEludGVybmFsTG9nZ2VyIGltcGxlbWVudHMgSUludGVybmFsTG9nZ2VyIHtcblx0cHJpdmF0ZSBfbWluTG9nTGV2ZWw6IExvZ0xldmVsO1xuXHRwcml2YXRlIF9sb2dUb0NvbnNvbGU6IGJvb2xlYW47XG5cblx0LyoqIExvZyBpbmZvIHRoYXQgbWVldHMgbWluTG9nTGV2ZWwgaXMgc2VudCB0byB0aGlzIGNhbGxiYWNrLiAqL1xuXHRwcml2YXRlIF9sb2dDYWxsYmFjazogSUxvZ0NhbGxiYWNrO1xuXG5cdC8qKiBXcml0ZSBzdGVhbSBmb3IgbG9nIGZpbGUgKi9cblx0cHJpdmF0ZSBfbG9nRmlsZVN0cmVhbTogZnMuV3JpdGVTdHJlYW07XG5cblx0LyoqIERpc3Bvc2UgYW5kIGFsbG93IGV4aXQgdG8gY29udGludWUgbm9ybWFsbHkgKi9cblx0cHJpdmF0ZSBiZWZvcmVFeGl0Q2FsbGJhY2sgPSAoKSA9PiB0aGlzLmRpc3Bvc2UoKTtcblxuXHQvKiogRGlzcG9zZSBhbmQgZXhpdCAqL1xuXHRwcml2YXRlIGRpc3Bvc2VDYWxsYmFjaztcblxuXHQvKiogV2hldGhlciB0byBhZGQgYSB0aW1lc3RhbXAgdG8gbWVzc2FnZXMgaW4gdGhlIGxvZ2ZpbGUgKi9cblx0cHJpdmF0ZSBfcHJlcGVuZFRpbWVzdGFtcDogYm9vbGVhbjtcblxuXHRjb25zdHJ1Y3Rvcihsb2dDYWxsYmFjazogSUxvZ0NhbGxiYWNrLCBpc1NlcnZlcj86IGJvb2xlYW4pIHtcblx0XHR0aGlzLl9sb2dDYWxsYmFjayA9IGxvZ0NhbGxiYWNrO1xuXHRcdHRoaXMuX2xvZ1RvQ29uc29sZSA9IGlzU2VydmVyO1xuXG5cdFx0dGhpcy5fbWluTG9nTGV2ZWwgPSBMb2dMZXZlbC5XYXJuO1xuXG5cdFx0dGhpcy5kaXNwb3NlQ2FsbGJhY2sgPSAoc2lnbmFsOiBzdHJpbmcsIGNvZGU6IG51bWJlcikgPT4ge1xuXHRcdFx0dGhpcy5kaXNwb3NlKCk7XG5cblx0XHRcdC8vIEV4aXQgd2l0aCAxMjggKyB2YWx1ZSBvZiB0aGUgc2lnbmFsIGNvZGUuXG5cdFx0XHQvLyBodHRwczovL25vZGVqcy5vcmcvYXBpL3Byb2Nlc3MuaHRtbCNwcm9jZXNzX2V4aXRfY29kZXNcblx0XHRcdGNvZGUgPSBjb2RlIHx8IDI7IC8vIFNJR0lOVFxuXHRcdFx0Y29kZSArPSAxMjg7XG5cblx0XHRcdHByb2Nlc3MuZXhpdChjb2RlKTtcblx0XHR9O1xuXHR9XG5cblx0cHVibGljIGFzeW5jIHNldHVwKG9wdGlvbnM6IElJbnRlcm5hbExvZ2dlck9wdGlvbnMpOiBQcm9taXNlPHZvaWQ+IHtcblx0XHR0aGlzLl9taW5Mb2dMZXZlbCA9IG9wdGlvbnMuY29uc29sZU1pbkxvZ0xldmVsO1xuXHRcdHRoaXMuX3ByZXBlbmRUaW1lc3RhbXAgPSBvcHRpb25zLnByZXBlbmRUaW1lc3RhbXA7XG5cblx0XHQvLyBPcGVuIGEgbG9nIGZpbGUgaW4gdGhlIHNwZWNpZmllZCBsb2NhdGlvbi4gT3ZlcndyaXR0ZW4gb24gZWFjaCBydW4uXG5cdFx0aWYgKG9wdGlvbnMubG9nRmlsZVBhdGgpIHtcblx0XHRcdGlmICghcGF0aC5pc0Fic29sdXRlKG9wdGlvbnMubG9nRmlsZVBhdGgpKSB7XG5cdFx0XHRcdHRoaXMubG9nKGBsb2dGaWxlUGF0aCBtdXN0IGJlIGFuIGFic29sdXRlIHBhdGg6ICR7b3B0aW9ucy5sb2dGaWxlUGF0aH1gLCBMb2dMZXZlbC5FcnJvcik7XG5cdFx0XHR9IGVsc2Uge1xuXHRcdFx0XHRjb25zdCBoYW5kbGVFcnJvciA9IGVyciA9PiB0aGlzLnNlbmRMb2coYEVycm9yIGNyZWF0aW5nIGxvZyBmaWxlIGF0IHBhdGg6ICR7b3B0aW9ucy5sb2dGaWxlUGF0aH0uIEVycm9yOiAke2Vyci50b1N0cmluZygpfVxcbmAsIExvZ0xldmVsLkVycm9yKTtcblxuXHRcdFx0XHR0cnkge1xuXHRcdFx0XHRcdGF3YWl0IG1rZGlycFByb21pc2UocGF0aC5kaXJuYW1lKG9wdGlvbnMubG9nRmlsZVBhdGgpKTtcblx0XHRcdFx0XHR0aGlzLmxvZyhgVmVyYm9zZSBsb2dzIGFyZSB3cml0dGVuIHRvOlxcbmAsIExvZ0xldmVsLldhcm4pO1xuXHRcdFx0XHRcdHRoaXMubG9nKG9wdGlvbnMubG9nRmlsZVBhdGggKyAnXFxuJywgTG9nTGV2ZWwuV2Fybik7XG5cblx0XHRcdFx0XHR0aGlzLl9sb2dGaWxlU3RyZWFtID0gZnMuY3JlYXRlV3JpdGVTdHJlYW0ob3B0aW9ucy5sb2dGaWxlUGF0aCk7XG5cdFx0XHRcdFx0dGhpcy5sb2dEYXRlVGltZSgpO1xuXHRcdFx0XHRcdHRoaXMuc2V0dXBTaHV0ZG93bkxpc3RlbmVycygpO1xuXHRcdFx0XHRcdHRoaXMuX2xvZ0ZpbGVTdHJlYW0ub24oJ2Vycm9yJywgZXJyID0+IHtcblx0XHRcdFx0XHRcdGhhbmRsZUVycm9yKGVycik7XG5cdFx0XHRcdFx0fSk7XG5cdFx0XHRcdH0gY2F0Y2ggKGVycikge1xuXHRcdFx0XHRcdGhhbmRsZUVycm9yKGVycik7XG5cdFx0XHRcdH1cblx0XHRcdH1cblx0XHR9XG5cdH1cblxuXHRwcml2YXRlIGxvZ0RhdGVUaW1lKCk6IHZvaWQge1xuXHRcdGxldCBkID0gbmV3IERhdGUoKTtcblx0XHRsZXQgZGF0ZVN0cmluZyA9IGQuZ2V0VVRDRnVsbFllYXIoKSArICctJyArIGAke2QuZ2V0VVRDTW9udGgoKSArIDF9YCArICctJyArIGQuZ2V0VVRDRGF0ZSgpO1xuXHRcdGNvbnN0IHRpbWVBbmREYXRlU3RhbXAgPSBkYXRlU3RyaW5nICsgJywgJyArIGdldEZvcm1hdHRlZFRpbWVTdHJpbmcoKTtcblx0XHR0aGlzLmxvZyh0aW1lQW5kRGF0ZVN0YW1wICsgJ1xcbicsIExvZ0xldmVsLlZlcmJvc2UsIGZhbHNlKTtcblx0fVxuXG5cdHByaXZhdGUgc2V0dXBTaHV0ZG93bkxpc3RlbmVycygpOiB2b2lkIHtcblx0XHRwcm9jZXNzLmFkZExpc3RlbmVyKCdiZWZvcmVFeGl0JywgdGhpcy5iZWZvcmVFeGl0Q2FsbGJhY2spO1xuXHRcdHByb2Nlc3MuYWRkTGlzdGVuZXIoJ1NJR1RFUk0nLCB0aGlzLmRpc3Bvc2VDYWxsYmFjayk7XG5cdFx0cHJvY2Vzcy5hZGRMaXN0ZW5lcignU0lHSU5UJywgdGhpcy5kaXNwb3NlQ2FsbGJhY2spO1xuXHR9XG5cblx0cHJpdmF0ZSByZW1vdmVTaHV0ZG93bkxpc3RlbmVycygpOiB2b2lkIHtcblx0XHRwcm9jZXNzLnJlbW92ZUxpc3RlbmVyKCdiZWZvcmVFeGl0JywgdGhpcy5iZWZvcmVFeGl0Q2FsbGJhY2spO1xuXHRcdHByb2Nlc3MucmVtb3ZlTGlzdGVuZXIoJ1NJR1RFUk0nLCB0aGlzLmRpc3Bvc2VDYWxsYmFjayk7XG5cdFx0cHJvY2Vzcy5yZW1vdmVMaXN0ZW5lcignU0lHSU5UJywgdGhpcy5kaXNwb3NlQ2FsbGJhY2spO1xuXHR9XG5cblx0cHVibGljIGRpc3Bvc2UoKTogUHJvbWlzZTx2b2lkPiB7XG5cdFx0cmV0dXJuIG5ldyBQcm9taXNlKHJlc29sdmUgPT4ge1xuXHRcdFx0dGhpcy5yZW1vdmVTaHV0ZG93bkxpc3RlbmVycygpO1xuXHRcdFx0aWYgKHRoaXMuX2xvZ0ZpbGVTdHJlYW0pIHtcblx0XHRcdFx0dGhpcy5fbG9nRmlsZVN0cmVhbS5lbmQocmVzb2x2ZSk7XG5cdFx0XHRcdHRoaXMuX2xvZ0ZpbGVTdHJlYW0gPSBudWxsO1xuXHRcdFx0fSBlbHNlIHtcblx0XHRcdFx0cmVzb2x2ZSgpO1xuXHRcdFx0fVxuXHRcdH0pO1xuXHR9XG5cblx0cHVibGljIGxvZyhtc2c6IHN0cmluZywgbGV2ZWw6IExvZ0xldmVsLCBwcmVwZW5kVGltZXN0YW1wID0gdHJ1ZSk6IHZvaWQge1xuXHRcdGlmICh0aGlzLl9taW5Mb2dMZXZlbCA9PT0gTG9nTGV2ZWwuU3RvcCkge1xuXHRcdFx0cmV0dXJuO1xuXHRcdH1cblxuXHRcdGlmIChsZXZlbCA+PSB0aGlzLl9taW5Mb2dMZXZlbCkge1xuXHRcdFx0dGhpcy5zZW5kTG9nKG1zZywgbGV2ZWwpO1xuXHRcdH1cblxuXHRcdGlmICh0aGlzLl9sb2dUb0NvbnNvbGUpIHtcblx0XHRcdGNvbnN0IGxvZ0ZuID1cblx0XHRcdFx0bGV2ZWwgPT09IExvZ0xldmVsLkVycm9yID8gY29uc29sZS5lcnJvciA6XG5cdFx0XHRcdGxldmVsID09PSBMb2dMZXZlbC5XYXJuID8gY29uc29sZS53YXJuIDpcblx0XHRcdFx0bnVsbDtcblxuXHRcdFx0aWYgKGxvZ0ZuKSB7XG5cdFx0XHRcdGxvZ0ZuKHRyaW1MYXN0TmV3bGluZShtc2cpKTtcblx0XHRcdH1cblx0XHR9XG5cblx0XHQvLyBJZiBhbiBlcnJvciwgcHJlcGVuZCB3aXRoICdbRXJyb3JdJ1xuXHRcdGlmIChsZXZlbCA9PT0gTG9nTGV2ZWwuRXJyb3IpIHtcblx0XHRcdG1zZyA9IGBbJHtMb2dMZXZlbFtsZXZlbF19XSAke21zZ31gO1xuXHRcdH1cblxuXHRcdGlmICh0aGlzLl9wcmVwZW5kVGltZXN0YW1wICYmIHByZXBlbmRUaW1lc3RhbXApIHtcblx0XHRcdG1zZyA9ICdbJyArIGdldEZvcm1hdHRlZFRpbWVTdHJpbmcoKSArICddICcgKyBtc2c7XG5cdFx0fVxuXG5cdFx0aWYgKHRoaXMuX2xvZ0ZpbGVTdHJlYW0pIHtcblx0XHRcdHRoaXMuX2xvZ0ZpbGVTdHJlYW0ud3JpdGUobXNnKTtcblx0XHR9XG5cdH1cblxuXHRwcml2YXRlIHNlbmRMb2cobXNnOiBzdHJpbmcsIGxldmVsOiBMb2dMZXZlbCk6IHZvaWQge1xuXHRcdC8vIFRydW5jYXRlIGxvbmcgbWVzc2FnZXMsIHRoZXkgY2FuIGhhbmcgVlMgQ29kZVxuXHRcdGlmIChtc2cubGVuZ3RoID4gMTUwMCkge1xuXHRcdFx0Y29uc3QgZW5kc0luTmV3bGluZSA9ICEhbXNnLm1hdGNoKC8oXFxufFxcclxcbikkLyk7XG5cdFx0XHRtc2cgPSBtc2cuc3Vic3RyKDAsIDE1MDApICsgJ1suLi5dJztcblx0XHRcdGlmIChlbmRzSW5OZXdsaW5lKSB7XG5cdFx0XHRcdG1zZyA9IG1zZyArICdcXG4nO1xuXHRcdFx0fVxuXHRcdH1cblxuXHRcdGlmICh0aGlzLl9sb2dDYWxsYmFjaykge1xuXHRcdFx0Y29uc3QgZXZlbnQgPSBuZXcgTG9nT3V0cHV0RXZlbnQobXNnLCBsZXZlbCk7XG5cdFx0XHR0aGlzLl9sb2dDYWxsYmFjayhldmVudCk7XG5cdFx0fVxuXHR9XG59XG5cbmZ1bmN0aW9uIGdldEZvcm1hdHRlZFRpbWVTdHJpbmcoKTogc3RyaW5nIHtcblx0bGV0IGQgPSBuZXcgRGF0ZSgpO1xuXHRsZXQgaG91clN0cmluZyA9IF9wYWRaZXJvZXMoMiwgU3RyaW5nKGQuZ2V0VVRDSG91cnMoKSkpO1xuXHRsZXQgbWludXRlU3RyaW5nID0gX3BhZFplcm9lcygyLCBTdHJpbmcoZC5nZXRVVENNaW51dGVzKCkpKTtcblx0bGV0IHNlY29uZFN0cmluZyA9IF9wYWRaZXJvZXMoMiwgU3RyaW5nKGQuZ2V0VVRDU2Vjb25kcygpKSk7XG5cdGxldCBtaWxsaXNlY29uZFN0cmluZyA9IF9wYWRaZXJvZXMoMywgU3RyaW5nKGQuZ2V0VVRDTWlsbGlzZWNvbmRzKCkpKTtcblx0cmV0dXJuIGhvdXJTdHJpbmcgKyAnOicgKyBtaW51dGVTdHJpbmcgKyAnOicgKyBzZWNvbmRTdHJpbmcgKyAnLicgKyBtaWxsaXNlY29uZFN0cmluZyArICcgVVRDJztcbn1cblxuZnVuY3Rpb24gX3BhZFplcm9lcyhtaW5EZXNpcmVkTGVuZ3RoOiBudW1iZXIsIG51bWJlclRvUGFkOiBzdHJpbmcpOiBzdHJpbmcge1xuXHRpZiAobnVtYmVyVG9QYWQubGVuZ3RoID49IG1pbkRlc2lyZWRMZW5ndGgpIHtcblx0XHRyZXR1cm4gbnVtYmVyVG9QYWQ7XG5cdH0gZWxzZSB7XG5cdFx0cmV0dXJuIFN0cmluZygnMCcucmVwZWF0KG1pbkRlc2lyZWRMZW5ndGgpICsgbnVtYmVyVG9QYWQpLnNsaWNlKC1taW5EZXNpcmVkTGVuZ3RoKTtcblx0fVxufVxuIl19