"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ElectronMainMenuFactory = void 0;
/* eslint-disable @typescript-eslint/no-explicit-any */
const electron = require("../../../shared/electron");
const inversify_1 = require("inversify");
const common_1 = require("../../common");
const browser_1 = require("../../browser");
const context_key_service_1 = require("../../browser/context-key-service");
const debounce = require("lodash.debounce");
const context_menu_context_1 = require("../../browser/menu/context-menu-context");
const theia_dock_panel_1 = require("../../browser/shell/theia-dock-panel");
let ElectronMainMenuFactory = class ElectronMainMenuFactory {
    constructor(commandRegistry, preferencesService, menuProvider, keybindingRegistry) {
        this.commandRegistry = commandRegistry;
        this.preferencesService = preferencesService;
        this.menuProvider = menuProvider;
        this.keybindingRegistry = keybindingRegistry;
        this._toggledCommands = new Set();
        preferencesService.onPreferenceChanged(debounce(e => {
            if (e.preferenceName === 'window.menuBarVisibility') {
                this.setMenuBar();
            }
            if (this._menu) {
                for (const item of this._toggledCommands) {
                    this._menu.getMenuItemById(item).checked = this.commandRegistry.isToggled(item);
                }
                electron.remote.getCurrentWindow().setMenu(this._menu);
            }
        }, 10));
        keybindingRegistry.onKeybindingsChanged(() => {
            this.setMenuBar();
        });
    }
    async setMenuBar() {
        await this.preferencesService.ready;
        const createdMenuBar = this.createMenuBar();
        if (common_1.isOSX) {
            electron.remote.Menu.setApplicationMenu(createdMenuBar);
        }
        else {
            electron.remote.getCurrentWindow().setMenu(createdMenuBar);
        }
    }
    createMenuBar() {
        const preference = this.preferencesService.get('window.menuBarVisibility') || 'classic';
        const maxWidget = document.getElementsByClassName(theia_dock_panel_1.MAXIMIZED_CLASS);
        if (preference === 'visible' || (preference === 'classic' && maxWidget.length === 0)) {
            const menuModel = this.menuProvider.getMenu(common_1.MAIN_MENU_BAR);
            const template = this.fillMenuTemplate([], menuModel);
            if (common_1.isOSX) {
                template.unshift(this.createOSXMenu());
            }
            const menu = electron.remote.Menu.buildFromTemplate(template);
            this._menu = menu;
            return this._menu;
        }
        this._menu = undefined;
        // eslint-disable-next-line no-null/no-null
        return null;
    }
    createContextMenu(menuPath, args) {
        const menuModel = this.menuProvider.getMenu(menuPath);
        const template = this.fillMenuTemplate([], menuModel, args, { showDisabled: false });
        return electron.remote.Menu.buildFromTemplate(template);
    }
    fillMenuTemplate(items, menuModel, args = [], options) {
        const showDisabled = ((options === null || options === void 0 ? void 0 : options.showDisabled) === undefined) ? true : options === null || options === void 0 ? void 0 : options.showDisabled;
        for (const menu of menuModel.children) {
            if (menu instanceof common_1.CompositeMenuNode) {
                if (menu.children.length > 0) {
                    // do not render empty nodes
                    if (menu.isSubmenu) { // submenu node
                        const submenu = this.fillMenuTemplate([], menu, args, options);
                        if (submenu.length === 0) {
                            continue;
                        }
                        items.push({
                            label: menu.label,
                            submenu
                        });
                    }
                    else { // group node
                        // process children
                        const submenu = this.fillMenuTemplate([], menu, args, options);
                        if (submenu.length === 0) {
                            continue;
                        }
                        if (items.length > 0) {
                            // do not put a separator above the first group
                            items.push({
                                type: 'separator'
                            });
                        }
                        // render children
                        items.push(...submenu);
                    }
                }
            }
            else if (menu instanceof common_1.ActionMenuNode) {
                const node = menu.altNode && this.context.altPressed ? menu.altNode : menu;
                const commandId = node.action.commandId;
                // That is only a sanity check at application startup.
                if (!this.commandRegistry.getCommand(commandId)) {
                    console.debug(`Skipping menu item with missing command: "${commandId}".`);
                    continue;
                }
                if (!this.commandRegistry.isVisible(commandId, ...args)
                    || (!!node.action.when && !this.contextKeyService.match(node.action.when))) {
                    continue;
                }
                // We should omit rendering context-menu items which are disabled.
                if (!showDisabled && !this.commandRegistry.isEnabled(commandId, ...args)) {
                    continue;
                }
                const bindings = this.keybindingRegistry.getKeybindingsForCommand(commandId);
                let accelerator;
                /* Only consider the first keybinding. */
                if (bindings.length > 0) {
                    const binding = bindings[0];
                    accelerator = this.acceleratorFor(binding);
                }
                const menuItem = {
                    id: node.id,
                    label: node.label,
                    type: this.commandRegistry.getToggledHandler(commandId, ...args) ? 'checkbox' : 'normal',
                    checked: this.commandRegistry.isToggled(commandId, ...args),
                    enabled: true,
                    visible: true,
                    accelerator,
                    click: () => this.execute(commandId, args)
                };
                if (common_1.isOSX) {
                    const role = this.roleFor(node.id);
                    if (role) {
                        menuItem.role = role;
                        delete menuItem.click;
                    }
                }
                items.push(menuItem);
                if (this.commandRegistry.getToggledHandler(commandId, ...args)) {
                    this._toggledCommands.add(commandId);
                }
            }
            else {
                items.push(...this.handleDefault(menu, args, options));
            }
        }
        return items;
    }
    handleDefault(menuNode, args = [], options) {
        return [];
    }
    /**
     * Return a user visible representation of a keybinding.
     */
    acceleratorFor(keybinding) {
        const bindingKeySequence = this.keybindingRegistry.resolveKeybinding(keybinding);
        // FIXME see https://github.com/electron/electron/issues/11740
        // Key Sequences can't be represented properly in the electron menu.
        //
        // We can do what VS Code does, and append the chords as a suffix to the menu label.
        // https://github.com/eclipse-theia/theia/issues/1199#issuecomment-430909480
        if (bindingKeySequence.length > 1) {
            return '';
        }
        const keyCode = bindingKeySequence[0];
        return this.keybindingRegistry.acceleratorForKeyCode(keyCode, '+');
    }
    roleFor(id) {
        let role;
        switch (id) {
            case browser_1.CommonCommands.UNDO.id:
                role = 'undo';
                break;
            case browser_1.CommonCommands.REDO.id:
                role = 'redo';
                break;
            case browser_1.CommonCommands.CUT.id:
                role = 'cut';
                break;
            case browser_1.CommonCommands.COPY.id:
                role = 'copy';
                break;
            case browser_1.CommonCommands.PASTE.id:
                role = 'paste';
                break;
            case browser_1.CommonCommands.SELECT_ALL.id:
                role = 'selectAll';
                break;
            default:
                break;
        }
        return role;
    }
    async execute(command, args) {
        try {
            // This is workaround for https://github.com/eclipse-theia/theia/issues/446.
            // Electron menus do not update based on the `isEnabled`, `isVisible` property of the command.
            // We need to check if we can execute it.
            if (this.commandRegistry.isEnabled(command, ...args)) {
                await this.commandRegistry.executeCommand(command, ...args);
                if (this._menu && this.commandRegistry.isVisible(command, ...args)) {
                    this._menu.getMenuItemById(command).checked = this.commandRegistry.isToggled(command, ...args);
                    electron.remote.getCurrentWindow().setMenu(this._menu);
                }
            }
        }
        catch (_a) {
            // no-op
        }
    }
    createOSXMenu() {
        return {
            label: 'Theia',
            submenu: [
                {
                    role: 'about'
                },
                {
                    type: 'separator'
                },
                {
                    role: 'services',
                    submenu: []
                },
                {
                    type: 'separator'
                },
                {
                    role: 'hide'
                },
                {
                    role: 'hideOthers'
                },
                {
                    role: 'unhide'
                },
                {
                    type: 'separator'
                },
                {
                    role: 'quit'
                }
            ]
        };
    }
};
__decorate([
    inversify_1.inject(context_key_service_1.ContextKeyService),
    __metadata("design:type", context_key_service_1.ContextKeyService)
], ElectronMainMenuFactory.prototype, "contextKeyService", void 0);
__decorate([
    inversify_1.inject(context_menu_context_1.ContextMenuContext),
    __metadata("design:type", context_menu_context_1.ContextMenuContext)
], ElectronMainMenuFactory.prototype, "context", void 0);
ElectronMainMenuFactory = __decorate([
    inversify_1.injectable(),
    __param(0, inversify_1.inject(common_1.CommandRegistry)),
    __param(1, inversify_1.inject(browser_1.PreferenceService)),
    __param(2, inversify_1.inject(common_1.MenuModelRegistry)),
    __param(3, inversify_1.inject(browser_1.KeybindingRegistry)),
    __metadata("design:paramtypes", [common_1.CommandRegistry, Object, common_1.MenuModelRegistry,
        browser_1.KeybindingRegistry])
], ElectronMainMenuFactory);
exports.ElectronMainMenuFactory = ElectronMainMenuFactory;
//# sourceMappingURL=electron-main-menu-factory.js.map