"use strict";
/********************************************************************************
 * Copyright (C) 2018 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.PreferenceRegistryExtImpl = void 0;
/* eslint-disable @typescript-eslint/no-explicit-any */
const event_1 = require("@theia/core/lib/common/event");
const plugin_api_rpc_1 = require("../common/plugin-api-rpc");
const types_1 = require("../common/types");
const configuration_1 = require("./preferences/configuration");
const cloneDeep = require("lodash.clonedeep");
const injectionRe = /\b__proto__\b|\bconstructor\.prototype\b/;
var ConfigurationTarget;
(function (ConfigurationTarget) {
    ConfigurationTarget[ConfigurationTarget["Global"] = 1] = "Global";
    ConfigurationTarget[ConfigurationTarget["Workspace"] = 2] = "Workspace";
    ConfigurationTarget[ConfigurationTarget["WorkspaceFolder"] = 3] = "WorkspaceFolder";
})(ConfigurationTarget || (ConfigurationTarget = {}));
var PreferenceScope;
(function (PreferenceScope) {
    PreferenceScope[PreferenceScope["Default"] = 0] = "Default";
    PreferenceScope[PreferenceScope["User"] = 1] = "User";
    PreferenceScope[PreferenceScope["Workspace"] = 2] = "Workspace";
    PreferenceScope[PreferenceScope["Folder"] = 3] = "Folder";
})(PreferenceScope || (PreferenceScope = {}));
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function lookUp(tree, key) {
    if (!key) {
        return;
    }
    const parts = key.split('.');
    let node = tree;
    for (let i = 0; node && i < parts.length; i++) {
        node = node[parts[i]];
    }
    return node;
}
class PreferenceRegistryExtImpl {
    constructor(rpc, workspace) {
        this.workspace = workspace;
        this._onDidChangeConfiguration = new event_1.Emitter();
        this.onDidChangeConfiguration = this._onDidChangeConfiguration.event;
        this.OVERRIDE_PROPERTY = '\\[(.*)\\]$';
        this.OVERRIDE_PROPERTY_PATTERN = new RegExp(this.OVERRIDE_PROPERTY);
        this.proxy = rpc.getProxy(plugin_api_rpc_1.PLUGIN_RPC_CONTEXT.PREFERENCE_REGISTRY_MAIN);
    }
    init(data) {
        this._preferences = this.parse(data);
    }
    $acceptConfigurationChanged(data, eventData) {
        this.init(data);
        this._onDidChangeConfiguration.fire(this.toConfigurationChangeEvent(eventData));
    }
    getConfiguration(section, resource, extensionId) {
        resource = resource === null ? undefined : resource;
        const preferences = this.toReadonlyValue(section
            ? lookUp(this._preferences.getValue(undefined, this.workspace, resource), section)
            : this._preferences.getValue(undefined, this.workspace, resource));
        const configuration = {
            has(key) {
                return typeof lookUp(preferences, key) !== 'undefined';
            },
            get: (key, defaultValue) => {
                const result = lookUp(preferences, key);
                if (typeof result === 'undefined') {
                    return defaultValue;
                }
                else {
                    let clonedConfig = undefined;
                    const cloneOnWriteProxy = (target, accessor) => {
                        let clonedTarget = undefined;
                        const cloneTarget = () => {
                            clonedConfig = clonedConfig ? clonedConfig : cloneDeep(preferences);
                            clonedTarget = clonedTarget ? clonedTarget : lookUp(clonedConfig, accessor);
                        };
                        if (!types_1.isObject(target)) {
                            return target;
                        }
                        return new Proxy(target, {
                            get: (targ, prop) => {
                                if (typeof prop === 'string' && prop.toLowerCase() === 'tojson') {
                                    cloneTarget();
                                    return () => clonedTarget;
                                }
                                if (clonedConfig) {
                                    clonedTarget = clonedTarget ? clonedTarget : lookUp(clonedConfig, accessor);
                                    return clonedTarget[prop];
                                }
                                const res = targ[prop];
                                if (typeof prop === 'string') {
                                    return cloneOnWriteProxy(res, `${accessor}.${prop}`);
                                }
                                return res;
                            },
                            set: (targ, prop, val) => {
                                cloneTarget();
                                clonedTarget[prop] = val;
                                return true;
                            },
                            deleteProperty: (targ, prop) => {
                                cloneTarget();
                                delete clonedTarget[prop];
                                return true;
                            },
                            defineProperty: (targ, prop, descr) => {
                                cloneTarget();
                                Object.defineProperty(clonedTarget, prop, descr);
                                return true;
                            }
                        });
                    };
                    return cloneOnWriteProxy(result, key);
                }
            },
            update: (key, value, arg) => {
                key = section ? `${section}.${key}` : key;
                const resourceStr = resource ? resource.toString() : undefined;
                if (typeof value !== 'undefined') {
                    return this.proxy.$updateConfigurationOption(arg, key, value, resourceStr);
                }
                else {
                    return this.proxy.$removeConfigurationOption(arg, key, resourceStr);
                }
            },
            inspect: (key) => {
                key = section ? `${section}.${key}` : key;
                resource = resource === null ? undefined : resource;
                const result = cloneDeep(this._preferences.inspect(key, this.workspace, resource));
                if (!result) {
                    return undefined;
                }
                const configInspect = { key };
                if (typeof result.default !== 'undefined') {
                    configInspect.defaultValue = result.default;
                }
                if (typeof result.user !== 'undefined') {
                    configInspect.globalValue = result.user;
                }
                if (typeof result.workspace !== 'undefined') {
                    configInspect.workspaceValue = result.workspace;
                }
                if (typeof result.workspaceFolder !== 'undefined') {
                    configInspect.workspaceFolderValue = result.workspaceFolder;
                }
                return configInspect;
            }
        };
        if (typeof preferences === 'object') {
            types_1.mixin(configuration, preferences, false);
        }
        return Object.freeze(configuration);
    }
    toReadonlyValue(data) {
        const readonlyProxy = (target) => types_1.isObject(target)
            ? new Proxy(target, {
                get: (targ, prop) => readonlyProxy(targ[prop]),
                set: (targ, prop, val) => {
                    throw new Error(`TypeError: Cannot assign to read only property '${prop}' of object`);
                },
                deleteProperty: (targ, prop) => {
                    throw new Error(`TypeError: Cannot delete read only property '${prop}' of object`);
                },
                defineProperty: (targ, prop) => {
                    throw new Error(`TypeError: Cannot define property '${prop}' of a readonly object`);
                },
                setPrototypeOf: (targ) => {
                    throw new Error('TypeError: Cannot set prototype for a readonly object');
                },
                isExtensible: () => false,
                preventExtensions: () => true
            })
            : target;
        return readonlyProxy(data);
    }
    parse(data) {
        const defaultConfiguration = this.getConfigurationModel(data[PreferenceScope.Default]);
        const userConfiguration = this.getConfigurationModel(data[PreferenceScope.User]);
        const workspaceConfiguration = this.getConfigurationModel(data[PreferenceScope.Workspace]);
        const folderConfigurations = {};
        Object.keys(data[PreferenceScope.Folder]).forEach(resource => {
            folderConfigurations[resource] = this.getConfigurationModel(data[PreferenceScope.Folder][resource]);
        });
        return new configuration_1.Configuration(defaultConfiguration, userConfiguration, workspaceConfiguration, folderConfigurations);
    }
    getConfigurationModel(data) {
        if (!data) {
            return new configuration_1.ConfigurationModel();
        }
        return new configuration_1.ConfigurationModel(this.parseConfigurationData(data), Object.keys(data));
    }
    parseConfigurationData(data) {
        return Object.keys(data).reduce((result, key) => {
            if (injectionRe.test(key)) {
                return result;
            }
            const parts = key.split('.');
            let branch = result;
            for (let i = 0; i < parts.length; i++) {
                if (i === parts.length - 1) {
                    branch[parts[i]] = data[key];
                    continue;
                }
                if (!branch[parts[i]]) {
                    branch[parts[i]] = Object.create(null);
                }
                branch = branch[parts[i]];
                // overridden properties should be transformed into
                // "[overridden_identifier]" : {
                //              "property1" : "value1"
                //              "property2" : "value2"
                //  }
                if (i === 0 && this.OVERRIDE_PROPERTY_PATTERN.test(parts[i])) {
                    branch[key.substring(parts[0].length + 1)] = data[key];
                    break;
                }
            }
            return result;
        }, Object.create(null));
    }
    toConfigurationChangeEvent(eventData) {
        return Object.freeze({
            affectsConfiguration: (section, uri) => {
                // TODO respect uri
                // TODO respect scopes shadowing
                for (const change of eventData) {
                    const tree = change.preferenceName
                        .split('.')
                        .reverse()
                        .reduce((prevValue, curValue) => ({ [curValue]: prevValue }), change.newValue);
                    return typeof lookUp(tree, section) !== 'undefined';
                }
                return false;
            }
        });
    }
}
exports.PreferenceRegistryExtImpl = PreferenceRegistryExtImpl;
//# sourceMappingURL=preference-registry.js.map