"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.IPCConnectionProvider = void 0;
const cp = require("child_process");
const path = require("path");
const inversify_1 = require("inversify");
const vscode_ws_jsonrpc_1 = require("vscode-ws-jsonrpc");
const common_1 = require("../../common");
const ipc_protocol_1 = require("./ipc-protocol");
let IPCConnectionProvider = class IPCConnectionProvider {
    listen(options, acceptor) {
        return this.doListen(Object.assign({ logger: this.logger, args: [] }, options), acceptor);
    }
    doListen(options, acceptor) {
        const childProcess = this.fork(options);
        const connection = this.createConnection(childProcess, options);
        const toStop = new common_1.DisposableCollection();
        const toCancelStop = toStop.push(common_1.Disposable.create(() => childProcess.kill()));
        const errorHandler = options.errorHandler;
        if (errorHandler) {
            connection.onError((e) => {
                if (errorHandler.shouldStop(e[0], e[1], e[2])) {
                    toStop.dispose();
                }
            });
            connection.onClose(() => {
                if (toStop.disposed) {
                    return;
                }
                if (errorHandler.shouldRestart()) {
                    toCancelStop.dispose();
                    toStop.push(this.doListen(options, acceptor));
                }
            });
        }
        acceptor(connection);
        return toStop;
    }
    createConnection(childProcess, options) {
        const reader = new vscode_ws_jsonrpc_1.IPCMessageReader(childProcess);
        const writer = new vscode_ws_jsonrpc_1.IPCMessageWriter(childProcess);
        const connection = vscode_ws_jsonrpc_1.createMessageConnection(reader, writer, {
            error: (message) => this.logger.error(`[${options.serverName}: ${childProcess.pid}] ${message}`),
            warn: (message) => this.logger.warn(`[${options.serverName}: ${childProcess.pid}] ${message}`),
            info: (message) => this.logger.info(`[${options.serverName}: ${childProcess.pid}] ${message}`),
            log: (message) => this.logger.info(`[${options.serverName}: ${childProcess.pid}] ${message}`)
        });
        const traceVerbosity = this.logger.isDebug() ? vscode_ws_jsonrpc_1.Trace.Verbose : vscode_ws_jsonrpc_1.Trace.Off;
        connection.trace(traceVerbosity, {
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            log: (message, data) => this.logger.debug(`[${options.serverName}: ${childProcess.pid}] ${message}` + (typeof data === 'string' ? ' ' + data : ''))
        });
        return connection;
    }
    fork(options) {
        const forkOptions = {
            silent: true,
            env: ipc_protocol_1.createIpcEnv(options),
            execArgv: []
        };
        const inspectArgPrefix = `--${options.serverName}-inspect`;
        const inspectArg = process.argv.find(v => v.startsWith(inspectArgPrefix));
        if (inspectArg !== undefined) {
            forkOptions.execArgv = ['--nolazy', `--inspect${inspectArg.substr(inspectArgPrefix.length)}`];
        }
        const childProcess = cp.fork(path.join(__dirname, 'ipc-bootstrap'), options.args, forkOptions);
        childProcess.stdout.on('data', data => this.logger.info(`[${options.serverName}: ${childProcess.pid}] ${data.toString().trim()}`));
        childProcess.stderr.on('data', data => this.logger.error(`[${options.serverName}: ${childProcess.pid}] ${data.toString().trim()}`));
        this.logger.debug(`[${options.serverName}: ${childProcess.pid}] IPC started`);
        childProcess.once('exit', () => this.logger.debug(`[${options.serverName}: ${childProcess.pid}] IPC exited`));
        return childProcess;
    }
};
__decorate([
    inversify_1.inject(common_1.ILogger),
    __metadata("design:type", Object)
], IPCConnectionProvider.prototype, "logger", void 0);
IPCConnectionProvider = __decorate([
    inversify_1.injectable()
], IPCConnectionProvider);
exports.IPCConnectionProvider = IPCConnectionProvider;
//# sourceMappingURL=ipc-connection-provider.js.map