"use strict";
/********************************************************************************
 * Copyright (C) 2021 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.MarkdownRenderer = void 0;
const DOMPurify = require("dompurify");
const markdownit = require("markdown-it");
class MarkdownRenderer {
    constructor(engine) {
        this.callbacks = new Map();
        this.engine = engine !== null && engine !== void 0 ? engine : markdownit();
    }
    /**
     * Adds a modification callback that is applied to every element with the specified tag after rendering to HTML.
     *
     * @param tag The tag that this modification applies to.
     * @param callback The modification to apply on every selected rendered element. Can either modify the element in place or return a new element.
     */
    modify(tag, callback) {
        if (this.callbacks.has(tag)) {
            this.callbacks.get(tag).push(callback);
        }
        else {
            this.callbacks.set(tag, [callback]);
        }
        return this;
    }
    render(markdown) {
        return this.renderInternal(this.engine.render(markdown));
    }
    renderInline(markdown) {
        return this.renderInternal(this.engine.renderInline(markdown));
    }
    renderInternal(renderedHtml) {
        const div = this.sanitizeHtml(renderedHtml);
        for (const [tag, calls] of this.callbacks) {
            for (const callback of calls) {
                const elements = Array.from(div.getElementsByTagName(tag));
                for (const element of elements) {
                    const result = callback(element);
                    if (result) {
                        const parent = element.parentElement;
                        if (parent) {
                            parent.replaceChild(result, element);
                        }
                    }
                }
            }
        }
        return div;
    }
    sanitizeHtml(html) {
        const div = document.createElement('div');
        div.innerHTML = DOMPurify.sanitize(html);
        return div;
    }
}
exports.MarkdownRenderer = MarkdownRenderer;
//# sourceMappingURL=markdown-renderer.js.map