"use strict";
/********************************************************************************
 * Copyright (C) 2018 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var MonacoEditorProvider_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonacoEditorProvider = exports.MonacoEditorFactory = void 0;
/* eslint-disable @typescript-eslint/no-explicit-any */
const uri_1 = require("@theia/core/lib/common/uri");
const browser_1 = require("@theia/editor/lib/browser");
const diff_uris_1 = require("@theia/core/lib/browser/diff-uris");
const inversify_1 = require("@theia/core/shared/inversify");
const common_1 = require("@theia/core/lib/common");
const vscode_languageserver_protocol_1 = require("@theia/core/shared/vscode-languageserver-protocol");
const monaco_command_service_1 = require("./monaco-command-service");
const monaco_context_menu_1 = require("./monaco-context-menu");
const monaco_diff_editor_1 = require("./monaco-diff-editor");
const monaco_diff_navigator_factory_1 = require("./monaco-diff-navigator-factory");
const monaco_editor_1 = require("./monaco-editor");
const monaco_editor_model_1 = require("./monaco-editor-model");
const monaco_editor_service_1 = require("./monaco-editor-service");
const monaco_text_model_service_1 = require("./monaco-text-model-service");
const monaco_workspace_1 = require("./monaco-workspace");
const monaco_bulk_edit_service_1 = require("./monaco-bulk-edit-service");
const application_protocol_1 = require("@theia/core/lib/common/application-protocol");
const core_1 = require("@theia/core");
const browser_2 = require("@theia/core/lib/browser");
const monaco_resolved_keybinding_1 = require("./monaco-resolved-keybinding");
const monaco_to_protocol_converter_1 = require("./monaco-to-protocol-converter");
const protocol_to_monaco_converter_1 = require("./protocol-to-monaco-converter");
const browser_3 = require("@theia/filesystem/lib/browser");
const monaco_quick_input_service_1 = require("./monaco-quick-input-service");
exports.MonacoEditorFactory = Symbol('MonacoEditorFactory');
let MonacoEditorProvider = MonacoEditorProvider_1 = class MonacoEditorProvider {
    constructor(codeEditorService, textModelService, contextMenuService, m2p, p2m, workspace, commandServiceFactory, editorPreferences, diffNavigatorFactory, 
    /** @deprecated since 1.6.0 */
    applicationServer, contextKeyService) {
        this.codeEditorService = codeEditorService;
        this.textModelService = textModelService;
        this.contextMenuService = contextMenuService;
        this.m2p = m2p;
        this.p2m = p2m;
        this.workspace = workspace;
        this.commandServiceFactory = commandServiceFactory;
        this.editorPreferences = editorPreferences;
        this.diffNavigatorFactory = diffNavigatorFactory;
        this.applicationServer = applicationServer;
        this.contextKeyService = contextKeyService;
        const staticServices = monaco.services.StaticServices;
        const init = staticServices.init.bind(monaco.services.StaticServices);
        monaco.services.StaticServices.init = o => {
            const result = init(o);
            result[0].set(monaco.services.ICodeEditorService, codeEditorService);
            return result;
        };
    }
    /**
     * Returns the last focused MonacoEditor.
     * It takes into account inline editors as well.
     * If you are interested only in standalone editors then use `MonacoEditor.getCurrent(EditorManager)`
     */
    get current() {
        return this._current;
    }
    async getModel(uri, toDispose) {
        const reference = await this.textModelService.createModelReference(uri);
        // if document is invalid makes sure that all events from underlying resource are processed before throwing invalid model
        if (!reference.object.valid) {
            await reference.object.sync();
        }
        if (!reference.object.valid) {
            reference.dispose();
            throw Object.assign(new Error(`'${uri.toString()}' is invalid`), { code: 'MODEL_IS_INVALID' });
        }
        toDispose.push(reference);
        return reference.object;
    }
    async get(uri) {
        await this.editorPreferences.ready;
        return this.doCreateEditor(uri, (override, toDispose) => this.createEditor(uri, override, toDispose));
    }
    async doCreateEditor(uri, factory) {
        const commandService = this.commandServiceFactory();
        const domNode = document.createElement('div');
        const contextKeyService = this.contextKeyService.createScoped(domNode);
        const { codeEditorService, textModelService, contextMenuService } = this;
        const IWorkspaceEditService = this.bulkEditService;
        const toDispose = new common_1.DisposableCollection(commandService);
        const openerService = new monaco.services.OpenerService(codeEditorService, commandService);
        openerService.registerOpener({
            open: (u, options) => this.interceptOpen(u, options)
        });
        const editor = await factory({
            codeEditorService,
            textModelService,
            contextMenuService,
            commandService,
            IWorkspaceEditService,
            contextKeyService,
            openerService,
            quickInputService: this.quickInputService
        }, toDispose);
        editor.onDispose(() => toDispose.dispose());
        this.suppressMonacoKeybindingListener(editor);
        this.injectKeybindingResolver(editor);
        const standaloneCommandService = new monaco.services.StandaloneCommandService(editor.instantiationService);
        commandService.setDelegate(standaloneCommandService);
        toDispose.push(this.installReferencesController(editor));
        toDispose.push(editor.onFocusChanged(focused => {
            if (focused) {
                this._current = editor;
            }
        }));
        toDispose.push(common_1.Disposable.create(() => {
            if (this._current === editor) {
                this._current = undefined;
            }
        }));
        return editor;
    }
    /**
     * Intercept internal Monaco open calls and delegate to OpenerService.
     */
    async interceptOpen(monacoUri, monacoOptions) {
        let options = undefined;
        if (monacoOptions) {
            if ('openToSide' in monacoOptions && monacoOptions.openToSide) {
                options = Object.assign(options || {}, {
                    widgetOptions: {
                        mode: 'split-right'
                    }
                });
            }
            if ('openExternal' in monacoOptions && monacoOptions.openExternal) {
                options = Object.assign(options || {}, {
                    openExternal: true
                });
            }
        }
        const uri = new uri_1.default(monacoUri.toString());
        try {
            await browser_2.open(this.openerService, uri, options);
            return true;
        }
        catch (e) {
            console.error(`Fail to open '${uri.toString()}':`, e);
            return false;
        }
    }
    /**
     * Suppresses Monaco keydown listener to avoid triggering default Monaco keybindings
     * if they are overridden by a user. Monaco keybindings should be registered as Theia keybindings
     * to allow a user to customize them.
     */
    suppressMonacoKeybindingListener(editor) {
        let keydownListener;
        const keybindingService = editor.getControl()._standaloneKeybindingService;
        for (const listener of keybindingService._store._toDispose) {
            if ('_type' in listener && listener['_type'] === 'keydown') {
                keydownListener = listener;
                break;
            }
        }
        if (keydownListener) {
            keydownListener.dispose();
        }
    }
    injectKeybindingResolver(editor) {
        const keybindingService = editor.getControl()._standaloneKeybindingService;
        keybindingService.resolveKeybinding = keybinding => [new monaco_resolved_keybinding_1.MonacoResolvedKeybinding(monaco_resolved_keybinding_1.MonacoResolvedKeybinding.keySequence(keybinding), this.keybindingRegistry)];
        keybindingService.resolveKeyboardEvent = keyboardEvent => {
            const keybinding = new monaco.keybindings.SimpleKeybinding(keyboardEvent.ctrlKey, keyboardEvent.shiftKey, keyboardEvent.altKey, keyboardEvent.metaKey, keyboardEvent.keyCode).toChord();
            return new monaco_resolved_keybinding_1.MonacoResolvedKeybinding(monaco_resolved_keybinding_1.MonacoResolvedKeybinding.keySequence(keybinding), this.keybindingRegistry);
        };
    }
    createEditor(uri, override, toDispose) {
        if (diff_uris_1.DiffUris.isDiffUri(uri)) {
            return this.createMonacoDiffEditor(uri, override, toDispose);
        }
        return this.createMonacoEditor(uri, override, toDispose);
    }
    get preferencePrefixes() {
        return ['editor.'];
    }
    async createMonacoEditor(uri, override, toDispose) {
        const model = await this.getModel(uri, toDispose);
        const options = this.createMonacoEditorOptions(model);
        const factory = this.factories.getContributions().find(({ scheme }) => uri.scheme === scheme);
        const editor = factory
            ? factory.create(model, options, override)
            : new monaco_editor_1.MonacoEditor(uri, model, document.createElement('div'), this.services, options, override);
        toDispose.push(this.editorPreferences.onPreferenceChanged(event => {
            if (event.affects(uri.toString(), model.languageId)) {
                this.updateMonacoEditorOptions(editor, event);
            }
        }));
        toDispose.push(editor.onLanguageChanged(() => this.updateMonacoEditorOptions(editor)));
        editor.document.onWillSaveModel(event => event.waitUntil(this.formatOnSave(editor, event)));
        return editor;
    }
    createMonacoEditorOptions(model) {
        const options = this.createOptions(this.preferencePrefixes, model.uri, model.languageId);
        options.model = model.textEditorModel;
        options.readOnly = model.readOnly;
        options.lineNumbersMinChars = model.lineNumbersMinChars;
        return options;
    }
    updateMonacoEditorOptions(editor, event) {
        if (event) {
            const preferenceName = event.preferenceName;
            const overrideIdentifier = editor.document.languageId;
            const newValue = this.editorPreferences.get({ preferenceName, overrideIdentifier }, undefined, editor.uri.toString());
            editor.getControl().updateOptions(this.setOption(preferenceName, newValue, this.preferencePrefixes));
        }
        else {
            const options = this.createMonacoEditorOptions(editor.document);
            delete options.model;
            editor.getControl().updateOptions(options);
        }
    }
    shouldFormat(editor, event) {
        var _a;
        if (event.reason !== vscode_languageserver_protocol_1.TextDocumentSaveReason.Manual) {
            return false;
        }
        if ((_a = event.options) === null || _a === void 0 ? void 0 : _a.formatType) {
            switch (event.options.formatType) {
                case 1 /* ON */: return true;
                case 2 /* OFF */: return false;
                case 3 /* DIRTY */: return editor.document.dirty;
            }
        }
        return true;
    }
    async formatOnSave(editor, event) {
        if (!this.shouldFormat(editor, event)) {
            return [];
        }
        const overrideIdentifier = editor.document.languageId;
        const uri = editor.uri.toString();
        const formatOnSave = this.editorPreferences.get({ preferenceName: 'editor.formatOnSave', overrideIdentifier }, undefined, uri);
        if (formatOnSave) {
            const formatOnSaveTimeout = this.editorPreferences.get({ preferenceName: 'editor.formatOnSaveTimeout', overrideIdentifier }, undefined, uri);
            await Promise.race([
                new Promise((_, reject) => setTimeout(() => reject(new Error(`Aborted format on save after ${formatOnSaveTimeout}ms`)), formatOnSaveTimeout)),
                editor.runAction('editor.action.formatDocument')
            ]);
        }
        const shouldRemoveWhiteSpace = this.filePreferences.get({ preferenceName: 'files.trimTrailingWhitespace', overrideIdentifier }, undefined, uri);
        if (shouldRemoveWhiteSpace) {
            await editor.runAction('editor.action.trimTrailingWhitespace');
        }
        return [];
    }
    get diffPreferencePrefixes() {
        return [...this.preferencePrefixes, 'diffEditor.'];
    }
    async createMonacoDiffEditor(uri, override, toDispose) {
        const [original, modified] = diff_uris_1.DiffUris.decode(uri);
        const [originalModel, modifiedModel] = await Promise.all([this.getModel(original, toDispose), this.getModel(modified, toDispose)]);
        const options = this.createMonacoDiffEditorOptions(originalModel, modifiedModel);
        const editor = new monaco_diff_editor_1.MonacoDiffEditor(uri, document.createElement('div'), originalModel, modifiedModel, this.services, this.diffNavigatorFactory, options, override);
        toDispose.push(this.editorPreferences.onPreferenceChanged(event => {
            const originalFileUri = original.withoutQuery().withScheme('file').toString();
            if (event.affects(originalFileUri, editor.document.languageId)) {
                this.updateMonacoDiffEditorOptions(editor, event, originalFileUri);
            }
        }));
        toDispose.push(editor.onLanguageChanged(() => this.updateMonacoDiffEditorOptions(editor)));
        return editor;
    }
    createMonacoDiffEditorOptions(original, modified) {
        const options = this.createOptions(this.diffPreferencePrefixes, modified.uri, modified.languageId);
        options.originalEditable = !original.readOnly;
        options.readOnly = modified.readOnly;
        return options;
    }
    updateMonacoDiffEditorOptions(editor, event, resourceUri) {
        if (event) {
            const preferenceName = event.preferenceName;
            const overrideIdentifier = editor.document.languageId;
            const newValue = this.editorPreferences.get({ preferenceName, overrideIdentifier }, undefined, resourceUri);
            editor.diffEditor.updateOptions(this.setOption(preferenceName, newValue, this.diffPreferencePrefixes));
        }
        else {
            const options = this.createMonacoDiffEditorOptions(editor.originalModel, editor.modifiedModel);
            editor.diffEditor.updateOptions(options);
        }
    }
    createOptions(prefixes, uri, overrideIdentifier) {
        return Object.keys(this.editorPreferences).reduce((options, preferenceName) => {
            const value = this.editorPreferences.get({ preferenceName, overrideIdentifier }, undefined, uri);
            return this.setOption(preferenceName, common_1.deepClone(value), prefixes, options);
        }, {});
    }
    setOption(preferenceName, value, prefixes, options = {}) {
        const optionName = this.toOptionName(preferenceName, prefixes);
        this.doSetOption(options, value, optionName.split('.'));
        return options;
    }
    toOptionName(preferenceName, prefixes) {
        for (const prefix of prefixes) {
            if (preferenceName.startsWith(prefix)) {
                return preferenceName.substr(prefix.length);
            }
        }
        return preferenceName;
    }
    doSetOption(obj, value, names, idx = 0) {
        const name = names[idx];
        if (!obj[name]) {
            if (names.length > (idx + 1)) {
                obj[name] = {};
                this.doSetOption(obj[name], value, names, (idx + 1));
            }
            else {
                obj[name] = value;
            }
        }
    }
    installReferencesController(editor) {
        const control = editor.getControl();
        const referencesController = control._contributions['editor.contrib.referencesController'];
        const originalGotoReference = referencesController._gotoReference;
        referencesController._gotoReference = async (ref) => {
            if (referencesController._widget) {
                referencesController._widget.hide();
            }
            referencesController._ignoreModelChangeEvent = true;
            const range = monaco.Range.lift(ref.range).collapseToStart();
            // preserve the model that it does not get disposed if an editor preview replaces an editor
            const model = referencesController._model;
            referencesController._model = undefined;
            referencesController._editorService.openCodeEditor({
                resource: ref.uri,
                options: { selection: range }
            }, control).then(openedEditor => {
                referencesController._model = model;
                referencesController._ignoreModelChangeEvent = false;
                if (!openedEditor) {
                    referencesController.closeWidget();
                    return;
                }
                if (openedEditor !== control) {
                    // preserve the model that it does not get disposed in `referencesController.closeWidget`
                    referencesController._model = undefined;
                    // to preserve the active editor
                    const focus = control.focus;
                    control.focus = () => { };
                    referencesController.closeWidget();
                    control.focus = focus;
                    const modelPromise = Promise.resolve(model);
                    modelPromise.cancel = () => { };
                    openedEditor._contributions['editor.contrib.referencesController'].toggleWidget(range, modelPromise, true);
                    return;
                }
                if (referencesController._widget) {
                    referencesController._widget.show(range);
                    referencesController._widget.focusOnReferenceTree();
                }
            }, (e) => {
                referencesController._ignoreModelChangeEvent = false;
                monaco.error.onUnexpectedError(e);
            });
        };
        return common_1.Disposable.create(() => referencesController._gotoReference = originalGotoReference);
    }
    getDiffNavigator(editor) {
        if (editor instanceof monaco_diff_editor_1.MonacoDiffEditor) {
            return editor.diffNavigator;
        }
        return monaco_diff_navigator_factory_1.MonacoDiffNavigatorFactory.nullNavigator;
    }
    async createInline(uri, node, options) {
        return this.doCreateEditor(uri, async (override, toDispose) => {
            override.contextMenuService = {
                showContextMenu: () => { }
            };
            const document = new monaco_editor_model_1.MonacoEditorModel({
                uri,
                readContents: async () => '',
                dispose: () => { }
            }, this.m2p, this.p2m);
            toDispose.push(document);
            const model = (await document.load()).textEditorModel;
            return new monaco_editor_1.MonacoEditor(uri, document, node, this.services, Object.assign({
                model,
                isSimpleWidget: true,
                autoSizing: false,
                minHeight: 1,
                maxHeight: 1
            }, MonacoEditorProvider_1.inlineOptions, options), override);
        });
    }
};
MonacoEditorProvider.inlineOptions = {
    wordWrap: 'on',
    overviewRulerLanes: 0,
    glyphMargin: false,
    lineNumbers: 'off',
    folding: false,
    selectOnLineNumbers: false,
    hideCursorInOverviewRuler: true,
    selectionHighlight: false,
    scrollbar: {
        horizontal: 'hidden'
    },
    lineDecorationsWidth: 0,
    overviewRulerBorder: false,
    scrollBeyondLastLine: false,
    renderLineHighlight: 'none',
    fixedOverflowWidgets: true,
    acceptSuggestionOnEnter: 'smart',
    minimap: {
        enabled: false
    }
};
__decorate([
    inversify_1.inject(core_1.ContributionProvider),
    inversify_1.named(exports.MonacoEditorFactory),
    __metadata("design:type", Object)
], MonacoEditorProvider.prototype, "factories", void 0);
__decorate([
    inversify_1.inject(monaco_bulk_edit_service_1.MonacoBulkEditService),
    __metadata("design:type", monaco_bulk_edit_service_1.MonacoBulkEditService)
], MonacoEditorProvider.prototype, "bulkEditService", void 0);
__decorate([
    inversify_1.inject(monaco_editor_1.MonacoEditorServices),
    __metadata("design:type", monaco_editor_1.MonacoEditorServices)
], MonacoEditorProvider.prototype, "services", void 0);
__decorate([
    inversify_1.inject(browser_2.KeybindingRegistry),
    __metadata("design:type", browser_2.KeybindingRegistry)
], MonacoEditorProvider.prototype, "keybindingRegistry", void 0);
__decorate([
    inversify_1.inject(browser_2.OpenerService),
    __metadata("design:type", Object)
], MonacoEditorProvider.prototype, "openerService", void 0);
__decorate([
    inversify_1.inject(browser_3.FileSystemPreferences),
    __metadata("design:type", Object)
], MonacoEditorProvider.prototype, "filePreferences", void 0);
__decorate([
    inversify_1.inject(monaco_quick_input_service_1.MonacoQuickInputImplementation),
    __metadata("design:type", monaco_quick_input_service_1.MonacoQuickInputImplementation)
], MonacoEditorProvider.prototype, "quickInputService", void 0);
MonacoEditorProvider = MonacoEditorProvider_1 = __decorate([
    inversify_1.injectable(),
    __param(0, inversify_1.inject(monaco_editor_service_1.MonacoEditorService)),
    __param(1, inversify_1.inject(monaco_text_model_service_1.MonacoTextModelService)),
    __param(2, inversify_1.inject(monaco_context_menu_1.MonacoContextMenuService)),
    __param(3, inversify_1.inject(monaco_to_protocol_converter_1.MonacoToProtocolConverter)),
    __param(4, inversify_1.inject(protocol_to_monaco_converter_1.ProtocolToMonacoConverter)),
    __param(5, inversify_1.inject(monaco_workspace_1.MonacoWorkspace)),
    __param(6, inversify_1.inject(monaco_command_service_1.MonacoCommandServiceFactory)),
    __param(7, inversify_1.inject(browser_1.EditorPreferences)),
    __param(8, inversify_1.inject(monaco_diff_navigator_factory_1.MonacoDiffNavigatorFactory)),
    __param(9, inversify_1.inject(application_protocol_1.ApplicationServer)),
    __param(10, inversify_1.inject(monaco.contextKeyService.ContextKeyService)),
    __metadata("design:paramtypes", [monaco_editor_service_1.MonacoEditorService,
        monaco_text_model_service_1.MonacoTextModelService,
        monaco_context_menu_1.MonacoContextMenuService,
        monaco_to_protocol_converter_1.MonacoToProtocolConverter,
        protocol_to_monaco_converter_1.ProtocolToMonacoConverter,
        monaco_workspace_1.MonacoWorkspace, Function, Object, monaco_diff_navigator_factory_1.MonacoDiffNavigatorFactory, Object, monaco.contextKeyService.ContextKeyService])
], MonacoEditorProvider);
exports.MonacoEditorProvider = MonacoEditorProvider;
//# sourceMappingURL=monaco-editor-provider.js.map