"use strict";
/********************************************************************************
 * Copyright (C) 2018 Ericsson and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AbstractResourcePreferenceProvider = void 0;
/* eslint-disable @typescript-eslint/no-explicit-any */
/* eslint-disable no-null/no-null */
const jsoncparser = require("jsonc-parser");
const async_mutex_1 = require("async-mutex");
const inversify_1 = require("@theia/core/shared/inversify");
const message_service_1 = require("@theia/core/lib/common/message-service");
const disposable_1 = require("@theia/core/lib/common/disposable");
const browser_1 = require("@theia/core/lib/browser");
const uri_1 = require("@theia/core/lib/common/uri");
const preference_configurations_1 = require("@theia/core/lib/browser/preferences/preference-configurations");
const monaco_text_model_service_1 = require("@theia/monaco/lib/browser/monaco-text-model-service");
const monaco_workspace_1 = require("@theia/monaco/lib/browser/monaco-workspace");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const file_service_1 = require("@theia/filesystem/lib/browser/file-service");
const core_1 = require("@theia/core");
const browser_2 = require("@theia/editor/lib/browser");
let AbstractResourcePreferenceProvider = class AbstractResourcePreferenceProvider extends browser_1.PreferenceProvider {
    constructor() {
        super(...arguments);
        this.preferences = {};
        this.loading = new promise_util_1.Deferred();
        this.modelInitialized = false;
        this.singleChangeLock = new async_mutex_1.Mutex();
        this.transactionLock = new async_mutex_1.Mutex();
        this.pendingTransaction = new promise_util_1.Deferred();
    }
    async init() {
        this.pendingTransaction.resolve(true);
        const uri = this.getUri();
        this.toDispose.push(disposable_1.Disposable.create(() => this.loading.reject(new Error(`preference provider for '${uri}' was disposed`))));
        await this.readPreferencesFromFile();
        this._ready.resolve();
        const reference = await this.textModelService.createModelReference(uri);
        if (this.toDispose.disposed) {
            reference.dispose();
            return;
        }
        this.model = reference.object;
        this.loading.resolve();
        this.modelInitialized = true;
        this.toDispose.push(reference);
        this.toDispose.push(disposable_1.Disposable.create(() => this.model = undefined));
        this.toDispose.push(this.model.onDidChangeContent(() => !this.transactionLock.isLocked() && this.readPreferences()));
        this.toDispose.push(this.model.onDirtyChanged(() => !this.transactionLock.isLocked() && this.readPreferences()));
        this.toDispose.push(this.model.onDidChangeValid(() => this.readPreferences()));
        this.toDispose.push(disposable_1.Disposable.create(() => this.reset()));
    }
    get valid() {
        var _a;
        return this.modelInitialized ? !!((_a = this.model) === null || _a === void 0 ? void 0 : _a.valid) : Object.keys(this.preferences).length > 0;
    }
    getConfigUri(resourceUri) {
        if (!resourceUri) {
            return this.getUri();
        }
        return this.valid && this.contains(resourceUri) ? this.getUri() : undefined;
    }
    contains(resourceUri) {
        if (!resourceUri) {
            return true;
        }
        const domain = this.getDomain();
        if (!domain) {
            return true;
        }
        const resourcePath = new uri_1.default(resourceUri).path;
        return domain.some(uri => new uri_1.default(uri).path.relativity(resourcePath) >= 0);
    }
    getPreferences(resourceUri) {
        return this.valid && this.contains(resourceUri) ? this.preferences : {};
    }
    async setPreference(key, value, resourceUri) {
        const locks = await this.acquireLocks();
        let shouldSave = Boolean(locks === null || locks === void 0 ? void 0 : locks.releaseTransaction);
        try {
            await this.loading.promise;
            let path;
            if (!this.model || !(path = this.getPath(key)) || !this.contains(resourceUri)) {
                return false;
            }
            if (!locks) {
                throw new core_1.CancellationError();
            }
            if (shouldSave) {
                if (this.model.dirty) {
                    shouldSave = await this.handleDirtyEditor();
                }
                if (!shouldSave) {
                    throw new core_1.CancellationError();
                }
            }
            const editOperations = this.getEditOperations(path, value);
            if (editOperations.length > 0) {
                await this.workspace.applyBackgroundEdit(this.model, editOperations, false);
            }
            return this.pendingTransaction.promise;
        }
        catch (e) {
            if (e instanceof core_1.CancellationError) {
                throw e;
            }
            const message = `Failed to update the value of '${key}' in '${this.getUri()}'.`;
            this.messageService.error(`${message} Please check if it is corrupted.`);
            console.error(`${message}`, e);
            return false;
        }
        finally {
            this.releaseLocks(locks, shouldSave);
        }
    }
    /**
     * @returns `undefined` if the queue has been cleared by a user action.
     */
    async acquireLocks() {
        // Request locks immediately
        const releaseTransactionPromise = this.transactionLock.isLocked() ? undefined : this.transactionLock.acquire();
        const releaseChangePromise = this.singleChangeLock.acquire().catch(() => {
            releaseTransactionPromise === null || releaseTransactionPromise === void 0 ? void 0 : releaseTransactionPromise.then(release => release());
            return undefined;
        });
        if (releaseTransactionPromise) {
            await this.pendingTransaction.promise; // Ensure previous transaction complete before starting a new one.
            this.pendingTransaction = new promise_util_1.Deferred();
        }
        // Wait to acquire locks
        const [releaseTransaction, releaseChange] = await Promise.all([releaseTransactionPromise, releaseChangePromise]);
        return releaseChange && { releaseTransaction, releaseChange };
    }
    releaseLocks(locks, shouldSave) {
        if (locks === null || locks === void 0 ? void 0 : locks.releaseTransaction) {
            if (shouldSave) {
                this.singleChangeLock.waitForUnlock().then(() => this.singleChangeLock.runExclusive(async () => {
                    var _a;
                    locks.releaseTransaction(); // Release lock so that no new changes join this transaction.
                    let success = false;
                    try {
                        await ((_a = this.model) === null || _a === void 0 ? void 0 : _a.save());
                        success = true;
                    }
                    finally {
                        this.readPreferences();
                        await this.fireDidPreferencesChanged(); // Ensure all consumers of the event have received it.
                        this.pendingTransaction.resolve(success);
                    }
                }));
            }
            else { // User canceled the operation.
                this.singleChangeLock.cancel();
                locks.releaseTransaction();
                this.pendingTransaction.resolve(false);
            }
        }
        locks === null || locks === void 0 ? void 0 : locks.releaseChange();
    }
    getEditOperations(path, value) {
        const textModel = this.model.textEditorModel;
        const content = this.model.getText().trim();
        // Everything is already undefined - no need for changes.
        if (!content && value === undefined) {
            return [];
        }
        // Delete the entire document.
        if (!path.length && value === undefined) {
            return [{
                    range: textModel.getFullModelRange(),
                    text: null,
                    forceMoveMarkers: false
                }];
        }
        const { insertSpaces, tabSize, defaultEOL } = textModel.getOptions();
        const jsonCOptions = {
            formattingOptions: {
                insertSpaces,
                tabSize,
                eol: defaultEOL === monaco.editor.DefaultEndOfLine.LF ? '\n' : '\r\n'
            }
        };
        return jsoncparser.modify(content, path, value, jsonCOptions).map(edit => {
            const start = textModel.getPositionAt(edit.offset);
            const end = textModel.getPositionAt(edit.offset + edit.length);
            return {
                range: monaco.Range.fromPositions(start, end),
                text: edit.content || null,
                forceMoveMarkers: false
            };
        });
    }
    getPath(preferenceName) {
        return [preferenceName];
    }
    async readPreferencesFromFile() {
        const content = await this.fileService.read(this.getUri()).catch(() => ({ value: '' }));
        this.readPreferencesFromContent(content.value);
    }
    /**
     * It HAS to be sync to ensure that `setPreference` returns only when values are updated
     * or any other operation modifying the monaco model content.
     */
    readPreferences() {
        const model = this.model;
        if (!model || model.dirty) {
            return;
        }
        try {
            const content = model.valid ? model.getText() : '';
            this.readPreferencesFromContent(content);
        }
        catch (e) {
            console.error(`Failed to load preferences from '${this.getUri()}'.`, e);
        }
    }
    readPreferencesFromContent(content) {
        let preferencesInJson;
        try {
            preferencesInJson = this.parse(content);
        }
        catch (_a) {
            preferencesInJson = {};
        }
        const parsedPreferences = this.getParsedContent(preferencesInJson);
        this.handlePreferenceChanges(parsedPreferences);
    }
    parse(content) {
        content = content.trim();
        if (!content) {
            return undefined;
        }
        const strippedContent = jsoncparser.stripComments(content);
        return jsoncparser.parse(strippedContent);
    }
    handlePreferenceChanges(newPrefs) {
        const oldPrefs = Object.assign({}, this.preferences);
        this.preferences = newPrefs;
        const prefNames = new Set([...Object.keys(oldPrefs), ...Object.keys(newPrefs)]);
        const prefChanges = [];
        const uri = this.getUri();
        for (const prefName of prefNames.values()) {
            const oldValue = oldPrefs[prefName];
            const newValue = newPrefs[prefName];
            const schemaProperties = this.schemaProvider.getCombinedSchema().properties[prefName];
            if (schemaProperties) {
                const scope = schemaProperties.scope;
                // do not emit the change event if the change is made out of the defined preference scope
                if (!this.schemaProvider.isValidInScope(prefName, this.getScope())) {
                    console.warn(`Preference ${prefName} in ${uri} can only be defined in scopes: ${browser_1.PreferenceScope.getScopeNames(scope).join(', ')}.`);
                    continue;
                }
            }
            if (!browser_1.PreferenceProvider.deepEqual(newValue, oldValue)) {
                prefChanges.push({
                    preferenceName: prefName, newValue, oldValue, scope: this.getScope(), domain: this.getDomain()
                });
            }
        }
        if (prefChanges.length > 0) {
            this.emitPreferencesChangedEvent(prefChanges);
        }
    }
    reset() {
        const preferences = this.preferences;
        this.preferences = {};
        const changes = [];
        for (const prefName of Object.keys(preferences)) {
            const value = preferences[prefName];
            if (value !== undefined) {
                changes.push({
                    preferenceName: prefName, newValue: undefined, oldValue: value, scope: this.getScope(), domain: this.getDomain()
                });
            }
        }
        if (changes.length > 0) {
            this.emitPreferencesChangedEvent(changes);
        }
    }
    /**
     * @returns whether the setting operation in progress, and any others started in the meantime, should continue.
     */
    async handleDirtyEditor() {
        const saveAndRetry = core_1.nls.localizeByDefault('Save and Retry');
        const open = core_1.nls.localizeByDefault('Open File');
        const msg = await this.messageService.error(core_1.nls.localizeByDefault('Unable to write into {0} settings because the file has unsaved changes. Please save the {0} settings file first and then try again.', core_1.nls.localizeByDefault(browser_1.PreferenceScope[this.getScope()].toLocaleLowerCase())), saveAndRetry, open);
        if (this.model) {
            if (msg === open) {
                this.editorManager.open(new uri_1.default(this.model.uri));
                return false;
            }
            else if (msg === saveAndRetry) {
                await this.model.save();
                return true;
            }
        }
        return false;
    }
};
__decorate([
    inversify_1.inject(message_service_1.MessageService),
    __metadata("design:type", message_service_1.MessageService)
], AbstractResourcePreferenceProvider.prototype, "messageService", void 0);
__decorate([
    inversify_1.inject(browser_1.PreferenceSchemaProvider),
    __metadata("design:type", browser_1.PreferenceSchemaProvider)
], AbstractResourcePreferenceProvider.prototype, "schemaProvider", void 0);
__decorate([
    inversify_1.inject(file_service_1.FileService),
    __metadata("design:type", file_service_1.FileService)
], AbstractResourcePreferenceProvider.prototype, "fileService", void 0);
__decorate([
    inversify_1.inject(browser_2.EditorManager),
    __metadata("design:type", browser_2.EditorManager)
], AbstractResourcePreferenceProvider.prototype, "editorManager", void 0);
__decorate([
    inversify_1.inject(preference_configurations_1.PreferenceConfigurations),
    __metadata("design:type", preference_configurations_1.PreferenceConfigurations)
], AbstractResourcePreferenceProvider.prototype, "configurations", void 0);
__decorate([
    inversify_1.inject(monaco_text_model_service_1.MonacoTextModelService),
    __metadata("design:type", monaco_text_model_service_1.MonacoTextModelService)
], AbstractResourcePreferenceProvider.prototype, "textModelService", void 0);
__decorate([
    inversify_1.inject(monaco_workspace_1.MonacoWorkspace),
    __metadata("design:type", monaco_workspace_1.MonacoWorkspace)
], AbstractResourcePreferenceProvider.prototype, "workspace", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], AbstractResourcePreferenceProvider.prototype, "init", null);
AbstractResourcePreferenceProvider = __decorate([
    inversify_1.injectable()
], AbstractResourcePreferenceProvider);
exports.AbstractResourcePreferenceProvider = AbstractResourcePreferenceProvider;
//# sourceMappingURL=abstract-resource-preference-provider.js.map