"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.disposableTimeout = exports.DisposableGroup = exports.DisposableCollection = exports.Disposable = void 0;
// *****************************************************************************
// Copyright (C) 2017 TypeFox and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
const event_1 = require("./event");
var Disposable;
(function (Disposable) {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    function is(arg) {
        return !!arg && typeof arg === 'object' && 'dispose' in arg && typeof arg['dispose'] === 'function';
    }
    Disposable.is = is;
    function create(func) {
        return { dispose: func };
    }
    Disposable.create = create;
})(Disposable = exports.Disposable || (exports.Disposable = {}));
/**
 * Ensures that every reference to {@link Disposable.NULL} returns a new object,
 * as sharing a disposable between multiple {@link DisposableCollection} can have unexpected side effects
 */
Object.defineProperty(Disposable, 'NULL', {
    configurable: false,
    enumerable: true,
    get() {
        return { dispose: () => { } };
    }
});
class DisposableCollection {
    constructor(...toDispose) {
        this.disposables = [];
        this.onDisposeEmitter = new event_1.Emitter();
        this.disposingElements = false;
        toDispose.forEach(d => this.push(d));
    }
    /**
     * This event is fired only once
     * on first dispose of not empty collection.
     */
    get onDispose() {
        return this.onDisposeEmitter.event;
    }
    checkDisposed() {
        if (this.disposed && !this.disposingElements) {
            this.onDisposeEmitter.fire(undefined);
            this.onDisposeEmitter.dispose();
        }
    }
    get disposed() {
        return this.disposables.length === 0;
    }
    dispose() {
        if (this.disposed || this.disposingElements) {
            return;
        }
        this.disposingElements = true;
        while (!this.disposed) {
            try {
                this.disposables.pop().dispose();
            }
            catch (e) {
                console.error(e);
            }
        }
        this.disposingElements = false;
        this.checkDisposed();
    }
    push(disposable) {
        const disposables = this.disposables;
        disposables.push(disposable);
        const originalDispose = disposable.dispose.bind(disposable);
        const toRemove = Disposable.create(() => {
            const index = disposables.indexOf(disposable);
            if (index !== -1) {
                disposables.splice(index, 1);
            }
            this.checkDisposed();
        });
        disposable.dispose = () => {
            toRemove.dispose();
            disposable.dispose = originalDispose;
            originalDispose();
        };
        return toRemove;
    }
    pushAll(disposables) {
        return disposables.map(disposable => this.push(disposable));
    }
}
exports.DisposableCollection = DisposableCollection;
var DisposableGroup;
(function (DisposableGroup) {
    function canPush(candidate) {
        return Boolean(candidate && candidate.push);
    }
    DisposableGroup.canPush = canPush;
    function canAdd(candidate) {
        return Boolean(candidate && candidate.add);
    }
    DisposableGroup.canAdd = canAdd;
})(DisposableGroup = exports.DisposableGroup || (exports.DisposableGroup = {}));
function disposableTimeout(...args) {
    const handle = setTimeout(...args);
    return { dispose: () => clearTimeout(handle) };
}
exports.disposableTimeout = disposableTimeout;
//# sourceMappingURL=disposable.js.map