"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.createMonacoEditorAPI = exports.registerCommand = exports.remeasureFonts = exports.setTheme = exports.defineTheme = exports.tokenize = exports.colorizeModelLine = exports.colorize = exports.colorizeElement = exports.createWebWorker = exports.onDidChangeModelLanguage = exports.onWillDisposeModel = exports.onDidCreateModel = exports.getModels = exports.getModel = exports.onDidChangeMarkers = exports.getModelMarkers = exports.removeAllMarkers = exports.setModelMarkers = exports.setModelLanguage = exports.createModel = exports.createDiffNavigator = exports.createDiffEditor = exports.getDiffEditors = exports.getEditors = exports.onDidCreateDiffEditor = exports.onDidCreateEditor = exports.create = void 0;
require("./standalone-tokens.css");
const strings_1 = require("../../../base/common/strings");
const fontMeasurements_1 = require("../../browser/config/fontMeasurements");
const codeEditorService_1 = require("../../browser/services/codeEditorService");
const diffNavigator_1 = require("../../browser/widget/diffNavigator");
const editorOptions_1 = require("../../common/config/editorOptions");
const fontInfo_1 = require("../../common/config/fontInfo");
const editorCommon_1 = require("../../common/editorCommon");
const model_1 = require("../../common/model");
const languages = require("../../common/languages");
const languageConfigurationRegistry_1 = require("../../common/languages/languageConfigurationRegistry");
const nullTokenize_1 = require("../../common/languages/nullTokenize");
const language_1 = require("../../common/languages/language");
const model_2 = require("../../common/services/model");
const webWorker_1 = require("../../browser/services/webWorker");
const standaloneEnums = require("../../common/standalone/standaloneEnums");
const colorizer_1 = require("./colorizer");
const standaloneCodeEditor_1 = require("./standaloneCodeEditor");
const standaloneServices_1 = require("./standaloneServices");
const standaloneTheme_1 = require("../common/standaloneTheme");
const commands_1 = require("../../../platform/commands/common/commands");
const markers_1 = require("../../../platform/markers/common/markers");
/**
 * Create a new editor under `domElement`.
 * `domElement` should be empty (not contain other dom nodes).
 * The editor will read the size of `domElement`.
 */
function create(domElement, options, override) {
    const instantiationService = standaloneServices_1.StandaloneServices.initialize(override || {});
    return instantiationService.createInstance(standaloneCodeEditor_1.StandaloneEditor, domElement, options);
}
exports.create = create;
/**
 * Emitted when an editor is created.
 * Creating a diff editor might cause this listener to be invoked with the two editors.
 * @event
 */
function onDidCreateEditor(listener) {
    const codeEditorService = standaloneServices_1.StandaloneServices.get(codeEditorService_1.ICodeEditorService);
    return codeEditorService.onCodeEditorAdd((editor) => {
        listener(editor);
    });
}
exports.onDidCreateEditor = onDidCreateEditor;
/**
 * Emitted when an diff editor is created.
 * @event
 */
function onDidCreateDiffEditor(listener) {
    const codeEditorService = standaloneServices_1.StandaloneServices.get(codeEditorService_1.ICodeEditorService);
    return codeEditorService.onDiffEditorAdd((editor) => {
        listener(editor);
    });
}
exports.onDidCreateDiffEditor = onDidCreateDiffEditor;
/**
 * Get all the created editors.
 */
function getEditors() {
    const codeEditorService = standaloneServices_1.StandaloneServices.get(codeEditorService_1.ICodeEditorService);
    return codeEditorService.listCodeEditors();
}
exports.getEditors = getEditors;
/**
 * Get all the created diff editors.
 */
function getDiffEditors() {
    const codeEditorService = standaloneServices_1.StandaloneServices.get(codeEditorService_1.ICodeEditorService);
    return codeEditorService.listDiffEditors();
}
exports.getDiffEditors = getDiffEditors;
/**
 * Create a new diff editor under `domElement`.
 * `domElement` should be empty (not contain other dom nodes).
 * The editor will read the size of `domElement`.
 */
function createDiffEditor(domElement, options, override) {
    const instantiationService = standaloneServices_1.StandaloneServices.initialize(override || {});
    return instantiationService.createInstance(standaloneCodeEditor_1.StandaloneDiffEditor, domElement, options);
}
exports.createDiffEditor = createDiffEditor;
function createDiffNavigator(diffEditor, opts) {
    return new diffNavigator_1.DiffNavigator(diffEditor, opts);
}
exports.createDiffNavigator = createDiffNavigator;
/**
 * Create a new editor model.
 * You can specify the language that should be set for this model or let the language be inferred from the `uri`.
 */
function createModel(value, language, uri) {
    const languageService = standaloneServices_1.StandaloneServices.get(language_1.ILanguageService);
    const languageId = languageService.getLanguageIdByMimeType(language) || language;
    return (0, standaloneCodeEditor_1.createTextModel)(standaloneServices_1.StandaloneServices.get(model_2.IModelService), languageService, value, languageId, uri);
}
exports.createModel = createModel;
/**
 * Change the language for a model.
 */
function setModelLanguage(model, languageId) {
    const languageService = standaloneServices_1.StandaloneServices.get(language_1.ILanguageService);
    const modelService = standaloneServices_1.StandaloneServices.get(model_2.IModelService);
    modelService.setMode(model, languageService.createById(languageId));
}
exports.setModelLanguage = setModelLanguage;
/**
 * Set the markers for a model.
 */
function setModelMarkers(model, owner, markers) {
    if (model) {
        const markerService = standaloneServices_1.StandaloneServices.get(markers_1.IMarkerService);
        markerService.changeOne(owner, model.uri, markers);
    }
}
exports.setModelMarkers = setModelMarkers;
/**
 * Remove all markers of an owner.
 */
function removeAllMarkers(owner) {
    const markerService = standaloneServices_1.StandaloneServices.get(markers_1.IMarkerService);
    markerService.changeAll(owner, []);
}
exports.removeAllMarkers = removeAllMarkers;
/**
 * Get markers for owner and/or resource
 *
 * @returns list of markers
 */
function getModelMarkers(filter) {
    const markerService = standaloneServices_1.StandaloneServices.get(markers_1.IMarkerService);
    return markerService.read(filter);
}
exports.getModelMarkers = getModelMarkers;
/**
 * Emitted when markers change for a model.
 * @event
 */
function onDidChangeMarkers(listener) {
    const markerService = standaloneServices_1.StandaloneServices.get(markers_1.IMarkerService);
    return markerService.onMarkerChanged(listener);
}
exports.onDidChangeMarkers = onDidChangeMarkers;
/**
 * Get the model that has `uri` if it exists.
 */
function getModel(uri) {
    const modelService = standaloneServices_1.StandaloneServices.get(model_2.IModelService);
    return modelService.getModel(uri);
}
exports.getModel = getModel;
/**
 * Get all the created models.
 */
function getModels() {
    const modelService = standaloneServices_1.StandaloneServices.get(model_2.IModelService);
    return modelService.getModels();
}
exports.getModels = getModels;
/**
 * Emitted when a model is created.
 * @event
 */
function onDidCreateModel(listener) {
    const modelService = standaloneServices_1.StandaloneServices.get(model_2.IModelService);
    return modelService.onModelAdded(listener);
}
exports.onDidCreateModel = onDidCreateModel;
/**
 * Emitted right before a model is disposed.
 * @event
 */
function onWillDisposeModel(listener) {
    const modelService = standaloneServices_1.StandaloneServices.get(model_2.IModelService);
    return modelService.onModelRemoved(listener);
}
exports.onWillDisposeModel = onWillDisposeModel;
/**
 * Emitted when a different language is set to a model.
 * @event
 */
function onDidChangeModelLanguage(listener) {
    const modelService = standaloneServices_1.StandaloneServices.get(model_2.IModelService);
    return modelService.onModelLanguageChanged((e) => {
        listener({
            model: e.model,
            oldLanguage: e.oldLanguageId
        });
    });
}
exports.onDidChangeModelLanguage = onDidChangeModelLanguage;
/**
 * Create a new web worker that has model syncing capabilities built in.
 * Specify an AMD module to load that will `create` an object that will be proxied.
 */
function createWebWorker(opts) {
    return (0, webWorker_1.createWebWorker)(standaloneServices_1.StandaloneServices.get(model_2.IModelService), standaloneServices_1.StandaloneServices.get(languageConfigurationRegistry_1.ILanguageConfigurationService), opts);
}
exports.createWebWorker = createWebWorker;
/**
 * Colorize the contents of `domNode` using attribute `data-lang`.
 */
function colorizeElement(domNode, options) {
    const languageService = standaloneServices_1.StandaloneServices.get(language_1.ILanguageService);
    const themeService = standaloneServices_1.StandaloneServices.get(standaloneTheme_1.IStandaloneThemeService);
    themeService.registerEditorContainer(domNode);
    return colorizer_1.Colorizer.colorizeElement(themeService, languageService, domNode, options);
}
exports.colorizeElement = colorizeElement;
/**
 * Colorize `text` using language `languageId`.
 */
function colorize(text, languageId, options) {
    const languageService = standaloneServices_1.StandaloneServices.get(language_1.ILanguageService);
    const themeService = standaloneServices_1.StandaloneServices.get(standaloneTheme_1.IStandaloneThemeService);
    themeService.registerEditorContainer(document.body);
    return colorizer_1.Colorizer.colorize(languageService, text, languageId, options);
}
exports.colorize = colorize;
/**
 * Colorize a line in a model.
 */
function colorizeModelLine(model, lineNumber, tabSize = 4) {
    const themeService = standaloneServices_1.StandaloneServices.get(standaloneTheme_1.IStandaloneThemeService);
    themeService.registerEditorContainer(document.body);
    return colorizer_1.Colorizer.colorizeModelLine(model, lineNumber, tabSize);
}
exports.colorizeModelLine = colorizeModelLine;
/**
 * @internal
 */
function getSafeTokenizationSupport(language) {
    const tokenizationSupport = languages.TokenizationRegistry.get(language);
    if (tokenizationSupport) {
        return tokenizationSupport;
    }
    return {
        getInitialState: () => nullTokenize_1.NullState,
        tokenize: (line, hasEOL, state) => (0, nullTokenize_1.nullTokenize)(language, state)
    };
}
/**
 * Tokenize `text` using language `languageId`
 */
function tokenize(text, languageId) {
    // Needed in order to get the mode registered for subsequent look-ups
    languages.TokenizationRegistry.getOrCreate(languageId);
    const tokenizationSupport = getSafeTokenizationSupport(languageId);
    const lines = (0, strings_1.splitLines)(text);
    const result = [];
    let state = tokenizationSupport.getInitialState();
    for (let i = 0, len = lines.length; i < len; i++) {
        const line = lines[i];
        const tokenizationResult = tokenizationSupport.tokenize(line, true, state);
        result[i] = tokenizationResult.tokens;
        state = tokenizationResult.endState;
    }
    return result;
}
exports.tokenize = tokenize;
/**
 * Define a new theme or update an existing theme.
 */
function defineTheme(themeName, themeData) {
    const standaloneThemeService = standaloneServices_1.StandaloneServices.get(standaloneTheme_1.IStandaloneThemeService);
    standaloneThemeService.defineTheme(themeName, themeData);
}
exports.defineTheme = defineTheme;
/**
 * Switches to a theme.
 */
function setTheme(themeName) {
    const standaloneThemeService = standaloneServices_1.StandaloneServices.get(standaloneTheme_1.IStandaloneThemeService);
    standaloneThemeService.setTheme(themeName);
}
exports.setTheme = setTheme;
/**
 * Clears all cached font measurements and triggers re-measurement.
 */
function remeasureFonts() {
    fontMeasurements_1.FontMeasurements.clearAllFontInfos();
}
exports.remeasureFonts = remeasureFonts;
/**
 * Register a command.
 */
function registerCommand(id, handler) {
    return commands_1.CommandsRegistry.registerCommand({ id, handler });
}
exports.registerCommand = registerCommand;
/**
 * @internal
 */
function createMonacoEditorAPI() {
    return {
        // methods
        create: create,
        getEditors: getEditors,
        getDiffEditors: getDiffEditors,
        onDidCreateEditor: onDidCreateEditor,
        onDidCreateDiffEditor: onDidCreateDiffEditor,
        createDiffEditor: createDiffEditor,
        createDiffNavigator: createDiffNavigator,
        createModel: createModel,
        setModelLanguage: setModelLanguage,
        setModelMarkers: setModelMarkers,
        getModelMarkers: getModelMarkers,
        removeAllMarkers: removeAllMarkers,
        onDidChangeMarkers: onDidChangeMarkers,
        getModels: getModels,
        getModel: getModel,
        onDidCreateModel: onDidCreateModel,
        onWillDisposeModel: onWillDisposeModel,
        onDidChangeModelLanguage: onDidChangeModelLanguage,
        createWebWorker: createWebWorker,
        colorizeElement: colorizeElement,
        colorize: colorize,
        colorizeModelLine: colorizeModelLine,
        tokenize: tokenize,
        defineTheme: defineTheme,
        setTheme: setTheme,
        remeasureFonts: remeasureFonts,
        registerCommand: registerCommand,
        // enums
        AccessibilitySupport: standaloneEnums.AccessibilitySupport,
        ContentWidgetPositionPreference: standaloneEnums.ContentWidgetPositionPreference,
        CursorChangeReason: standaloneEnums.CursorChangeReason,
        DefaultEndOfLine: standaloneEnums.DefaultEndOfLine,
        EditorAutoIndentStrategy: standaloneEnums.EditorAutoIndentStrategy,
        EditorOption: standaloneEnums.EditorOption,
        EndOfLinePreference: standaloneEnums.EndOfLinePreference,
        EndOfLineSequence: standaloneEnums.EndOfLineSequence,
        MinimapPosition: standaloneEnums.MinimapPosition,
        MouseTargetType: standaloneEnums.MouseTargetType,
        OverlayWidgetPositionPreference: standaloneEnums.OverlayWidgetPositionPreference,
        OverviewRulerLane: standaloneEnums.OverviewRulerLane,
        RenderLineNumbersType: standaloneEnums.RenderLineNumbersType,
        RenderMinimap: standaloneEnums.RenderMinimap,
        ScrollbarVisibility: standaloneEnums.ScrollbarVisibility,
        ScrollType: standaloneEnums.ScrollType,
        TextEditorCursorBlinkingStyle: standaloneEnums.TextEditorCursorBlinkingStyle,
        TextEditorCursorStyle: standaloneEnums.TextEditorCursorStyle,
        TrackedRangeStickiness: standaloneEnums.TrackedRangeStickiness,
        WrappingIndent: standaloneEnums.WrappingIndent,
        InjectedTextCursorStops: standaloneEnums.InjectedTextCursorStops,
        PositionAffinity: standaloneEnums.PositionAffinity,
        // classes
        ConfigurationChangedEvent: editorOptions_1.ConfigurationChangedEvent,
        BareFontInfo: fontInfo_1.BareFontInfo,
        FontInfo: fontInfo_1.FontInfo,
        TextModelResolvedOptions: model_1.TextModelResolvedOptions,
        FindMatch: model_1.FindMatch,
        ApplyUpdateResult: editorOptions_1.ApplyUpdateResult,
        // vars
        EditorType: editorCommon_1.EditorType,
        EditorOptions: editorOptions_1.EditorOptions
    };
}
exports.createMonacoEditorAPI = createMonacoEditorAPI;
//# sourceMappingURL=standaloneEditor.js.map