"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnitControllerHistoryHandler = void 0;
class UnitControllerHistoryHandler {
    constructor(uC) {
        this.index = 0;
        this.maxAllowedIndex = 0;
        this.restoring = false;
        this.history = [];
        this.unitController = uC;
    }
    addCurrentState() {
        const { selectionRange, viewRange } = this.unitController;
        this.enqueueItem({ selectionRange, viewRange });
    }
    undo() {
        if (this.canUndo) {
            this.index--;
            this.restore();
        }
    }
    redo() {
        if (this.canRedo) {
            this.index++;
            this.restore();
        }
    }
    clear() {
        this.index = 0;
        this.maxAllowedIndex = 0;
    }
    enqueueItem(item) {
        /**
         * Since scrolling with the scroll-bar or dragging handle triggers many changes per second
         * we don't want to actually push if another request comes in quick succession.
         *
         * Don't add anything if we are currently restoring.
         */
        if (this.restoring) {
            return;
        }
        if (this.timeout) {
            clearTimeout(this.timeout);
        }
        this.timeout = setTimeout(() => this.add(item), 500);
    }
    add(item) {
        const isDuplicate = this.isEntryDuplicate(item);
        if (!isDuplicate) {
            this.index++;
            this.maxAllowedIndex = this.index;
            this.history[this.index] = item;
        }
    }
    restore() {
        this.restoring = true;
        const { selectionRange, viewRange } = this.history[this.index];
        this.unitController.selectionRange = selectionRange;
        this.unitController.viewRange = viewRange;
        setTimeout(() => this.restoring = false, 500);
    }
    isEntryDuplicate(item) {
        // Checks if stack entry is same as previous entry.
        if (this.index === 0) {
            return false;
        }
        let oneIsDifferent = false;
        const { selectionRange: itemSR, viewRange: itemVR } = item;
        const { selectionRange: prevSR, viewRange: prevVR } = this.history[this.index];
        const check = (value1, value2) => {
            if (oneIsDifferent) {
                return;
            }
            oneIsDifferent = (value1 !== value2);
        };
        check(itemSR === null || itemSR === void 0 ? void 0 : itemSR.start, prevSR === null || prevSR === void 0 ? void 0 : prevSR.start);
        check(itemSR === null || itemSR === void 0 ? void 0 : itemSR.end, prevSR === null || prevSR === void 0 ? void 0 : prevSR.end);
        check(itemVR.start, prevVR.start);
        check(itemVR.end, prevVR.end);
        return !oneIsDifferent;
    }
    get canRedo() {
        return this.index < this.maxAllowedIndex;
    }
    get canUndo() {
        return this.index > 1;
    }
}
exports.UnitControllerHistoryHandler = UnitControllerHistoryHandler;
//# sourceMappingURL=unit-controller-history-handler.js.map