"use strict";
/*********************************************************************
 * Copyright (c) 2018 QNX Software Systems and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const path = require("path");
const utils_1 = require("./utils");
const chai = require("chai");
const chaistring = require("chai-string");
chai.use(chaistring);
describe('Variables Test Suite', function () {
    let dc;
    let scope;
    const varsProgram = path.join(utils_1.testProgramsDir, 'vars');
    const varsSrc = path.join(utils_1.testProgramsDir, 'vars.c');
    const numVars = 9; // number of variables in the main() scope of vars.c
    const lineTags = {
        'STOP HERE': 0,
        'After array init': 0,
    };
    const hexValueRegex = /^0x[\da-fA-F]+$/;
    before(function () {
        (0, utils_1.resolveLineTagLocations)(varsSrc, lineTags);
    });
    beforeEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            dc = yield (0, utils_1.standardBeforeEach)();
            yield dc.hitBreakpoint((0, utils_1.fillDefaults)(this.currentTest, {
                program: varsProgram,
            }), {
                path: varsSrc,
                line: lineTags['STOP HERE'],
            });
            scope = yield (0, utils_1.getScopes)(dc);
            (0, chai_1.expect)(scope.scopes.body.scopes.length, 'Unexpected number of scopes returned').to.equal(2);
        });
    });
    afterEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        });
    });
    it('can read and set simple variables in a program', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // read the variables
            let vr = scope.scopes.body.scopes[0].variablesReference;
            let vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.equal(numVars);
            (0, utils_1.verifyVariable)(vars.body.variables[0], 'a', 'int', '1');
            (0, utils_1.verifyVariable)(vars.body.variables[1], 'b', 'int', '2');
            // set the variables to something different
            const setAinHex = yield dc.setVariableRequest({
                name: 'a',
                value: '0x25',
                variablesReference: vr,
            });
            (0, chai_1.expect)(setAinHex.body.value).to.equal('37');
            const setA = yield dc.setVariableRequest({
                name: 'a',
                value: '25',
                variablesReference: vr,
            });
            (0, chai_1.expect)(setA.body.value).to.equal('25');
            const setB = yield dc.setVariableRequest({
                name: 'b',
                value: '10',
                variablesReference: vr,
            });
            (0, chai_1.expect)(setB.body.value).to.equal('10');
            // assert that the variables have been updated to the new values
            vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.equal(numVars);
            (0, utils_1.verifyVariable)(vars.body.variables[0], 'a', 'int', '25');
            (0, utils_1.verifyVariable)(vars.body.variables[1], 'b', 'int', '10');
            // step the program and see that the values were passed to the program and evaluated.
            yield dc.next({ threadId: scope.thread.id }, { path: varsSrc, line: lineTags['STOP HERE'] + 1 });
            scope = yield (0, utils_1.getScopes)(dc);
            (0, chai_1.expect)(scope.scopes.body.scopes.length, 'Unexpected number of scopes returned').to.equal(2);
            vr = scope.scopes.body.scopes[0].variablesReference;
            vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.equal(numVars);
            (0, utils_1.verifyVariable)(vars.body.variables[2], 'c', 'int', '35');
        });
    });
    it('can read and set simple registers in a program', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // read the registers
            const vr = scope.scopes.body.scopes[1].variablesReference;
            const vr1 = scope.scopes.body.scopes[0].variablesReference;
            const vars = yield dc.variablesRequest({ variablesReference: vr });
            const vars1 = yield dc.variablesRequest({ variablesReference: vr1 });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.be.greaterThanOrEqual(5); // 5 is a good bet to make sure that code has probably worked
            const r0 = vars.body.variables[0];
            const r1 = vars.body.variables[1];
            const rn = vars.body.variables[vars.body.variables.length - 1];
            // can't check specific names or register values easily as that
            // is not cross platform
            (0, chai_1.expect)(r0.evaluateName).to.startWith('$');
            (0, chai_1.expect)(r0.name).to.not.equal(r1.name);
            // add other useful tests here, especially ones that test boundary conditions
            (0, chai_1.expect)(rn === null || rn === void 0 ? void 0 : rn.evaluateName).to.startWith('$'); // check last registers
            // set the registers value to something different
            const setR0 = yield dc.setVariableRequest({
                name: r0.name,
                value: '55',
                variablesReference: vr,
            });
            (0, chai_1.expect)(setR0.body.value).to.equal('0x37');
            const setR0inHex = yield dc.setVariableRequest({
                name: r0.name,
                value: '0x55',
                variablesReference: vr,
            });
            (0, chai_1.expect)(setR0inHex.body.value).to.equal('0x55');
            const setR1inHex = yield dc.setVariableRequest({
                name: r1.name,
                value: '0x45',
                variablesReference: vr,
            });
            (0, chai_1.expect)(setR1inHex.body.value).to.equal('0x45');
            const setR1 = yield dc.setVariableRequest({
                name: r1.name,
                value: '45',
                variablesReference: vr,
            });
            (0, chai_1.expect)(setR1.body.value).to.equal('0x2d');
            // assert that the registers value have been updated to the new values
            const vars2 = yield dc.variablesRequest({ variablesReference: vr });
            const vars3 = yield dc.variablesRequest({ variablesReference: vr1 });
            (0, chai_1.expect)(vars2.body.variables.length, 'There is a different number of registers than expected').to.equal(vars.body.variables.length);
            (0, utils_1.verifyRegister)(vars2.body.variables[0], r0.name, '0x55');
            (0, utils_1.verifyRegister)(vars2.body.variables[1], r1.name, '0x2d');
            (0, utils_1.verifyRegister)(vars3.body.variables[8], r0.name, vars1.body.variables[8].value);
        });
    });
    it('can read and set struct variables in a program', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // step past the initialization for the structure
            yield dc.next({ threadId: scope.thread.id }, { path: varsSrc, line: lineTags['STOP HERE'] + 1 });
            yield dc.next({ threadId: scope.thread.id }, { path: varsSrc, line: lineTags['STOP HERE'] + 2 });
            scope = yield (0, utils_1.getScopes)(dc);
            (0, chai_1.expect)(scope.scopes.body.scopes.length, 'Unexpected number of scopes returned').to.equal(2);
            // assert we can see the struct and its elements
            let vr = scope.scopes.body.scopes[0].variablesReference;
            let vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.equal(numVars);
            (0, utils_1.verifyVariable)(vars.body.variables[3], 'r', 'struct foo', '{...}', {
                hasChildren: true,
            });
            const childVR = vars.body.variables[3].variablesReference;
            let children = yield dc.variablesRequest({
                variablesReference: childVR,
            });
            (0, chai_1.expect)(children.body.variables.length, 'There is a different number of child variables than expected').to.equal(3);
            (0, utils_1.verifyVariable)(children.body.variables[0], 'x', 'int', '1', {
                hasMemoryReference: false,
            });
            (0, utils_1.verifyVariable)(children.body.variables[1], 'y', 'int', '2', {
                hasMemoryReference: false,
            });
            (0, utils_1.verifyVariable)(children.body.variables[2], 'z', 'struct bar', '{...}', {
                hasChildren: true,
                hasMemoryReference: false,
            });
            // set the variables to something different
            const setXinHex = yield dc.setVariableRequest({
                name: 'x',
                value: '0x25',
                variablesReference: childVR,
            });
            (0, chai_1.expect)(setXinHex.body.value).to.equal('37');
            const setX = yield dc.setVariableRequest({
                name: 'x',
                value: '25',
                variablesReference: childVR,
            });
            (0, chai_1.expect)(setX.body.value).to.equal('25');
            const setY = yield dc.setVariableRequest({
                name: 'y',
                value: '10',
                variablesReference: childVR,
            });
            (0, chai_1.expect)(setY.body.value).to.equal('10');
            // assert that the variables have been updated to the new values
            children = yield dc.variablesRequest({ variablesReference: childVR });
            (0, chai_1.expect)(children.body.variables.length, 'There is a different number of child variables than expected').to.equal(3);
            (0, utils_1.verifyVariable)(children.body.variables[0], 'x', 'int', '25', {
                hasMemoryReference: false,
            });
            (0, utils_1.verifyVariable)(children.body.variables[1], 'y', 'int', '10', {
                hasMemoryReference: false,
            });
            // step the program and see that the values were passed to the program and evaluated.
            yield dc.next({ threadId: scope.thread.id }, { path: varsSrc, line: lineTags['STOP HERE'] + 3 });
            scope = yield (0, utils_1.getScopes)(dc);
            (0, chai_1.expect)(scope.scopes.body.scopes.length, 'Unexpected number of scopes returned').to.equal(2);
            vr = scope.scopes.body.scopes[0].variablesReference;
            vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.equal(numVars);
            (0, utils_1.verifyVariable)(vars.body.variables[4], 'd', 'int', '35');
        });
    });
    it('can read and set nested struct variables in a program', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // step past the initialization for the structure
            yield dc.next({ threadId: scope.thread.id }, { path: varsSrc, line: lineTags['STOP HERE'] + 1 });
            yield dc.next({ threadId: scope.thread.id }, { path: varsSrc, line: lineTags['STOP HERE'] + 2 });
            scope = yield (0, utils_1.getScopes)(dc);
            (0, chai_1.expect)(scope.scopes.body.scopes.length, 'Unexpected number of scopes returned').to.equal(2);
            // assert we can see the 'foo' struct and its child 'bar' struct
            let vr = scope.scopes.body.scopes[0].variablesReference;
            let vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.equal(numVars);
            (0, utils_1.verifyVariable)(vars.body.variables[3], 'r', 'struct foo', '{...}', {
                hasChildren: true,
            });
            const childVR = vars.body.variables[3].variablesReference;
            const children = yield dc.variablesRequest({
                variablesReference: childVR,
            });
            (0, chai_1.expect)(children.body.variables.length, 'There is a different number of child variables than expected').to.equal(3);
            (0, utils_1.verifyVariable)(children.body.variables[2], 'z', 'struct bar', '{...}', {
                hasChildren: true,
                hasMemoryReference: false,
            });
            // assert we can see the elements of z
            const subChildVR = children.body.variables[2].variablesReference;
            let subChildren = yield dc.variablesRequest({
                variablesReference: subChildVR,
            });
            (0, chai_1.expect)(subChildren.body.variables.length, 'There is a different number of grandchild variables than expected').to.equal(2);
            (0, utils_1.verifyVariable)(subChildren.body.variables[0], 'a', 'int', '3', {
                hasMemoryReference: false,
            });
            (0, utils_1.verifyVariable)(subChildren.body.variables[1], 'b', 'int', '4', {
                hasMemoryReference: false,
            });
            // set the variables to something different
            const setAinHex = yield dc.setVariableRequest({
                name: 'a',
                value: '0x25',
                variablesReference: subChildVR,
            });
            (0, chai_1.expect)(setAinHex.body.value).to.equal('37');
            const setA = yield dc.setVariableRequest({
                name: 'a',
                value: '25',
                variablesReference: subChildVR,
            });
            (0, chai_1.expect)(setA.body.value).to.equal('25');
            const setB = yield dc.setVariableRequest({
                name: 'b',
                value: '10',
                variablesReference: subChildVR,
            });
            (0, chai_1.expect)(setB.body.value).to.equal('10');
            // assert that the variables have been updated to the new values
            subChildren = yield dc.variablesRequest({
                variablesReference: subChildVR,
            });
            (0, chai_1.expect)(subChildren.body.variables.length, 'There is a different number of grandchild variables than expected').to.equal(2);
            (0, utils_1.verifyVariable)(subChildren.body.variables[0], 'a', 'int', '25', {
                hasMemoryReference: false,
            });
            (0, utils_1.verifyVariable)(subChildren.body.variables[1], 'b', 'int', '10', {
                hasMemoryReference: false,
            });
            // step the program and see that the values were passed to the program and evaluated.
            yield dc.next({ threadId: scope.thread.id }, { path: varsSrc, line: lineTags['STOP HERE'] + 3 });
            yield dc.next({ threadId: scope.thread.id }, { path: varsSrc, line: lineTags['STOP HERE'] + 4 });
            scope = yield (0, utils_1.getScopes)(dc);
            (0, chai_1.expect)(scope.scopes.body.scopes.length, 'Unexpected number of scopes returned').to.equal(2);
            vr = scope.scopes.body.scopes[0].variablesReference;
            vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.equal(numVars);
            (0, utils_1.verifyVariable)(vars.body.variables[5], 'e', 'int', '35');
        });
    });
    it('can read and set array elements in a program', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // skip ahead to array initialization
            const br = yield dc.setBreakpointsRequest({
                source: { path: varsSrc },
                breakpoints: [{ line: lineTags['After array init'] }],
            });
            (0, chai_1.expect)(br.success).to.equal(true);
            yield dc.continue({ threadId: scope.thread.id }, 'breakpoint', {
                line: 24,
                path: varsSrc,
            });
            scope = yield (0, utils_1.getScopes)(dc);
            (0, chai_1.expect)(scope.scopes.body.scopes.length, 'Unexpected number of scopes returned').to.equal(2);
            // assert we can see the array and its elements
            let vr = scope.scopes.body.scopes[0].variablesReference;
            let vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.equal(numVars);
            (0, utils_1.verifyVariable)(vars.body.variables[6], 'f', 'int [3]', undefined, {
                hasChildren: true,
            });
            (0, chai_1.expect)(vars.body.variables[6].value, 'The display value of the array is not a hexadecimal address').to.match(hexValueRegex);
            const childVR = vars.body.variables[6].variablesReference;
            let children = yield dc.variablesRequest({
                variablesReference: childVR,
            });
            (0, chai_1.expect)(children.body.variables.length, 'There is a different number of child variables than expected').to.equal(3);
            (0, utils_1.verifyVariable)(children.body.variables[0], '[0]', 'int', '1', {
                hasMemoryReference: false,
            });
            (0, utils_1.verifyVariable)(children.body.variables[1], '[1]', 'int', '2', {
                hasMemoryReference: false,
            });
            (0, utils_1.verifyVariable)(children.body.variables[2], '[2]', 'int', '3', {
                hasMemoryReference: false,
            });
            // set the variables to something different
            const set0inHex = yield dc.setVariableRequest({
                name: '[0]',
                value: '0x11',
                variablesReference: childVR,
            });
            (0, chai_1.expect)(set0inHex.body.value).to.equal('17');
            const set0 = yield dc.setVariableRequest({
                name: '[0]',
                value: '11',
                variablesReference: childVR,
            });
            (0, chai_1.expect)(set0.body.value).to.equal('11');
            const set1 = yield dc.setVariableRequest({
                name: '[1]',
                value: '22',
                variablesReference: childVR,
            });
            (0, chai_1.expect)(set1.body.value).to.equal('22');
            const set2 = yield dc.setVariableRequest({
                name: '[2]',
                value: '33',
                variablesReference: childVR,
            });
            (0, chai_1.expect)(set2.body.value).to.equal('33');
            // assert that the variables have been updated to the new values
            children = yield dc.variablesRequest({ variablesReference: childVR });
            (0, chai_1.expect)(children.body.variables.length, 'There is a different number of child variables than expected').to.equal(3);
            (0, utils_1.verifyVariable)(children.body.variables[0], '[0]', 'int', '11', {
                hasMemoryReference: false,
            });
            (0, utils_1.verifyVariable)(children.body.variables[1], '[1]', 'int', '22', {
                hasMemoryReference: false,
            });
            (0, utils_1.verifyVariable)(children.body.variables[2], '[2]', 'int', '33', {
                hasMemoryReference: false,
            });
            // step the program and see that the values were passed to the program and evaluated.
            yield dc.next({ threadId: scope.thread.id }, { path: varsSrc, line: 25 });
            scope = yield (0, utils_1.getScopes)(dc);
            (0, chai_1.expect)(scope.scopes.body.scopes.length, 'Unexpected number of scopes returned').to.equal(2);
            vr = scope.scopes.body.scopes[0].variablesReference;
            vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, chai_1.expect)(vars.body.variables.length, 'There is a different number of variables than expected').to.equal(numVars);
            (0, utils_1.verifyVariable)(vars.body.variables[7], 'g', 'int', '66');
        });
    });
});
//# sourceMappingURL=var.spec.js.map