# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: basics.py,v 1.8 1998/11/09 13:01:46 gobry Exp $

# --------------------------------------------------
# Basical `high level' functions
# --------------------------------------------------

import tempfile
from types import *
import os, sys
import traceback
import pyb, pybhelp
from pybopen import bibopen

# ----- Create elementary Test -----

pybhelp.register ('searching', """
The general syntax of a search is :

 dbase.where ( constraints )

...where `constraints' is an arbitrary construction of :

  -  has (field, value) : match if `field' matches `value' (as a REGEXP)
  -  &, | : boolean operators (& = and, | = or)

Negation of a constraint is noted `- constraint'.

Examples :

dbase.where (has ('author', 'me'))
dbase.where (has ('author', 'me') & - has ('title', 'failed'))

A search returns references on the entries that were found. It is then
possible to search these references to restrain the result.

See also: references
""")

def has (field, value):
	return pyb.Tester (field, value)


# ----- Generic reference holder -----

pybhelp.register ('ref',"""
Syntax : reference = ref (base, [entries])

This function returns a reference object (like .where () method) on
the specified database (eventually restricted to a list of entry
names).
""")

def ref (base, list = None):
	ref = pyb.Reference ()

	if list == None:
		ref.add (base)
	else:
		if type (list) is ListType:
			ref.add (base, list)
		else:
			ref.add (base, [ list ])
	return ref


# ----- Edition, as BibTeX file -----

pybhelp.register ('edit', """
Syntax : edit (references)

This command starts an editor on the specified entries.
""")

def edit (full_items):

	for base in full_items.bases ():

		items = full_items.refs (base)

		if items == None:
			print "warning: can't edit complete databases"
			continue
		
		name = tempfile.mktemp () + '.bib'
		file = open (name, 'w')

		try:
			# Sauver les donnees
			for entry in items:
				
				if type (entry) is InstanceType:
					file.write (entry)
				else:
					file.write (base [entry].__str__ ())
					
				file.write ('\n\n')
                
			file.close ()

			# What editor should we use ?
	
			editor = 'vi'
			if os.environ.has_key ('EDITOR'):
				editor = os.environ ['EDITOR']

			# Editer le fichier temporaire
			os.system (editor + ' ' + name)

			# Fusionner les donnes du fichier dans la base...
			dbase = bibopen (name)

			merge (base, dbase)
        
			os.unlink (name)
        
		except:
			traceback.print_exc ()
			print "temporary file was `%s'" % name

	
    
# ----- Generic Merging -----

pybhelp.register ('merge', """
Syntax: merge (target, source)

Add all entries from source in target.
""")

def merge (target, source):
    "Merges source into target"

    def merge_items (item, base):
        base [item.name ()] = item
        
    source.foreach (merge_items, target)
    target.update ()


# ----- Display -----

def pager_handle (inpager):
	if inpager:
		pagername = 'more'
	
		if os.environ.has_key ('PAGER'):
			pagername = os.environ ['PAGER']

		pager = os.popen (pagername, 'w')
	else:
		pager = sys.stdout

	return pager


pybhelp.register ('more', """
Syntax: more (references, [nopager])

Display references in BibTeX format. If nopager is 1, the entries are
sent to stdout instead of a pager.
""")

def more (refs, inpager = 1):
	"Output entries"

	def printer (entry, arg):
		arg.write (`entry` + '\n\n')

	try:
		refs.foreach (printer, pager_handle (inpager))
	except IOError:
		print "warning: broken pipe"
		

def ls (refs, inpager = 1):
	bibmore (refs, inpager)


pybhelp.register ('ls', """
Syntax: ls (references, [nopager])

Display only title/author and identifier for each entry.

See also : more, less
""")

def ls (refs, inpager = 1):
	"Output entries"

	def printer (entry, arg):
		title = 'no title'
		author = 'no author'
		if entry.has_key ('title'):
			title = entry ['title']
		if entry.has_key ('author'):
			author = entry ['author']

		title = title [0:34]
		author = author [0:24]
		name = entry.name () [0:15]
		
		arg.write ('%-35s %-25s [%-16s]\n' % (title, author, name))


	try:
		refs.foreach (printer, pager_handle (inpager))
	except IOError:
		print "warning: broken pipe"


# ----- Comment in another database -----

pybhelp.register ('comment', """
Syntax: comments = comment (references, database)

Create new entries in `database' that contain comments on the
specified `references', and edits these references. During edition,
the full items are shown, but be aware that only 'CrossRef' and
'Remarks' fields are kept in the database.

`comment' returns a reference on the comments.
""")

def comment (refs, base):
	"Ajouter a une base des commentaires sur des refs"

	def commenter (entry, arg):
		base, ret = arg
		
		name = entry.name () + '-comment'

		ret.add (base, [name])
		
		if base.has_key (name):
			print "warning: key %s already in %s" % (name, base)
			return

		com = pyb.Entry (name, entry.type (), {})

		com ['crossref'] = entry.name ()
		com ['remarks'] = 'put your remarks here'
		
		com = com + entry
		base [name] = com

	def simplifier (entry, base):
		name = entry.name ()

		if not base.has_key (name):
			print "warning: entry %s not available" % name
			return

		com = base [name]
		for e in com.keys ():
			if e != 'crossref' and e != 'remarks':
				del com [e]

		base [name] = com
		
	# Creer la base a commenter
	retrefs = pyb.Reference ()
	refs.foreach (commenter, (base, retrefs))

	# Editer les commentaires
	edit (retrefs)

	# Supprimer tout ce qui n'est pas commentaire
	retrefs.foreach (simplifier, base)
	base.update ()
	
	return retrefs

# ----- Create merged view -----

pybhelp.register ('complete', """
Syntax: complete (references, databases, [nopager])

This command acts like `more', but additionally looks into `databases'
to complete entries with `CrossRef' fields.
""")

def complete (refs, base, inpager = 1):
	"Create references completed from external databases"

	if inpager:
		pagername = 'more'
		
		if os.environ.has_key ('PAGER'):
			pagername = os.environ ['PAGER']
			
		pager = os.popen (pagername, 'w')
	else:
		pager = sys.stdout
		
	def printer (entry, arg):

		pager, base = arg

		if entry.has_key ('crossref'):
			key = entry ['crossref']

			for b in base:
				if b.has_key (key):
					entry = entry + b [key]
					
		pager.write (`entry` + '\n\n')

	try:
		if not (type (base) is ListType):
			base = [ base ]
		refs.foreach (printer, (pager, base))
		
	except IOError:
		print "warning: broken pipe"
