# This file is part of pybliographer
# 
# Copyright (C) 1998 Frederic GOBRY
# Email : gobry@idiap.ch
# 	   
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2 
# of the License, or (at your option) any later version.
#   
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details. 
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# $Id: Base.py,v 1.18 1999/07/30 12:32:11 gobry Exp $

from string import *
import copy, re
import Pyblio.Help
from Pyblio.Fields import *
from types import *

Entries = {}
Types   = {}

def getentry (entry):
	""" Returns an entry description given its name """
	l = lower (entry)
	
	if Entries.has_key (l):
		return Entries [l]
	
	return EntryDescription (entry)

def gettype (entry, field):
	""" Return a field type given its name and the entry it belongs to """
	
	if entry and entry.has_key (field):
		return entry [field].type
	
	if Types.has_key (field):
		return Types [field].type
	
	return TypeText

	
# ----- Boolean test -----

class Bool:
	""" Base class for boolean tests """

	def __init__ (self):
		self.neg = 0
		
	def __or__ (self, other):
		return OrConnect (self, other)

	def __and__ (self, other):
		return AndConnect (self, other)

	def __neg__ (self):
		self.neg = not self.neg
		return self

class Tester (Bool):

	def __init__ (self, field, value):
		Bool.__init__ (self)
		self.__test = re.compile (value, re.IGNORECASE)

		self.field = lower (field)
		self.value = value

	def match (self, entry):
		if self.neg:
			# Negative test
			if entry.has_key (self.field):
				field = entry [self.field]
				return field.match (self.__test) == None
			else:
				return 1
		else:
			if entry.has_key (self.field):
				field = entry [self.field]
				return field.match (self.__test) != None
			else:
				return 0
			
class Connecter (Bool):
	def __init__ (self, left, right):
		Bool.__init__ (self)
		self.left = left
		self.right = right


class OrConnect (Connecter):
	def match (self, entry):
		ret = (self.left.match (entry) or
		       self.right.match (entry)) 
		if self.neg:
			ret = not ret

		return ret

class AndConnect (Connecter):
	def match (self, entry):
		ret = (self.left.match (entry) and
			self.right.match (entry))  
		if self.neg:
			ret = not ret

		return ret


		
		
def format (string, width, first, next):
	"Format a string on a given width"

	out = []
	current = first
	
	while len (string) > width - current:
		
		pos = width - current - 1
		
		while pos > 0 and string [pos] <> ' ':
			pos = pos - 1

		if pos == 0:
			pos = width - current
			taille = len (string)
			while pos < taille and string [pos] <> ' ':
				pos = pos + 1

		out.append (' ' * current + string [0:pos])
		string = string [pos+1:]
		current = next

	out.append (' ' * current + string)

	return strip (join (out, '\n'))


class Key:
	"""
	A special key that embeds both database and database key
	Such a key is expected to be completely unique among the whole
	program and should be the reliable information checked to see
	if two entries are the same.

	The .base field is public and is the database from which the
	actual entry can be recovered, whereas the .key is private to
	the database itself.
	"""
	
	def __init__ (self, base, key):
		if type (key) is InstanceType:
			self.base = key.base
			self.key  = key.key
		else:
			self.base = base
			self.key  = key
		return

	def __repr__ (self):
		if self.base:
			return `self.base.key + ' - ' + self.key`
		else:
			return `self.key`
		
	def __cmp__ (self, other):
		r = cmp (self.base, other.base)
		if r: return r
		
		return cmp (self.key, other.key)

	def __hash__ (self):
		return hash (self.key + '\0' + self.base.key)


class Entry:

	id = 'VirtualEntry'
	
	def __init__ (self, key, name, type, dict):
		self.name = name
		self.type = type
		self.__dict = dict
		self.key    = key

		
	def keys (self):
		return self.__dict.keys ()

	def has_key (self, key):
		return self.__dict.has_key (key)

	def text (self, key):
		""" return text with indication of convertion loss """
		return self.__dict [key], 0

	def native (self, key):
		""" returns the field in its native form """
		return self [key]
	
	def __getitem__ (self, key):
		""" return raw text of a field """
		return self.text (key) [0]
	
	def foreach (self, function, argument = None):
		""" To provide compatibility with ref """
		function (self, argument)
	
	def __setitem__ (self, name, value):
		self.__dict [name] = value

	def __delitem__ (self, name):
		del self.__dict [name]

	def __add__ (self, other):
		"Merges two entries"

		ret = Entry (self.key, self.name, self.type, {})

		# Prendre ses propres entrees
		for f in self.keys ():
			ret [f] = self [f]

		# et ajouter celles qu'on n'a pas
		for f in other.keys ():
			if not self.has_key (f):
				ret [f] = other [f]

		return ret

	def __repr__ (self):
		return "<entry `" + self.name + "'>"


	def __str__ (self):
		""" Nice standard entry  """
		tp = self.type.name
		fields = self.type.fields
			
		text = '%s [%s]\n' % (tp, self.name)
		text = text + ('-' * 70) + '\n'
		
		dico = self.keys ()
			
		for f in fields:
			name = f.name
			lcname = lower (name)
			
			if not self.has_key (lcname): continue
			
			text = text + '  '
			text = text + format (
				'%-14s %s' % (name, str (self [lcname])),
				75, 0, 17)
			text = text + '\n'
			
			try:
				dico.remove (lcname)
			except ValueError:
				raise ValueError, ("error: field `%s' appears more than once " +
						   "in the definition of `%s'") % (name, tp)
			
		for f in dico:
			text = text + '  '
			text = text + format (
				'%-14s %s' % (f, str (self [f])), 75, 0, 17)
			text = text + '\n'
		
		return text
	
		
# Une base de references bibliographiques, comme un dictionnaire
# avec des extensions...

class DataBase:
	id    = 'VirtualDB'
	__count = 0
	
	def __init__ (self, basename):
		self.name = basename

		self.key  = 'base-' + str (DataBase.__count)
		DataBase.__count = DataBase.__count + 1
		return
	
		
	def keys (self):
		""
		raise NameError, "not inherited"

	def has_key (self, field, key):
		""
		raise NameError, "not inherited"

	def __getitem__ (self, name):
		""
		raise NameError, "not inherited"

	def __setitem__ (self, name, value):
		""
		raise NameError, "not inherited"

	def __delitem__ (self, name):
		""
		raise NameError, "not inherited"

	def __len__ (self, name):
		""
		return len (self.keys ())

	def __repr__ (self):
		""
		return "<generic bibliographic database (" + `len (self)` + \
		       " entries)>"
		
	# Specifique a la base biblio
	def where (self, key):
		"Chercher une entree selon des criteres"
		raise NameError, "not inherited"

	def foreach (self, function, args = None):
		"Parcourt de toutes les entrees"
		raise NameError, "not inherited"

	def update (self):
		"Remise a jour des donnes"
		raise NameError, "not inherited"


	def sort (self, sortkey):
		""" sort entries according to a given key """
		# collect key values
		dict = {}
		none = []
		
		def collect (entry, arg):
			sortkey, dict, none = arg
			
			if entry.has_key (sortkey):
				dict [entry.key] = entry [sortkey]
			else:
				none.append (entry.key)
			return

		self.foreach (collect, (sortkey, dict, none))
		
		keys = dict.keys ()
		def sort_method (a,b, dict = dict):
			return cmp (dict [a], dict [b])
		
		keys.sort (sort_method)
		
		r = Reference ()
		r.add (self, none + keys)
		
		return r



# ----- A class that holds references from several databases -----

Pyblio.Help.register ('references', """

References are generic holders of subsets or complete databases. One
can apply several method on a reference :

 - ref.add (base, items) : adds `items' from a given `base' in the
   reference
 
 - ref.foreach (method, argument) : calls `method' on every entry
 - ref.where ( condition ) : returns the reference of entries matching
   the condition

 - ref.bases () : a list of all the bases that compose the reference
 - ref.refs (base) : entries from a given database
 - ref.base_foreach (base) : like `foreach' but only on a given database

In addition, references can be merged with `+'.
""")


class Reference (DataBase):

	__count = 0
	id = 'Reference'
		
	def __init__ (self):
		self.__refs  = []
		self.__bases = {}
		
		self.key  = 'ref-' + str (Reference.__count)
		Reference.__count = Reference.__count + 1
		return
	

	def add (self, base, liste = None):
		"Ajouter des references liees a une base"

		target = None
		if self.__bases.has_key (base.key):
			target = self.__bases [base.key] [1]

		if target is None:
			# add a new database
			local = ( base, [] )
			target = local [1]

			self.__bases [base.key] = local
			self.__refs.append (local)

		if liste is None: return

		if base.id == 'Reference':
			if not (type (liste) is ListType):
				target.append (liste)
				return
		
			for i in liste:
				target.append (i)
				
		else:
			if not (type (liste) is ListType):
				target.append (liste.key)
				return
		
			for i in liste:
				target.append (i.key)
				
		return


	def __repr__ (self):
		""
		return `self.__refs`

	def bases (self):
		""" Returns the databases available in the Reference
		object """
		
		bases = []
		for b in self.__refs:
			bases.append (b [0])
			
		return bases

	def refs (self, base):
		""" Returns the references available in a given
		database """
		
		if self.__bases.has_key (base.key):
			return map (lambda x, base = base: Key (base, x),
				    self.__bases [base.key] [1])
		
		return None

	def base_foreach (self, base, function, arg = None):
		""" Foreach entry in a base """
		
		liste = self.refs (base)

		if liste is None: return
		
		if len (liste) > 0:
			# a subset of the database
			for e in liste:
				entry = base [Key (base, e)]
				function (entry, arg)
		else:
			# whole database
			base.foreach (function, arg)

			
	def foreach (self, function, arg = None):
		"Foreach entry"

		for b in self.bases ():
			self.base_foreach (b, function, arg)


	def __len__ (self):
		total = 0
		for b in self.bases ():
			refs = self.refs (b)
			if refs == []:
				total = total + len (b)
			else:
				total = total + len (refs)

		return total


	def keys (self):
		keys = []

		def appender (entry, keys):
			keys.append (entry.key)
			return
		
		self.foreach (appender, keys)
		
		return keys
	
					  
	def has_key (self, key):
		return key.base.has_key (key)

	def __getitem__ (self, key):
		return key.base [key]

	def __delitem__ (self, key):
		del key.base [key]

	def __setitem__ (self, key, value):
		key.base [key] = value
		return
	
	def __add__ (self, other):

		r = Reference ()

		for b in self.bases ():
			r.add (b, self.refs (b))

		for b in other.bases ():
			r.add (b, other.refs (b))

		return r


	def where (self, test):
		r = Reference ()

		for b in self.bases ():
			list = self.refs (b)
			
			if list is None: continue
			
			# Si on doit tester toute une base, on
			# utilise la commande appropriee
			if list == []:
				r = r + b.where (test)
			else:
				sublist = []
				# Sinon on test cas par cas
				for e in list:
					key = Key (b, e)
					if test.match (b [key]):
						sublist.append (key)
				
				# add to the reference
				if len (sublist) > 0:
					r.add (b, sublist)

		return r


	def sort (self, sortkey):
		""" sort entries according to a given key """
		
		# collect key values
		dict = {}
		none = []
		
		def collect (entry, arg):
			sortkey, dict, none = arg
			
			if entry.has_key (sortkey):
				dict [entry.key] = entry [sortkey]
			else:
				none.append (entry.key)
			return

		self.foreach (collect, (sortkey, dict, none))
		
		keys = dict.keys ()
		def sort_method (a,b, dict = dict):
			return cmp (dict [a], dict [b])
		
		keys.sort (sort_method)
		
		r = Reference ()
		r.add (self, none + keys)
		
		return r
