//
// Gendarme.Framework.Rocks.MethodRocks
//
// Authors:
//	Sebastien Pouliot  <sebastien@ximian.com>
//	Adrian Tsai <adrian_tsai@hotmail.com>
//	Daniel Abramov <ex@vingrad.ru>
//	Andreas Noever <andreas.noever@gmail.com>
//
// Copyright (C) 2007-2008 Novell, Inc (http://www.novell.com)
// Copyright (c) 2007 Adrian Tsai
// Copyright (C) 2008 Daniel Abramov
// (C) 2008 Andreas Noever
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

using System;

using Mono.Cecil;

namespace Gendarme.Framework.Rocks {

	// add Method[Reference|Definition][Collection] extensions methods here
	// only if:
	// * you supply minimal documentation for them (xml)
	// * you supply unit tests for them
	// * they are required somewhere to simplify, even indirectly, the rules
	//   (i.e. don't bloat the framework in case of x, y or z in the future)

	/// <summary>
	/// MethodRocks contains extensions methods for Method[Definition|Reference]
	/// and the related collection classes.
	/// 
	/// Note: whenever possible try to use MethodReference since it's extend the
	/// reach/usability of the code.
	/// </summary>
	public static class MethodRocks {

		/// <summary>
		/// Check if the MethodReference is defined as the entry point of it's assembly.
		/// </summary>
		/// <param name="self">The MethodReference on which the extension method can be called.</param>
		/// <returns>True if the method is defined as the entry point of it's assembly, False otherwise</returns>
		public static bool IsEntryPoint (this MethodReference self)
		{
			return (self == self.DeclaringType.Module.Assembly.EntryPoint);
		}

		/// <summary>
		/// Check if the MethodReference is a finalizer.
		/// </summary>
		/// <param name="self">The MethodReference on which the extension method can be called.</param>
		/// <returns>True if the method is a finalizer, False otherwise.</returns>
		public static bool IsFinalizer (this MethodReference self)
		{
			return (self.HasThis && !self.HasParameters && (self.Name == "Finalize") &&
				(self.ReturnType.ReturnType.FullName == "System.Void"));
		}

		/// <summary>
		/// Check if the method, or it's declaring type, was generated by the compiler or a tool (i.e. not by the developer).
		/// </summary>
		/// <param name="self">The MethodReference on which the extension method can be called.</param>
		/// <returns>True if the code is not generated directly by the developer, 
		/// False otherwise (e.g. compiler or tool generated)</returns>
		public static bool IsGeneratedCode (this MethodReference self)
		{
			MethodDefinition method = self.Resolve ();
			if ((method != null) && method.HasCustomAttributes) {
				if (method.CustomAttributes.ContainsAnyType (CustomAttributeRocks.GeneratedCodeAttributes))
					return true;
			}
			return self.DeclaringType.IsGeneratedCode ();
		}

		/// <summary>
		/// Check if the signature of a method is consitent for it's use as a Main method.
		/// Note: it doesn't check that the method is the EntryPoint of it's assembly.
		/// <code>
		/// static [void|int] Main ()
		/// static [void|int] Main (string[] args)
		/// </code>
		/// </summary>gre
		/// <param name="self">The MethodReference on which the extension method can be called.</param>
		/// <returns>True if the method is a valid Main, False otherwise.</returns>
		public static bool IsMain (this MethodReference self)
		{
			MethodDefinition method = self.Resolve ();
			// Main must be static
			if (!method.IsStatic)
				return false;

			if (method.Name != "Main")
				return false;

			// Main must return void or int
			switch (method.ReturnType.ReturnType.Name) {
			case "Void":
			case "Int32":
				// ok, continue checks
				break;
			default:
				return false;
			}

			// Main (void)
			if (!method.HasParameters)
				return true;

			if (method.Parameters.Count != 1)
				return false;

			// Main (string[] args)
			return (method.Parameters [0].ParameterType.Name == "String[]");
		}

		/// <summary>
		/// Check if a method is an override to a virtual method of a base type.
		/// </summary>
		/// <param name="self">The MethodReference on which the extension method can be called.</param>
		/// <returns>True if the method is an override to a virtual method, False otherwise</returns>
		public static bool IsOverride (this MethodReference self)
		{
			MethodDefinition method = self.Resolve ();
			if ((method == null) || !method.IsVirtual)
				return false;

			TypeDefinition parent = method.DeclaringType.Resolve ().BaseType.Resolve ();
			while (parent != null) {
				string name = method.Name;
				string retval = method.ReturnType.ReturnType.FullName;
				int pcount = method.HasParameters ? method.Parameters.Count : 0;
				foreach (MethodDefinition md in parent.Methods) {
					if (name != md.Name)
						continue;
					if (retval != md.ReturnType.ReturnType.FullName)
						continue;
					if ((md.HasParameters && (pcount == 0)) || (pcount != md.Parameters.Count))
						continue;

					bool ok = true;
					for (int i = 0; i < pcount; i++) {
						if (method.Parameters [i].ParameterType.FullName != md.Parameters [i].ParameterType.FullName) {
							ok = false;
							break;
						}
					}
					if (!ok)
						continue;

					return md.IsVirtual;
				}
				parent = parent.BaseType.Resolve ();
			}
			return false;
		}

		/// <summary>
		/// Check if the method corresponds to the get or set operation on a property.
		/// </summary>
		/// <param name="self">The MethodReference on which the extension method can be called.</param>
		/// <returns>True if the method is a getter or a setter, False otherwise</returns>
		public static bool IsProperty (this MethodReference self)
		{
			MethodDefinition method = self.Resolve ();
			if (method == null)
				return false;
			return ((method.SemanticsAttributes & (MethodSemanticsAttributes.Getter | MethodSemanticsAttributes.Setter)) != 0);
		}

		/// <summary>
		/// Check if the method is visible outside of the assembly.
		/// </summary>
		/// <param name="self">The MethodReference on which the extension method can be called.</param>
		/// <returns>True if the method can be used from outside of the assembly, false otherwise.</returns>
		public static bool IsVisible (this MethodReference self)
		{
			MethodDefinition method = self.Resolve ();
			if ((method == null) || method.IsPrivate || method.IsAssembly)
				return false;
			return self.DeclaringType.Resolve ().IsVisible ();
		}

		/// <summary>
		/// Resolve a MethodReference into a MethodDefinition.
		/// </summary>
		/// <param name="self">The MethodReference on which the extension method can be called.</param>
		/// <returns>A MethodDefinition if resolved, null otherwise.</returns>
		public static MethodDefinition Resolve (this MethodReference self)
		{
			MethodDefinition method = (self as MethodDefinition);
			if (method == null)
				method = AssemblyResolver.Resolver.Resolve (self);
			return method;
		}

		/// <summary>
		/// Check if the method has the signature of an Event callback.
		/// They are usually of the form: void Method (object sender, EventArgs ea), where
		/// the second parameters is either EventArgs or a subclass of it
		/// </summary>
		/// <param name="self">The MethodReference on which the extension method can be called.</param>
		/// <returns>True if the method has the signature of an event callback.</returns>
		public static bool IsEventCallback (this MethodReference self)
		{
			MethodDefinition method = self.Resolve ();
			if ((method == null) || !method.HasParameters)
				return false;

			ParameterDefinitionCollection parameters = method.Parameters;
			if (parameters.Count != 2)
				return false;

			TypeReference type = parameters [1].ParameterType;
			GenericParameter gp = (type as GenericParameter);
			if (gp == null)
				return type.Inherits ("System.EventArgs");

			if (gp.HasConstraints && (gp.Constraints.Count == 1))
				return (gp.Constraints [0].FullName == "System.EventArgs");

			return false;
		}
	}
}
