/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.digikam.org
 *
 * Date        : 2009-02-04
 * Description : a command line tool to test qt PGF interface
 *
 * Copyright (C) 2009-2011 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// C++ includes

#include <ctime>

// Qt includes

#include <QBuffer>
#include <QByteArray>
#include <QDataStream>
#include <QDebug>
#include <QFile>
#include <QIODevice>
#include <QImage>

// KDE includes

#include <kdebug.h>

// Local includes

#include "pgfutils.h"

using namespace Digikam;

int main(int /*argc*/, char** /*argv*/)
{
    clock_t    start, end;
    QImage     img;
    QByteArray pgfData, jpgData, pngData;

    // QImage => PGF conversion

    kDebug() << "Using LibPGF version: " << PGFUtils::libPGFVersion();
    kDebug() << "Using OpenMP with LibPGF : " << (PGFUtils::libPGFUseOpenMP() ? "yes" : "no");

    img.load("test.png");

    kDebug() << "Generate PGF file using file stream";

    // First, write QImage as PGF file using file stream
    if (!PGFUtils::writePGFImageFile(img, "test-filestream.pgf", 0, true))
    {
        kDebug() << "writePGFImageData failed...";
        return -1;
    }

    kDebug() << "Generate PGF file using data stream";

    start = clock();

    // Second, write QImage as PGF file using data stream
    if (!PGFUtils::writePGFImageData(img, pgfData, 0, true))
    {
        kDebug() << "writePGFImageData failed...";
        return -1;
    }

    end   = clock();

    kDebug() << "PGF Encoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    // Write PGF file.

    QFile file("test-datastream.pgf");

    if ( !file.open(QIODevice::WriteOnly) )
    {
        kDebug() << "Cannot open PGF file to write...";
        return -1;
    }

    QDataStream stream(&file);
    stream.writeRawData(pgfData.data(), pgfData.size());
    file.close();

    // PGF => QImage conversion

    kDebug() << "Load PGF file generated by data stream";

    start = clock();

    if (!PGFUtils::readPGFImageData(pgfData, img, true))
    {
        kDebug() << "readPGFImageData failed...";
        return -1;
    }

    end   = clock();

    img.save("test2.png", "PNG");

    kDebug() << "PGF Decoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    // JPEG tests for comparisons.

    img.load("test.png");

    kDebug() << "Generate JPG file to compare performances";

    start = clock();

    QBuffer buffer(&jpgData);
    buffer.open(QIODevice::WriteOnly);
    img.save(&buffer, "JPEG", 85);  // Here we will use JPEG quality = 85 to reduce artifacts.

    if (jpgData.isNull())
    {
        kDebug() << "Save JPG image data to byte array failed...";
        return -1;
    }

    end   = clock();

    kDebug() << "JPG Encoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    start = clock();

    buffer.open(QIODevice::ReadOnly);
    img.load(&buffer, "JPEG");

    if (jpgData.isNull())
    {
        kDebug() << "Load JPG image data from byte array failed...";
        return -1;
    }

    end   = clock();

    kDebug() << "JPG Decoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    // PNG tests for comparisons.

    img.load("test.png");

    kDebug() << "Generate PNG file to compare performances";

    start = clock();

    QBuffer buffer2(&pngData);
    buffer2.open(QIODevice::WriteOnly);
    img.save(&buffer2, "PNG", 100);

    if (pngData.isNull())
    {
        kDebug() << "Save PNG image data to byte array failed...";
        return -1;
    }

    end   = clock();

    kDebug() << "PNG Encoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    start = clock();

    buffer2.open(QIODevice::ReadOnly);
    img.load(&buffer2, "PNG");

    if (pngData.isNull())
    {
        kDebug() << "Load PNG image data from byte array failed...";
        return -1;
    }

    end   = clock();

    kDebug() << "PNG Decoding time: " << double(end - start)/CLOCKS_PER_SEC << " s";

    return 0;
}
