/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGOBJECTMODELBASE_H
#define SKGOBJECTMODELBASE_H
/** @file
 * This file defines classes skgbjectmodelbase.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgobjectbase.h"
#include "skgbasegui_export.h"
#include "skgdocument.h"

#include <kglobal.h>
#include <QAbstractItemModel>
#include <QStringList>

class QWidget;
/**
 * The Table model managing SKGObjectBase
 */
class SKGBASEGUI_EXPORT SKGObjectModelBase : public QAbstractItemModel
{
    Q_OBJECT
    /**
     * This enumerate defines type of attribute
     */
    Q_ENUMS(AttributeType)

public:
    /**
     * Default constructor
     * @param iDocument the document where to search
     * @param iTable the table where to search
     * @param iWhereClause the where clause
     * @param iParent parent QT object
     * @param iParentAttribute the attribute to find the parent of an object clause to find children
     * @param iResetOnCreation to reset data during creation
     */
    SKGObjectModelBase(SKGDocument* iDocument,
                       const QString& iTable,
                       const QString& iWhereClause,
                       QWidget* iParent = NULL,
                       const QString& iParentAttribute = "",
                       bool iResetOnCreation = true);

    /**
     * Destructor
     */
    virtual ~SKGObjectModelBase();

    /**
     * Allows to block a refresh
     * @param iBlocked blocking status
     * @return previous value
     */
    virtual bool blockRefresh(bool iBlocked);

    /**
     * To know if the refresh is blocked
     * @return true or false
     */
    virtual bool isRefreshBlocked();

    /**
     * Returns true if parent has at least one child.
     * @param iParent the parent
     * @return true or false
     */
    virtual bool hasChildren(const QModelIndex& iParent = QModelIndex()) const;

    /**
     * Returns the number of row for the given parent.
     * @param iParent the parent
     * @return the number of row
     */
    virtual int rowCount(const QModelIndex& iParent = QModelIndex()) const;

    /**
     * Returns the index of the item in the model specified by the given row, column and parent index.
     * @param row row
     * @param column column
     * @param iParent parent
     * @return the index
     */
    virtual QModelIndex index(int row, int column, const QModelIndex& iParent = QModelIndex()) const;

    /**
     * Returns the parent of the model item with the given index, or QModelIndex() if it has no parent.
     * @param index index
     * @return the parent
     */
    virtual QModelIndex parent(const QModelIndex& index) const;

    /**
     * Returns the number of columns for the given parent.
     * @param iParent the parent
     * @return the number of column
     */
    virtual int columnCount(const QModelIndex& iParent = QModelIndex()) const;

    /**
     * Returns the data stored under the given role for the item referred to by the index.
     * @param index the index
     * @param role the role
     * @return the returned value
     */
    virtual QVariant data(const QModelIndex& index, int role = Qt::DisplayRole) const;

    /**
     * Returns the SKGObjectBase for the item referred to by the index.
     * @param index the index
     * @return the returned SKGObjectBase
     */
    virtual SKGObjectBase getObject(const QModelIndex& index) const;

    /**
     * Returns the SKGObjectBase for the item referred to by the index.
     * @param index the index
     * @return the returned SKGObjectBase. Do not delete it !
     */
    virtual SKGObjectBase* getObjectPointer(const QModelIndex& index) const;

    /**
     * Returns the data for the given role and section in the header with the specified orientation.
     * @param section the section
     * @param orientation the orientation
     * @param role the role
     * @return the header data
     */
    virtual QVariant headerData(int section, Qt::Orientation orientation, int role = Qt::DisplayRole) const;

    /**
     * Returns the item flags for the given index.
     * @param index index of the object
     * @return flags of the given index
     */
    virtual Qt::ItemFlags flags(const QModelIndex& index) const;

    /**
     * Sets the role data for the item at index to value. Returns true if successful; otherwise returns false.
     * @param index index of the object
     * @param value value
     * @param role role
     * @return
     */
    virtual bool setData(const QModelIndex& index, const QVariant& value, int role = Qt::EditRole);

    /**
     * Return the index of the attribute, -1 if not found
     * @param iAttributeName the attribute name
     * @return index of this attibute
     */
    virtual int getIndexAttribute(const QString& iAttributeName) const;

    /**
     * Return the attribute for an index
     * @param iIndex the index
     * @return attribute
     */
    virtual QString getAttribute(int iIndex) const;

    /**
     * Return the type of attribute for an index
     * @param iIndex the index
     * @return type of attribute
     */
    virtual SKGServices::AttributeType getAttributeType(int iIndex) const;

    /**
     * Set the table. Don't forget to do a reset after that.
     * @param iTable the table name
     */
    virtual void setTable(const QString& iTable);

    /**
     * Get table name
     * @return table name
     */
    virtual QString getTable() const;

    /**
     * Set the attribute used for grouping
     * @param iAttribute the attibute name
     */
    virtual void setGroupBy(const QString& iAttribute = "");

    /**
     * Get the attribute used for grouping
     * @return attibute name
     */
    virtual QString getGroupBy() const;

    /**
     * Get the attribute used for parent / child relationship
     * @return attibute name
     */
    virtual QString getParentChildAttribute() const;


    /**
     * Get real table name
     * @return real table name
     */
    virtual QString getRealTable() const;

    /**
     * Get where clause
     * @return where clause
     */
    virtual QString getWhereClause() const;

    /**
     * Get document
     * @return document
     */
    virtual SKGDocument* getDocument() const;

    /**
     * Set the list of supported attributes. Don't forget to do a reset after that.
     * @param iListAttribute the list of attributes. If the list is empty then the default list is set.
     * The format of this string is the following one: attribute name[|visibility Y or N[|size]];attribute name[|visibility Y or N[|size]];...
     */
    virtual void setSupportedAttributes(const QStringList& iListAttribute);

    /**
     * Set the filter. Don't forget to do a reset after that.
     * @param iWhereClause the where clause
     * @return true is the filter is really changed
     */
    virtual bool setFilter(const QString& iWhereClause);

    /**
     * Returns the actions supported by the data in this model.
     * @return Qt::DropActions
     */
    virtual Qt::DropActions supportedDragActions() const;

    /**
     * Returns the actions supported by the data in this model.
     * @return Qt::DropActions
     */
    virtual Qt::DropActions supportedDropActions() const;

    /**
     * Get list of supported schemas
     * @return list of schemas @see SKGModelTemplate
     */
    virtual  QList<SKGDocument::SKGModelTemplate> getSchemas() const;

    /**
     * Returns a list of MIME types that can be used to describe a list of model indexes.
     * @return list of mime types
     */
    virtual QStringList mimeTypes() const;

    /**
     * Returns an object that contains serialized items of data corresponding to the list of indexes specified.
     *The formats used to describe the encoded data is obtained from the mimeTypes() function.
     * If the list of indexes is empty, or there are no supported MIME types, 0 is returned rather than a serialized empty list.
     * @param indexes the index object
     * @return the mime data
     */
    virtual QMimeData* mimeData(const QModelIndexList& indexes) const;

    /**
     * Handles the data supplied by a drag and drop operation that ended with the given action.
     * Returns true if the data and action can be handled by the model; otherwise returns false.
     * Although the specified row, column and parent indicate the location of an item in the model where the operation ended,
     *it is the responsibility of the view to provide a suitable location for where the data should be inserted.
     * @param data mime data
     * @param action action
     * @param row row
     * @param column column
     * @param iParent parent
     * @return true if the dropping was successful otherwise false.
     */
    virtual bool dropMimeData(const QMimeData* data,
                              Qt::DropAction action,
                              int row, int column,
                              const QModelIndex& iParent);
Q_SIGNALS:
    /**
     * Emitted before reset
     */
    void beforeReset();

    /**
     * Emitted after reset
     */
    void afterReset();

public Q_SLOTS:
    /**
     * Refresh the model.
     */
    virtual void refresh();

    /**
     * data are modified
     * @param iTableName table name
     * @param iIdTransaction the id of the transaction for direct modifications of the table (update of modify objects is enough)
     *or 0 in case of modifications by impact (full table must be refreshed)
     */
    virtual void dataModified(const QString& iTableName = "", int iIdTransaction = 0);

protected Q_SLOTS:
    /**
     * This method is called by refresh to build the cache (to improve performance)
     */
    virtual void buidCache();

protected:

    /**
     *list of attributes
     */
    QStringList m_listAttibutes;

    /**
     *list of attributes
     */
    QList<SKGServices::AttributeType> m_listAttributeTypes;

    /**
     *list of supported schemas
     */
    QList<SKGDocument::SKGModelTemplate> m_listSchema;

    /**
     *To disable/enable reset
     */
    bool m_isResetRealyNeeded;

    /**
     * Get the attribute value for grouping
     * @param iObject the object
     * @param iAttribute the attribute name
     * @return the value of the attribute
     */
    virtual QString getAttributeForGrouping(const SKGObjectBase& iObject, const QString& iAttribute) const;

    /**
     *The negative color
     */
    QVariant m_fontNegativeColor;

private Q_SLOTS:
    void pageChanged();

private:
    Q_DISABLE_COPY(SKGObjectModelBase)
    void clear();
    int getID(const SKGObjectBase* obj, int row) const;

    SKGDocument* m_document;
    QString m_table;
    QString m_realTable;
    QString m_whereClause;
    QString m_parentAttribute;
    QString m_groupby;

    SKGObjectBase::SKGListSKGObjectBase m_listObjects;
    QHash<int, SKGIntList> m_parentChildRelations;
    QHash<int, int> m_childParentRelations;
    QHash<int, SKGObjectBase*> m_objectsHashTable;
    QHash<int, int> m_objectsHashTableRows;

    QStringList m_listSupported;
    QList<bool> m_listVisibility;
    QList<int> m_listSize;
    bool m_doctransactionTable;
    bool m_nodeTable;
    bool m_refreshBlocked;
};

#endif
