<?php
/**
 * The Auth_imap:: class provides an IMAP implementation of the Horde
 * authentication system.
 *
 * Required parameters:
 * ====================
 *  'folder'    --  The initial folder / mailbox to open.
 *                  Should be null for 'imap' and 'nntp' protocols.
 *                  DEFAULT: null
 *  'hostspec'  --  The hostname or IP address of the server.
 *                  DEFAULT: 'localhost'
 *  'port'      --  The server port to which we will connect.
 *                  IMAP is generally 143, while IMAP-SSL is generally 993.
 *                  DEFAULT: 143
 *  'protocol'  --  The connection protocol (e.g. 'imap', 'pop3', 'nntp').
 *                  Protocol is one of 'imap/notls' (or only 'imap' if you
 *                  have a c-client version 2000c or older), 'imap/ssl', or
 *                  'imap/ssl/novalidate-cert' (for a self-signed certificate).
 *                  DEFAULT: 'imap'
 *
 * Optional parameters:
 * ====================
 *  'dsn'  --  The full IMAP connection string.
 *             If not present, this is built from 'folder', 'hostspec', 'port',
 *             and 'protocol' parameters.
 *
 *
 * If setting up as Horde auth handler in conf.php, this is a sample entry:
 *   $conf['auth']['params']['folder'] = 'INBOX';
 *   $conf['auth']['params']['hostspec'] = 'imap.example.com';
 *   $conf['auth']['params']['port'] = 143;
 *   $conf['auth']['params']['protocol'] = 'imap/notls/novalidate-cert';
 *
 * 
 * $Horde: horde/lib/Auth/imap.php,v 1.17 2003/07/10 21:42:56 slusarz Exp $
 *
 * Copyright 1999-2003 Chuck Hagenbuch <chuck@horde.org>
 * Copyright 1999-2003 Gaudenz Steinlin <gaudenz.steinlin@id.unibe.ch>
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @author  Gaudenz Steinlin <gaudenz.steinlin@id.unibe.ch>
 * @version $Revision: 1.17 $
 * @since   Horde 1.3
 * @package horde.auth
 */
class Auth_imap extends Auth {

    /**
     * Constructs a new IMAP authentication object.
     *
     * @access public
     *
     * @param optional array $params  A hash containing connection parameters.
     */
    function Auth_imap($params = array())
    {
        if (!Horde::extensionExists('imap')) {
            Horde::fatal(PEAR::raiseError(_("Auth_imap: Required IMAP extension not found."), __FILE__, __LINE__));
        }

        $default_params = array(
            'folder' => null,
            'hostspec' => 'localhost',
            'port' => '143',
            'protocol' => 'imap'
        );
        $this->_params = array_merge($default_params, $params);

        /* Create DSN string. */
        if (!isset($this->_params['dsn'])) {
            $this->_params['dsn'] = '{' . $this->_params['hostspec'] . '/' . $this->_params['protocol'] . ':' . $this->_params['port'] . '}';
            if (!empty($this->_params['folder'])) {
                $this->_params['dsn'] .= $this->_params['folder'];
            }
        }
    }

    /**
     * Find out if a set of login credentials are valid.
     *
     * @access private
     *
     * @param string $userID      The userID to check.
     * @param array $credentials  An array of login credentials. For IMAP,
     *                            this must contain a password entry.
     *
     * @return boolean  Whether or not the credentials are valid.
     */
    function _authenticate($userID, $credentials)
    {
        if (!array_key_exists('password', $credentials)) {
            Horde::fatal(PEAR::raiseError(_("No password provided for IMAP authentication.")), __FILE__, __LINE__);
        }

        $imap = @imap_open($this->_params['dsn'], $userID,
                           $credentials['password'], OP_HALFOPEN);

        if ($imap) {
            @imap_close($imap);
            return true;
        } else {
            @imap_close($imap);
            $this->_setAuthError();
            return false;
        }
    }

}
