# frozen_string_literal: true

require_relative 'table'

module GitlabQuality
  module TestTooling
    module CodeCoverage
      module ClickHouse
        class CoverageMetricsTable < GitlabQuality::TestTooling::CodeCoverage::ClickHouse::Table
          TABLE_NAME = "coverage_metrics"

          private

          # @return [Boolean] True if the record is valid, false otherwise
          def valid_record?(record)
            valid_file?(record) &&
              valid_line_coverage?(record) &&
              valid_branch_coverage?(record) &&
              valid_function_coverage?(record)
          end

          # @return [Boolean] True if the file field is present
          def valid_file?(record)
            return true unless record[:file].nil?

            logger.warn("#{LOG_PREFIX} Skipping record with nil file: #{record}")
            false
          end

          # @return [Boolean] True if line_coverage is present and not NaN
          def valid_line_coverage?(record)
            if record[:line_coverage].nil?
              logger.warn("#{LOG_PREFIX} Skipping record with nil line_coverage: #{record}")
              return false
            end

            return true unless record[:line_coverage].nan?

            logger.warn("#{LOG_PREFIX} Skipping record with NaN line_coverage: #{record}")
            false
          end

          # @return [Boolean] True if branch_coverage is not NaN (or is nil)
          def valid_branch_coverage?(record)
            return true unless record[:branch_coverage]&.nan?

            logger.warn("#{LOG_PREFIX} Skipping record with NaN branch_coverage: #{record}")
            false
          end

          # @return [Boolean] True if function_coverage is not NaN (or is nil)
          def valid_function_coverage?(record)
            return true unless record[:function_coverage]&.nan?

            logger.warn("#{LOG_PREFIX} Skipping record with NaN function_coverage: #{record}")
            false
          end

          # @return [Hash] Transformed coverage data including timestamp and CI metadata
          def sanitized_data_record(record)
            {
              timestamp: time,
              file: record[:file],
              line_coverage: record[:line_coverage],
              branch_coverage: record[:branch_coverage],
              function_coverage: record[:function_coverage],
              source_file_type: record[:source_file_type],
              is_responsible: record[:is_responsible],
              is_dependent: record[:is_dependent],
              category: record[:feature_category],
              **ci_metadata
            }
          end

          # @return [Hash] CI-related metadata
          def ci_metadata
            {
              ci_project_id: env_to_int('CI_PROJECT_ID'),
              ci_project_path: ENV.fetch('CI_PROJECT_PATH', nil),
              ci_job_name: ENV.fetch('CI_JOB_NAME', nil)&.gsub(%r{ \d{1,2}/\d{1,2}}, ''),
              ci_job_id: env_to_int('CI_JOB_ID'),
              ci_pipeline_id: env_to_int('CI_PIPELINE_ID'),
              ci_merge_request_iid: env_to_int('CI_MERGE_REQUEST_IID') || env_to_int('TOP_UPSTREAM_MERGE_REQUEST_IID'),
              ci_branch: ENV.fetch('CI_COMMIT_REF_NAME', nil),
              ci_target_branch: ENV.fetch('CI_MERGE_REQUEST_TARGET_BRANCH_NAME', nil)
            }
          end

          # @param name [String] Environment variable name
          # @return [Integer, nil] Environment variable converted to integer or
          #   nil if not present or empty
          def env_to_int(name)
            value = ENV.fetch(name, nil)
            return nil if value.nil? || value.empty?

            value.to_i
          end
        end
      end
    end
  end
end
