﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ecs/ECSRequest.h>
#include <aws/ecs/ECS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ECS {
namespace Model {

/**
 */
class GetTaskProtectionRequest : public ECSRequest {
 public:
  AWS_ECS_API GetTaskProtectionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetTaskProtection"; }

  AWS_ECS_API Aws::String SerializePayload() const override;

  AWS_ECS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The short name or full Amazon Resource Name (ARN) of the cluster that hosts
   * the service that the task sets exist in.</p>
   */
  inline const Aws::String& GetCluster() const { return m_cluster; }
  inline bool ClusterHasBeenSet() const { return m_clusterHasBeenSet; }
  template <typename ClusterT = Aws::String>
  void SetCluster(ClusterT&& value) {
    m_clusterHasBeenSet = true;
    m_cluster = std::forward<ClusterT>(value);
  }
  template <typename ClusterT = Aws::String>
  GetTaskProtectionRequest& WithCluster(ClusterT&& value) {
    SetCluster(std::forward<ClusterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of up to 100 task IDs or full ARN entries.</p>
   */
  inline const Aws::Vector<Aws::String>& GetTasks() const { return m_tasks; }
  inline bool TasksHasBeenSet() const { return m_tasksHasBeenSet; }
  template <typename TasksT = Aws::Vector<Aws::String>>
  void SetTasks(TasksT&& value) {
    m_tasksHasBeenSet = true;
    m_tasks = std::forward<TasksT>(value);
  }
  template <typename TasksT = Aws::Vector<Aws::String>>
  GetTaskProtectionRequest& WithTasks(TasksT&& value) {
    SetTasks(std::forward<TasksT>(value));
    return *this;
  }
  template <typename TasksT = Aws::String>
  GetTaskProtectionRequest& AddTasks(TasksT&& value) {
    m_tasksHasBeenSet = true;
    m_tasks.emplace_back(std::forward<TasksT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_cluster;

  Aws::Vector<Aws::String> m_tasks;
  bool m_clusterHasBeenSet = false;
  bool m_tasksHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECS
}  // namespace Aws
