﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/DomainType.h>
#include <aws/ec2/model/TagSpecification.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class AllocateAddressRequest : public EC2Request {
 public:
  AWS_EC2_API AllocateAddressRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AllocateAddress"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The network (<code>vpc</code>).</p>
   */
  inline DomainType GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  inline void SetDomain(DomainType value) {
    m_domainHasBeenSet = true;
    m_domain = value;
  }
  inline AllocateAddressRequest& WithDomain(DomainType value) {
    SetDomain(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Elastic IP address to recover or an IPv4 address from an address
   * pool.</p>
   */
  inline const Aws::String& GetAddress() const { return m_address; }
  inline bool AddressHasBeenSet() const { return m_addressHasBeenSet; }
  template <typename AddressT = Aws::String>
  void SetAddress(AddressT&& value) {
    m_addressHasBeenSet = true;
    m_address = std::forward<AddressT>(value);
  }
  template <typename AddressT = Aws::String>
  AllocateAddressRequest& WithAddress(AddressT&& value) {
    SetAddress(std::forward<AddressT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of an address pool that you own. Use this parameter to let Amazon EC2
   * select an address from the address pool. To specify a specific address from the
   * address pool, use the <code>Address</code> parameter instead.</p>
   */
  inline const Aws::String& GetPublicIpv4Pool() const { return m_publicIpv4Pool; }
  inline bool PublicIpv4PoolHasBeenSet() const { return m_publicIpv4PoolHasBeenSet; }
  template <typename PublicIpv4PoolT = Aws::String>
  void SetPublicIpv4Pool(PublicIpv4PoolT&& value) {
    m_publicIpv4PoolHasBeenSet = true;
    m_publicIpv4Pool = std::forward<PublicIpv4PoolT>(value);
  }
  template <typename PublicIpv4PoolT = Aws::String>
  AllocateAddressRequest& WithPublicIpv4Pool(PublicIpv4PoolT&& value) {
    SetPublicIpv4Pool(std::forward<PublicIpv4PoolT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A unique set of Availability Zones, Local Zones, or Wavelength Zones from
   * which Amazon Web Services advertises IP addresses. Use this parameter to limit
   * the IP address to this location. IP addresses cannot move between network border
   * groups.</p>
   */
  inline const Aws::String& GetNetworkBorderGroup() const { return m_networkBorderGroup; }
  inline bool NetworkBorderGroupHasBeenSet() const { return m_networkBorderGroupHasBeenSet; }
  template <typename NetworkBorderGroupT = Aws::String>
  void SetNetworkBorderGroup(NetworkBorderGroupT&& value) {
    m_networkBorderGroupHasBeenSet = true;
    m_networkBorderGroup = std::forward<NetworkBorderGroupT>(value);
  }
  template <typename NetworkBorderGroupT = Aws::String>
  AllocateAddressRequest& WithNetworkBorderGroup(NetworkBorderGroupT&& value) {
    SetNetworkBorderGroup(std::forward<NetworkBorderGroupT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a customer-owned address pool. Use this parameter to let Amazon EC2
   * select an address from the address pool. Alternatively, specify a specific
   * address from the address pool.</p>
   */
  inline const Aws::String& GetCustomerOwnedIpv4Pool() const { return m_customerOwnedIpv4Pool; }
  inline bool CustomerOwnedIpv4PoolHasBeenSet() const { return m_customerOwnedIpv4PoolHasBeenSet; }
  template <typename CustomerOwnedIpv4PoolT = Aws::String>
  void SetCustomerOwnedIpv4Pool(CustomerOwnedIpv4PoolT&& value) {
    m_customerOwnedIpv4PoolHasBeenSet = true;
    m_customerOwnedIpv4Pool = std::forward<CustomerOwnedIpv4PoolT>(value);
  }
  template <typename CustomerOwnedIpv4PoolT = Aws::String>
  AllocateAddressRequest& WithCustomerOwnedIpv4Pool(CustomerOwnedIpv4PoolT&& value) {
    SetCustomerOwnedIpv4Pool(std::forward<CustomerOwnedIpv4PoolT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to assign to the Elastic IP address.</p>
   */
  inline const Aws::Vector<TagSpecification>& GetTagSpecifications() const { return m_tagSpecifications; }
  inline bool TagSpecificationsHasBeenSet() const { return m_tagSpecificationsHasBeenSet; }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  void SetTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications = std::forward<TagSpecificationsT>(value);
  }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  AllocateAddressRequest& WithTagSpecifications(TagSpecificationsT&& value) {
    SetTagSpecifications(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  template <typename TagSpecificationsT = TagSpecification>
  AllocateAddressRequest& AddTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications.emplace_back(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of an IPAM pool which has an Amazon-provided or BYOIP public IPv4 CIDR
   * provisioned to it. For more information, see <a
   * href="https://docs.aws.amazon.com/vpc/latest/ipam/tutorials-eip-pool.html">Allocate
   * sequential Elastic IP addresses from an IPAM pool</a> in the <i>Amazon VPC IPAM
   * User Guide</i>.</p>
   */
  inline const Aws::String& GetIpamPoolId() const { return m_ipamPoolId; }
  inline bool IpamPoolIdHasBeenSet() const { return m_ipamPoolIdHasBeenSet; }
  template <typename IpamPoolIdT = Aws::String>
  void SetIpamPoolId(IpamPoolIdT&& value) {
    m_ipamPoolIdHasBeenSet = true;
    m_ipamPoolId = std::forward<IpamPoolIdT>(value);
  }
  template <typename IpamPoolIdT = Aws::String>
  AllocateAddressRequest& WithIpamPoolId(IpamPoolIdT&& value) {
    SetIpamPoolId(std::forward<IpamPoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline AllocateAddressRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  DomainType m_domain{DomainType::NOT_SET};

  Aws::String m_address;

  Aws::String m_publicIpv4Pool;

  Aws::String m_networkBorderGroup;

  Aws::String m_customerOwnedIpv4Pool;

  Aws::Vector<TagSpecification> m_tagSpecifications;

  Aws::String m_ipamPoolId;

  bool m_dryRun{false};
  bool m_domainHasBeenSet = false;
  bool m_addressHasBeenSet = false;
  bool m_publicIpv4PoolHasBeenSet = false;
  bool m_networkBorderGroupHasBeenSet = false;
  bool m_customerOwnedIpv4PoolHasBeenSet = false;
  bool m_tagSpecificationsHasBeenSet = false;
  bool m_ipamPoolIdHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
