﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudfront-keyvaluestore/CloudFrontKeyValueStoreRequest.h>
#include <aws/cloudfront-keyvaluestore/CloudFrontKeyValueStore_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudFrontKeyValueStore {
namespace Model {

/**
 */
class GetKeyRequest : public CloudFrontKeyValueStoreRequest {
 public:
  AWS_CLOUDFRONTKEYVALUESTORE_API GetKeyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetKey"; }

  AWS_CLOUDFRONTKEYVALUESTORE_API Aws::String SerializePayload() const override;

  /**
   * Helper function to collect parameters (configurable and static hardcoded) required for endpoint computation.
   */
  AWS_CLOUDFRONTKEYVALUESTORE_API EndpointParameters GetEndpointContextParams() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Key Value Store.</p>
   */
  inline const Aws::String& GetKvsARN() const { return m_kvsARN; }
  inline bool KvsARNHasBeenSet() const { return m_kvsARNHasBeenSet; }
  template <typename KvsARNT = Aws::String>
  void SetKvsARN(KvsARNT&& value) {
    m_kvsARNHasBeenSet = true;
    m_kvsARN = std::forward<KvsARNT>(value);
  }
  template <typename KvsARNT = Aws::String>
  GetKeyRequest& WithKvsARN(KvsARNT&& value) {
    SetKvsARN(std::forward<KvsARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key to get.</p>
   */
  inline const Aws::String& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::String>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::String>
  GetKeyRequest& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_kvsARN;

  Aws::String m_key;
  bool m_kvsARNHasBeenSet = false;
  bool m_keyHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFrontKeyValueStore
}  // namespace Aws
