﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticache/ElastiCacheRequest.h>
#include <aws/elasticache/ElastiCache_EXPORTS.h>
#include <aws/elasticache/model/Tag.h>

#include <utility>

namespace Aws {
namespace ElastiCache {
namespace Model {

/**
 * <p>Represents the input of a <code>CreateCacheParameterGroup</code>
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticache-2015-02-02/CreateCacheParameterGroupMessage">AWS
 * API Reference</a></p>
 */
class CreateCacheParameterGroupRequest : public ElastiCacheRequest {
 public:
  AWS_ELASTICACHE_API CreateCacheParameterGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateCacheParameterGroup"; }

  AWS_ELASTICACHE_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICACHE_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>A user-specified name for the cache parameter group.</p>
   */
  inline const Aws::String& GetCacheParameterGroupName() const { return m_cacheParameterGroupName; }
  inline bool CacheParameterGroupNameHasBeenSet() const { return m_cacheParameterGroupNameHasBeenSet; }
  template <typename CacheParameterGroupNameT = Aws::String>
  void SetCacheParameterGroupName(CacheParameterGroupNameT&& value) {
    m_cacheParameterGroupNameHasBeenSet = true;
    m_cacheParameterGroupName = std::forward<CacheParameterGroupNameT>(value);
  }
  template <typename CacheParameterGroupNameT = Aws::String>
  CreateCacheParameterGroupRequest& WithCacheParameterGroupName(CacheParameterGroupNameT&& value) {
    SetCacheParameterGroupName(std::forward<CacheParameterGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the cache parameter group family that the cache parameter group
   * can be used with.</p> <p>Valid values are: <code>valkey8</code> |
   * <code>valkey7</code> | <code>memcached1.4</code> | <code>memcached1.5</code> |
   * <code>memcached1.6</code> | <code>redis2.6</code> | <code>redis2.8</code> |
   * <code>redis3.2</code> | <code>redis4.0</code> | <code>redis5.0</code> |
   * <code>redis6.x</code> | <code>redis7</code> </p>
   */
  inline const Aws::String& GetCacheParameterGroupFamily() const { return m_cacheParameterGroupFamily; }
  inline bool CacheParameterGroupFamilyHasBeenSet() const { return m_cacheParameterGroupFamilyHasBeenSet; }
  template <typename CacheParameterGroupFamilyT = Aws::String>
  void SetCacheParameterGroupFamily(CacheParameterGroupFamilyT&& value) {
    m_cacheParameterGroupFamilyHasBeenSet = true;
    m_cacheParameterGroupFamily = std::forward<CacheParameterGroupFamilyT>(value);
  }
  template <typename CacheParameterGroupFamilyT = Aws::String>
  CreateCacheParameterGroupRequest& WithCacheParameterGroupFamily(CacheParameterGroupFamilyT&& value) {
    SetCacheParameterGroupFamily(std::forward<CacheParameterGroupFamilyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A user-specified description for the cache parameter group.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateCacheParameterGroupRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to be added to this resource. A tag is a key-value pair. A tag
   * key must be accompanied by a tag value, although null is accepted.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateCacheParameterGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateCacheParameterGroupRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_cacheParameterGroupName;

  Aws::String m_cacheParameterGroupFamily;

  Aws::String m_description;

  Aws::Vector<Tag> m_tags;
  bool m_cacheParameterGroupNameHasBeenSet = false;
  bool m_cacheParameterGroupFamilyHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElastiCache
}  // namespace Aws
