﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/frauddetector/FraudDetectorRequest.h>
#include <aws/frauddetector/FraudDetector_EXPORTS.h>
#include <aws/frauddetector/model/Language.h>
#include <aws/frauddetector/model/Tag.h>

#include <utility>

namespace Aws {
namespace FraudDetector {
namespace Model {

/**
 */
class CreateRuleRequest : public FraudDetectorRequest {
 public:
  AWS_FRAUDDETECTOR_API CreateRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRule"; }

  AWS_FRAUDDETECTOR_API Aws::String SerializePayload() const override;

  AWS_FRAUDDETECTOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The rule ID.</p>
   */
  inline const Aws::String& GetRuleId() const { return m_ruleId; }
  inline bool RuleIdHasBeenSet() const { return m_ruleIdHasBeenSet; }
  template <typename RuleIdT = Aws::String>
  void SetRuleId(RuleIdT&& value) {
    m_ruleIdHasBeenSet = true;
    m_ruleId = std::forward<RuleIdT>(value);
  }
  template <typename RuleIdT = Aws::String>
  CreateRuleRequest& WithRuleId(RuleIdT&& value) {
    SetRuleId(std::forward<RuleIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The detector ID for the rule's parent detector.</p>
   */
  inline const Aws::String& GetDetectorId() const { return m_detectorId; }
  inline bool DetectorIdHasBeenSet() const { return m_detectorIdHasBeenSet; }
  template <typename DetectorIdT = Aws::String>
  void SetDetectorId(DetectorIdT&& value) {
    m_detectorIdHasBeenSet = true;
    m_detectorId = std::forward<DetectorIdT>(value);
  }
  template <typename DetectorIdT = Aws::String>
  CreateRuleRequest& WithDetectorId(DetectorIdT&& value) {
    SetDetectorId(std::forward<DetectorIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The rule description.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateRuleRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The rule expression.</p>
   */
  inline const Aws::String& GetExpression() const { return m_expression; }
  inline bool ExpressionHasBeenSet() const { return m_expressionHasBeenSet; }
  template <typename ExpressionT = Aws::String>
  void SetExpression(ExpressionT&& value) {
    m_expressionHasBeenSet = true;
    m_expression = std::forward<ExpressionT>(value);
  }
  template <typename ExpressionT = Aws::String>
  CreateRuleRequest& WithExpression(ExpressionT&& value) {
    SetExpression(std::forward<ExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The language of the rule.</p>
   */
  inline Language GetLanguage() const { return m_language; }
  inline bool LanguageHasBeenSet() const { return m_languageHasBeenSet; }
  inline void SetLanguage(Language value) {
    m_languageHasBeenSet = true;
    m_language = value;
  }
  inline CreateRuleRequest& WithLanguage(Language value) {
    SetLanguage(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The outcome or outcomes returned when the rule expression matches.</p>
   */
  inline const Aws::Vector<Aws::String>& GetOutcomes() const { return m_outcomes; }
  inline bool OutcomesHasBeenSet() const { return m_outcomesHasBeenSet; }
  template <typename OutcomesT = Aws::Vector<Aws::String>>
  void SetOutcomes(OutcomesT&& value) {
    m_outcomesHasBeenSet = true;
    m_outcomes = std::forward<OutcomesT>(value);
  }
  template <typename OutcomesT = Aws::Vector<Aws::String>>
  CreateRuleRequest& WithOutcomes(OutcomesT&& value) {
    SetOutcomes(std::forward<OutcomesT>(value));
    return *this;
  }
  template <typename OutcomesT = Aws::String>
  CreateRuleRequest& AddOutcomes(OutcomesT&& value) {
    m_outcomesHasBeenSet = true;
    m_outcomes.emplace_back(std::forward<OutcomesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A collection of key and value pairs.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateRuleRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateRuleRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_ruleId;

  Aws::String m_detectorId;

  Aws::String m_description;

  Aws::String m_expression;

  Language m_language{Language::NOT_SET};

  Aws::Vector<Aws::String> m_outcomes;

  Aws::Vector<Tag> m_tags;
  bool m_ruleIdHasBeenSet = false;
  bool m_detectorIdHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_expressionHasBeenSet = false;
  bool m_languageHasBeenSet = false;
  bool m_outcomesHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace FraudDetector
}  // namespace Aws
