﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/fsx/FSxRequest.h>
#include <aws/fsx/FSx_EXPORTS.h>
#include <aws/fsx/model/Tag.h>

#include <utility>

namespace Aws {
namespace FSx {
namespace Model {

/**
 */
class CopyBackupRequest : public FSxRequest {
 public:
  AWS_FSX_API CopyBackupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CopyBackup"; }

  AWS_FSX_API Aws::String SerializePayload() const override;

  AWS_FSX_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{

  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  CopyBackupRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the source backup. Specifies the ID of the backup that's being
   * copied.</p>
   */
  inline const Aws::String& GetSourceBackupId() const { return m_sourceBackupId; }
  inline bool SourceBackupIdHasBeenSet() const { return m_sourceBackupIdHasBeenSet; }
  template <typename SourceBackupIdT = Aws::String>
  void SetSourceBackupId(SourceBackupIdT&& value) {
    m_sourceBackupIdHasBeenSet = true;
    m_sourceBackupId = std::forward<SourceBackupIdT>(value);
  }
  template <typename SourceBackupIdT = Aws::String>
  CopyBackupRequest& WithSourceBackupId(SourceBackupIdT&& value) {
    SetSourceBackupId(std::forward<SourceBackupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The source Amazon Web Services Region of the backup. Specifies the Amazon Web
   * Services Region from which the backup is being copied. The source and
   * destination Regions must be in the same Amazon Web Services partition. If you
   * don't specify a Region, <code>SourceRegion</code> defaults to the Region where
   * the request is sent from (in-Region copy).</p>
   */
  inline const Aws::String& GetSourceRegion() const { return m_sourceRegion; }
  inline bool SourceRegionHasBeenSet() const { return m_sourceRegionHasBeenSet; }
  template <typename SourceRegionT = Aws::String>
  void SetSourceRegion(SourceRegionT&& value) {
    m_sourceRegionHasBeenSet = true;
    m_sourceRegion = std::forward<SourceRegionT>(value);
  }
  template <typename SourceRegionT = Aws::String>
  CopyBackupRequest& WithSourceRegion(SourceRegionT&& value) {
    SetSourceRegion(std::forward<SourceRegionT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  inline bool KmsKeyIdHasBeenSet() const { return m_kmsKeyIdHasBeenSet; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  CopyBackupRequest& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A Boolean flag indicating whether tags from the source backup should be
   * copied to the backup copy. This value defaults to <code>false</code>.</p> <p>If
   * you set <code>CopyTags</code> to <code>true</code> and the source backup has
   * existing tags, you can use the <code>Tags</code> parameter to create new tags,
   * provided that the sum of the source backup tags and the new tags doesn't exceed
   * 50. Both sets of tags are merged. If there are tag conflicts (for example, two
   * tags with the same key but different values), the tags created with the
   * <code>Tags</code> parameter take precedence.</p>
   */
  inline bool GetCopyTags() const { return m_copyTags; }
  inline bool CopyTagsHasBeenSet() const { return m_copyTagsHasBeenSet; }
  inline void SetCopyTags(bool value) {
    m_copyTagsHasBeenSet = true;
    m_copyTags = value;
  }
  inline CopyBackupRequest& WithCopyTags(bool value) {
    SetCopyTags(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CopyBackupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CopyBackupRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_sourceBackupId;

  Aws::String m_sourceRegion;

  Aws::String m_kmsKeyId;

  bool m_copyTags{false};

  Aws::Vector<Tag> m_tags;
  bool m_clientRequestTokenHasBeenSet = true;
  bool m_sourceBackupIdHasBeenSet = false;
  bool m_sourceRegionHasBeenSet = false;
  bool m_kmsKeyIdHasBeenSet = false;
  bool m_copyTagsHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace FSx
}  // namespace Aws
