﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/mediatailor/MediaTailor_EXPORTS.h>
#include <aws/mediatailor/model/AccessConfiguration.h>
#include <aws/mediatailor/model/DefaultSegmentDeliveryConfiguration.h>
#include <aws/mediatailor/model/HttpConfiguration.h>
#include <aws/mediatailor/model/SegmentDeliveryConfiguration.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace MediaTailor {
namespace Model {
class UpdateSourceLocationResult {
 public:
  AWS_MEDIATAILOR_API UpdateSourceLocationResult() = default;
  AWS_MEDIATAILOR_API UpdateSourceLocationResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_MEDIATAILOR_API UpdateSourceLocationResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>Access configuration parameters. Configures the type of authentication used
   * to access content from your source location.</p>
   */
  inline const AccessConfiguration& GetAccessConfiguration() const { return m_accessConfiguration; }
  template <typename AccessConfigurationT = AccessConfiguration>
  void SetAccessConfiguration(AccessConfigurationT&& value) {
    m_accessConfigurationHasBeenSet = true;
    m_accessConfiguration = std::forward<AccessConfigurationT>(value);
  }
  template <typename AccessConfigurationT = AccessConfiguration>
  UpdateSourceLocationResult& WithAccessConfiguration(AccessConfigurationT&& value) {
    SetAccessConfiguration(std::forward<AccessConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) associated with the source location.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  UpdateSourceLocationResult& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp that indicates when the source location was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  UpdateSourceLocationResult& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The optional configuration for the host server that serves segments.</p>
   */
  inline const DefaultSegmentDeliveryConfiguration& GetDefaultSegmentDeliveryConfiguration() const {
    return m_defaultSegmentDeliveryConfiguration;
  }
  template <typename DefaultSegmentDeliveryConfigurationT = DefaultSegmentDeliveryConfiguration>
  void SetDefaultSegmentDeliveryConfiguration(DefaultSegmentDeliveryConfigurationT&& value) {
    m_defaultSegmentDeliveryConfigurationHasBeenSet = true;
    m_defaultSegmentDeliveryConfiguration = std::forward<DefaultSegmentDeliveryConfigurationT>(value);
  }
  template <typename DefaultSegmentDeliveryConfigurationT = DefaultSegmentDeliveryConfiguration>
  UpdateSourceLocationResult& WithDefaultSegmentDeliveryConfiguration(DefaultSegmentDeliveryConfigurationT&& value) {
    SetDefaultSegmentDeliveryConfiguration(std::forward<DefaultSegmentDeliveryConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The HTTP configuration for the source location.</p>
   */
  inline const HttpConfiguration& GetHttpConfiguration() const { return m_httpConfiguration; }
  template <typename HttpConfigurationT = HttpConfiguration>
  void SetHttpConfiguration(HttpConfigurationT&& value) {
    m_httpConfigurationHasBeenSet = true;
    m_httpConfiguration = std::forward<HttpConfigurationT>(value);
  }
  template <typename HttpConfigurationT = HttpConfiguration>
  UpdateSourceLocationResult& WithHttpConfiguration(HttpConfigurationT&& value) {
    SetHttpConfiguration(std::forward<HttpConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp that indicates when the source location was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  UpdateSourceLocationResult& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The segment delivery configurations for the source location. For information
   * about MediaTailor configurations, see <a
   * href="https://docs.aws.amazon.com/mediatailor/latest/ug/configurations.html">Working
   * with configurations in AWS Elemental MediaTailor</a>.</p>
   */
  inline const Aws::Vector<SegmentDeliveryConfiguration>& GetSegmentDeliveryConfigurations() const {
    return m_segmentDeliveryConfigurations;
  }
  template <typename SegmentDeliveryConfigurationsT = Aws::Vector<SegmentDeliveryConfiguration>>
  void SetSegmentDeliveryConfigurations(SegmentDeliveryConfigurationsT&& value) {
    m_segmentDeliveryConfigurationsHasBeenSet = true;
    m_segmentDeliveryConfigurations = std::forward<SegmentDeliveryConfigurationsT>(value);
  }
  template <typename SegmentDeliveryConfigurationsT = Aws::Vector<SegmentDeliveryConfiguration>>
  UpdateSourceLocationResult& WithSegmentDeliveryConfigurations(SegmentDeliveryConfigurationsT&& value) {
    SetSegmentDeliveryConfigurations(std::forward<SegmentDeliveryConfigurationsT>(value));
    return *this;
  }
  template <typename SegmentDeliveryConfigurationsT = SegmentDeliveryConfiguration>
  UpdateSourceLocationResult& AddSegmentDeliveryConfigurations(SegmentDeliveryConfigurationsT&& value) {
    m_segmentDeliveryConfigurationsHasBeenSet = true;
    m_segmentDeliveryConfigurations.emplace_back(std::forward<SegmentDeliveryConfigurationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the source location.</p>
   */
  inline const Aws::String& GetSourceLocationName() const { return m_sourceLocationName; }
  template <typename SourceLocationNameT = Aws::String>
  void SetSourceLocationName(SourceLocationNameT&& value) {
    m_sourceLocationNameHasBeenSet = true;
    m_sourceLocationName = std::forward<SourceLocationNameT>(value);
  }
  template <typename SourceLocationNameT = Aws::String>
  UpdateSourceLocationResult& WithSourceLocationName(SourceLocationNameT&& value) {
    SetSourceLocationName(std::forward<SourceLocationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to assign to the source location. Tags are key-value pairs that you
   * can associate with Amazon resources to help with organization, access control,
   * and cost tracking. For more information, see <a
   * href="https://docs.aws.amazon.com/mediatailor/latest/ug/tagging.html">Tagging
   * AWS Elemental MediaTailor Resources</a>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  UpdateSourceLocationResult& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  UpdateSourceLocationResult& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  UpdateSourceLocationResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  AccessConfiguration m_accessConfiguration;

  Aws::String m_arn;

  Aws::Utils::DateTime m_creationTime{};

  DefaultSegmentDeliveryConfiguration m_defaultSegmentDeliveryConfiguration;

  HttpConfiguration m_httpConfiguration;

  Aws::Utils::DateTime m_lastModifiedTime{};

  Aws::Vector<SegmentDeliveryConfiguration> m_segmentDeliveryConfigurations;

  Aws::String m_sourceLocationName;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_requestId;
  bool m_accessConfigurationHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_defaultSegmentDeliveryConfigurationHasBeenSet = false;
  bool m_httpConfigurationHasBeenSet = false;
  bool m_lastModifiedTimeHasBeenSet = false;
  bool m_segmentDeliveryConfigurationsHasBeenSet = false;
  bool m_sourceLocationNameHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaTailor
}  // namespace Aws
