﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/resiliencehub/ResilienceHub_EXPORTS.h>
#include <aws/resiliencehub/model/DisruptionType.h>
#include <aws/resiliencehub/model/ResiliencyScoreType.h>
#include <aws/resiliencehub/model/ScoringComponentResiliencyScore.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace ResilienceHub {
namespace Model {

/**
 * <p>The overall resiliency score, returned as an object that includes the
 * disruption score and outage score.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/resiliencehub-2020-04-30/ResiliencyScore">AWS
 * API Reference</a></p>
 */
class ResiliencyScore {
 public:
  AWS_RESILIENCEHUB_API ResiliencyScore() = default;
  AWS_RESILIENCEHUB_API ResiliencyScore(Aws::Utils::Json::JsonView jsonValue);
  AWS_RESILIENCEHUB_API ResiliencyScore& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_RESILIENCEHUB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The score generated by Resilience Hub for the scoring component after running
   * an assessment.</p> <p>For example, if the <code>score</code> is 25 points, it
   * indicates the overall score of your application generated by Resilience Hub
   * after running an assessment.</p>
   */
  inline const Aws::Map<ResiliencyScoreType, ScoringComponentResiliencyScore>& GetComponentScore() const { return m_componentScore; }
  inline bool ComponentScoreHasBeenSet() const { return m_componentScoreHasBeenSet; }
  template <typename ComponentScoreT = Aws::Map<ResiliencyScoreType, ScoringComponentResiliencyScore>>
  void SetComponentScore(ComponentScoreT&& value) {
    m_componentScoreHasBeenSet = true;
    m_componentScore = std::forward<ComponentScoreT>(value);
  }
  template <typename ComponentScoreT = Aws::Map<ResiliencyScoreType, ScoringComponentResiliencyScore>>
  ResiliencyScore& WithComponentScore(ComponentScoreT&& value) {
    SetComponentScore(std::forward<ComponentScoreT>(value));
    return *this;
  }
  inline ResiliencyScore& AddComponentScore(ResiliencyScoreType key, ScoringComponentResiliencyScore value) {
    m_componentScoreHasBeenSet = true;
    m_componentScore.emplace(key, value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The disruption score for a valid key.</p>
   */
  inline const Aws::Map<DisruptionType, double>& GetDisruptionScore() const { return m_disruptionScore; }
  inline bool DisruptionScoreHasBeenSet() const { return m_disruptionScoreHasBeenSet; }
  template <typename DisruptionScoreT = Aws::Map<DisruptionType, double>>
  void SetDisruptionScore(DisruptionScoreT&& value) {
    m_disruptionScoreHasBeenSet = true;
    m_disruptionScore = std::forward<DisruptionScoreT>(value);
  }
  template <typename DisruptionScoreT = Aws::Map<DisruptionType, double>>
  ResiliencyScore& WithDisruptionScore(DisruptionScoreT&& value) {
    SetDisruptionScore(std::forward<DisruptionScoreT>(value));
    return *this;
  }
  inline ResiliencyScore& AddDisruptionScore(DisruptionType key, double value) {
    m_disruptionScoreHasBeenSet = true;
    m_disruptionScore.emplace(key, value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The outage score for a valid key.</p>
   */
  inline double GetScore() const { return m_score; }
  inline bool ScoreHasBeenSet() const { return m_scoreHasBeenSet; }
  inline void SetScore(double value) {
    m_scoreHasBeenSet = true;
    m_score = value;
  }
  inline ResiliencyScore& WithScore(double value) {
    SetScore(value);
    return *this;
  }
  ///@}
 private:
  Aws::Map<ResiliencyScoreType, ScoringComponentResiliencyScore> m_componentScore;

  Aws::Map<DisruptionType, double> m_disruptionScore;

  double m_score{0.0};
  bool m_componentScoreHasBeenSet = false;
  bool m_disruptionScoreHasBeenSet = false;
  bool m_scoreHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResilienceHub
}  // namespace Aws
