﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/wafv2/WAFV2Request.h>
#include <aws/wafv2/WAFV2_EXPORTS.h>
#include <aws/wafv2/model/CustomResponseBody.h>
#include <aws/wafv2/model/Rule.h>
#include <aws/wafv2/model/Scope.h>
#include <aws/wafv2/model/VisibilityConfig.h>

#include <utility>

namespace Aws {
namespace WAFV2 {
namespace Model {

/**
 */
class UpdateRuleGroupRequest : public WAFV2Request {
 public:
  AWS_WAFV2_API UpdateRuleGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateRuleGroup"; }

  AWS_WAFV2_API Aws::String SerializePayload() const override;

  AWS_WAFV2_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the rule group. You cannot change the name of a rule group after
   * you create it.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateRuleGroupRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether this is for a global resource type, such as a Amazon
   * CloudFront distribution. For an Amplify application, use
   * <code>CLOUDFRONT</code>.</p> <p>To work with CloudFront, you must also specify
   * the Region US East (N. Virginia) as follows: </p> <ul> <li> <p>CLI - Specify the
   * Region when you use the CloudFront scope: <code>--scope=CLOUDFRONT
   * --region=us-east-1</code>. </p> </li> <li> <p>API and SDKs - For all calls, use
   * the Region endpoint us-east-1. </p> </li> </ul>
   */
  inline Scope GetScope() const { return m_scope; }
  inline bool ScopeHasBeenSet() const { return m_scopeHasBeenSet; }
  inline void SetScope(Scope value) {
    m_scopeHasBeenSet = true;
    m_scope = value;
  }
  inline UpdateRuleGroupRequest& WithScope(Scope value) {
    SetScope(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the rule group. This ID is returned in the responses
   * to create and list commands. You provide it to operations like update and
   * delete.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  UpdateRuleGroupRequest& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the rule group that helps with identification. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateRuleGroupRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <a>Rule</a> statements used to identify the web requests that you want to
   * manage. Each rule includes one top-level statement that WAF uses to identify
   * matching web requests, and parameters that govern how WAF handles them. </p>
   */
  inline const Aws::Vector<Rule>& GetRules() const { return m_rules; }
  inline bool RulesHasBeenSet() const { return m_rulesHasBeenSet; }
  template <typename RulesT = Aws::Vector<Rule>>
  void SetRules(RulesT&& value) {
    m_rulesHasBeenSet = true;
    m_rules = std::forward<RulesT>(value);
  }
  template <typename RulesT = Aws::Vector<Rule>>
  UpdateRuleGroupRequest& WithRules(RulesT&& value) {
    SetRules(std::forward<RulesT>(value));
    return *this;
  }
  template <typename RulesT = Rule>
  UpdateRuleGroupRequest& AddRules(RulesT&& value) {
    m_rulesHasBeenSet = true;
    m_rules.emplace_back(std::forward<RulesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Defines and enables Amazon CloudWatch metrics and web request sample
   * collection. </p>
   */
  inline const VisibilityConfig& GetVisibilityConfig() const { return m_visibilityConfig; }
  inline bool VisibilityConfigHasBeenSet() const { return m_visibilityConfigHasBeenSet; }
  template <typename VisibilityConfigT = VisibilityConfig>
  void SetVisibilityConfig(VisibilityConfigT&& value) {
    m_visibilityConfigHasBeenSet = true;
    m_visibilityConfig = std::forward<VisibilityConfigT>(value);
  }
  template <typename VisibilityConfigT = VisibilityConfig>
  UpdateRuleGroupRequest& WithVisibilityConfig(VisibilityConfigT&& value) {
    SetVisibilityConfig(std::forward<VisibilityConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token used for optimistic locking. WAF returns a token to your
   * <code>get</code> and <code>list</code> requests, to mark the state of the entity
   * at the time of the request. To make changes to the entity associated with the
   * token, you provide the token to operations like <code>update</code> and
   * <code>delete</code>. WAF uses the token to ensure that no changes have been made
   * to the entity since you last retrieved it. If a change has been made, the update
   * fails with a <code>WAFOptimisticLockException</code>. If this happens, perform
   * another <code>get</code>, and use the new token returned by that operation. </p>
   */
  inline const Aws::String& GetLockToken() const { return m_lockToken; }
  inline bool LockTokenHasBeenSet() const { return m_lockTokenHasBeenSet; }
  template <typename LockTokenT = Aws::String>
  void SetLockToken(LockTokenT&& value) {
    m_lockTokenHasBeenSet = true;
    m_lockToken = std::forward<LockTokenT>(value);
  }
  template <typename LockTokenT = Aws::String>
  UpdateRuleGroupRequest& WithLockToken(LockTokenT&& value) {
    SetLockToken(std::forward<LockTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map of custom response keys and content bodies. When you create a rule with
   * a block action, you can send a custom response to the web request. You define
   * these for the rule group, and then use them in the rules that you define in the
   * rule group. </p> <p>For information about customizing web requests and
   * responses, see <a
   * href="https://docs.aws.amazon.com/waf/latest/developerguide/waf-custom-request-response.html">Customizing
   * web requests and responses in WAF</a> in the <i>WAF Developer Guide</i>. </p>
   * <p>For information about the limits on count and size for custom request and
   * response settings, see <a
   * href="https://docs.aws.amazon.com/waf/latest/developerguide/limits.html">WAF
   * quotas</a> in the <i>WAF Developer Guide</i>. </p>
   */
  inline const Aws::Map<Aws::String, CustomResponseBody>& GetCustomResponseBodies() const { return m_customResponseBodies; }
  inline bool CustomResponseBodiesHasBeenSet() const { return m_customResponseBodiesHasBeenSet; }
  template <typename CustomResponseBodiesT = Aws::Map<Aws::String, CustomResponseBody>>
  void SetCustomResponseBodies(CustomResponseBodiesT&& value) {
    m_customResponseBodiesHasBeenSet = true;
    m_customResponseBodies = std::forward<CustomResponseBodiesT>(value);
  }
  template <typename CustomResponseBodiesT = Aws::Map<Aws::String, CustomResponseBody>>
  UpdateRuleGroupRequest& WithCustomResponseBodies(CustomResponseBodiesT&& value) {
    SetCustomResponseBodies(std::forward<CustomResponseBodiesT>(value));
    return *this;
  }
  template <typename CustomResponseBodiesKeyT = Aws::String, typename CustomResponseBodiesValueT = CustomResponseBody>
  UpdateRuleGroupRequest& AddCustomResponseBodies(CustomResponseBodiesKeyT&& key, CustomResponseBodiesValueT&& value) {
    m_customResponseBodiesHasBeenSet = true;
    m_customResponseBodies.emplace(std::forward<CustomResponseBodiesKeyT>(key), std::forward<CustomResponseBodiesValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Scope m_scope{Scope::NOT_SET};

  Aws::String m_id;

  Aws::String m_description;

  Aws::Vector<Rule> m_rules;

  VisibilityConfig m_visibilityConfig;

  Aws::String m_lockToken;

  Aws::Map<Aws::String, CustomResponseBody> m_customResponseBodies;
  bool m_nameHasBeenSet = false;
  bool m_scopeHasBeenSet = false;
  bool m_idHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_rulesHasBeenSet = false;
  bool m_visibilityConfigHasBeenSet = false;
  bool m_lockTokenHasBeenSet = false;
  bool m_customResponseBodiesHasBeenSet = false;
};

}  // namespace Model
}  // namespace WAFV2
}  // namespace Aws
