/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.quarkus.component.univocity.parsers.it;

import java.util.Arrays;
import java.util.List;

import io.quarkus.test.junit.QuarkusTest;
import org.junit.jupiter.api.Test;

import static io.restassured.RestAssured.given;
import static org.apache.camel.quarkus.component.univocity.parsers.it.UniVocityTestHelper.asMap;
import static org.apache.camel.quarkus.component.univocity.parsers.it.UniVocityTestHelper.join;
import static org.junit.jupiter.api.Assertions.assertEquals;

@QuarkusTest
public class UniVocityFixedWidthDataFormatUnmarshalTest {

    @Test
    void shouldUnmarshalWithDefaultConfiguration() {
        String content = join("A  B  C    ", "1  2  3    ", "onetwothree");
        String url = "/univocity-parsers/unmarshal/fixed-width/default";
        List<?> unmarshalleds = given().when().body(content).get(url).then().statusCode(200).extract().as(List.class);
        assertEquals(3, unmarshalleds.size());
        assertEquals(Arrays.asList("A", "B", "C"), unmarshalleds.get(0));
        assertEquals(Arrays.asList("1", "2", "3"), unmarshalleds.get(1));
        assertEquals(Arrays.asList("one", "two", "three"), unmarshalleds.get(2));
    }

    @Test
    void shouldUnmarshalAsMap() {
        String content = join("A  B  C    ", "1  2  3    ", "onetwothree");
        String url = "/univocity-parsers/unmarshal/fixed-width/map";
        List<?> unmarshalleds = given().when().body(content).get(url).then().statusCode(200).extract().as(List.class);
        assertEquals(2, unmarshalleds.size());
        assertEquals(asMap("A", "1", "B", "2", "C", "3"), unmarshalleds.get(0));
        assertEquals(asMap("A", "one", "B", "two", "C", "three"), unmarshalleds.get(1));
    }

    @Test
    void shouldUnmarshalAsMapWithHeaders() {
        String content = join("1  2  3    ", "onetwothree");
        String url = "/univocity-parsers/unmarshal/fixed-width/mapWithHeaders";
        List<?> unmarshalleds = given().when().body(content).get(url).then().statusCode(200).extract().as(List.class);
        assertEquals(2, unmarshalleds.size());
        assertEquals(asMap("A", "1", "B", "2", "C", "3"), unmarshalleds.get(0));
        assertEquals(asMap("A", "one", "B", "two", "C", "three"), unmarshalleds.get(1));
    }

    @Test
    void shouldUnmarshalUsingIterator() {
        String content = join("A  B  C    ", "1  2  3    ", "onetwothree");
        String url = "/univocity-parsers/unmarshal/fixed-width/lazy";
        List<?> unmarshalleds = given().when().body(content).get(url).then().statusCode(200).extract().as(List.class);
        assertEquals(3, unmarshalleds.size());
        assertEquals(Arrays.asList("A", "B", "C"), unmarshalleds.get(0));
        assertEquals(Arrays.asList("1", "2", "3"), unmarshalleds.get(1));
        assertEquals(Arrays.asList("one", "two", "three"), unmarshalleds.get(2));
    }

    @Test
    void shouldUnmarshalUsingAdvancedConfiguration() {
        String content = join("!This is comment", "!This is comment too", "A__B__", "", "___D__");
        String url = "/univocity-parsers/unmarshal/fixed-width/advanced";
        List<?> unmarshalleds = given().when().body(content).get(url).then().statusCode(200).extract().as(List.class);
        assertEquals(2, unmarshalleds.size());
        assertEquals(Arrays.asList("A", "B"), unmarshalleds.get(0));
        assertEquals(Arrays.asList("N/A", "D"), unmarshalleds.get(1));
    }

}
