/*
 *
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package flex2.compiler.mxml.rep.decl;

import macromedia.asc.parser.ExpressionStatementNode;
import macromedia.asc.parser.ListNode;
import macromedia.asc.parser.Node;
import macromedia.asc.parser.NodeFactory;
import macromedia.asc.parser.StatementListNode;
import macromedia.asc.parser.ThisExpressionNode;
import java.util.HashSet;
import flex2.compiler.mxml.lang.StandardDefs;
import flex2.compiler.mxml.reflect.Type;
import flex2.compiler.mxml.rep.Model;
import flex2.compiler.mxml.rep.Vector;
import flex2.compiler.mxml.rep.init.DynamicPropertyInitializer;
import flex2.compiler.util.NameFormatter;

/**
 * This class represents an initializer for a newly declared property
 * (rather than one which preexists in a superclass).  We're reusing
 * the infrastructure of DynamicPropertyInitializer, which decouples
 * name and type from an actual reflect.Property.  This is pure
 * implementation re(ab)use though - the property itself will be
 * statically declared in the generated class.
 */
public class InitializedPropertyDeclaration extends DynamicPropertyInitializer
    implements PropertyDeclaration
{
	final boolean topLevel;

	public InitializedPropertyDeclaration(Model model, boolean topLevel, int line)
	{
		super(model.getType(), model.getId(), model, line, model.getStandardDefs());
		this.topLevel = topLevel;
	}

	public Type getLValueType()
	{
		return ((Model)value).getRepeaterLevel() > 0 ? type.getTypeTable().arrayType : type;
	}

    public String getTypeExpr()
    {
        return NameFormatter.toDot(getLValueType().getName());
    }

	public boolean getInspectable()
	{
		return ((Model)value).getInspectable();
	}

	public boolean getTopLevel()
	{
		return topLevel;
	}

	public boolean getIdIsAutogenerated()
	{
		return ((Model)value).getIdIsAutogenerated();
	}
	
	public boolean getBindabilityEnsured()
	{
		return ((Model)value).getBindabilityEnsured();
	}

	/**
	 * Note: the true case simply avoids assigning a top-level property twice, since the definition function of
	 * a declared property must also assign the property by side effect. Aside from this optimization, the else
	 * case could be generated unconditionally.
	 */
	public String getAssignExpr(String lvalueBase)
	{
		if (lvalueBase.equals("this") && hasDefinition() && ((Model)value).isDeclared())
		{
			return getValueExpr();
		}
		else
		{
			return super.getAssignExpr(lvalueBase);
		}
	}

	/**
	 * Note: the true case simply avoids assigning a top-level property twice, since the definition function of
	 * a declared property must also assign the property by side effect. Aside from this optimization, the else
	 * case could be generated unconditionally.
	 */
	public StatementListNode generateAssignExpr(NodeFactory nodeFactory,
                                                HashSet<String> configNamespaces,
                                                boolean generateDocComments,
                                                StatementListNode statementList,
                                                Node lvalueBase)
	{
		if ((lvalueBase instanceof ThisExpressionNode) &&
            hasDefinition() &&
            ((Model) value).isDeclared())
		{
			Node valueExpr = generateValueExpr(nodeFactory, configNamespaces, generateDocComments);
            ListNode list = nodeFactory.list(null, valueExpr);
            ExpressionStatementNode expressionStatement = nodeFactory.expressionStatement(list);
            return nodeFactory.statementList(statementList, expressionStatement);
		}
		else
		{
			return super.generateAssignExpr(nodeFactory, configNamespaces, generateDocComments,
                                            statementList, lvalueBase);
		}
	}
}
