/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import java.io.IOException;

import org.apache.commons.chain.Command;
import org.apache.commons.chain.Context;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;

/**
 * <p>Abstract {@link Command} base class encapsulating a unit of
 * reverse proxy processing work to be performed, whose purpose is
 * to support life cycle management with {@link #initialize()} and
 * {@link #destroy()} methods, and to examine and/or modify the state of a
 * reverse proxy transaction that is represented by a {@link ProxyContext}.
 * Individual {@link AbstractProxyCommand}s can be assembled into a {@link ProxyProcessingChain},
 * which allows them to either complete the required processing or delegate further
 * processing to the next {@link AbstractProxyCommand} in the {@link AbstractProxyCommand}.</p>
 * <p><em>Note: </em>{@link AbstractProxyCommand} implementations should be designed in a thread-safe
 * manner, suitable for inclusion in multiple {@link ProxyProcessingChain}s that might be
 * processed by different threads simultaneously.</p>
 * 
 * @see {@link Command}
 */
public abstract class AbstractProxyCommand implements Command
{

    /**
     * Initializes the proxy command before using in a proxy processing chain.
     */
    public void initialize()
    {
    }

    /**
     * {@inheritDoc}
     * <p>
     * <em>Note:</em> This method simply casts the {@link Context} argument
     * to {@link ProxyContext} and delegates the call to {@link #executeInternal(ProxyContext)}
     * which must be implemented by a concrete command class.
     * </p>
     */
    public final boolean execute(Context context) throws ReverseProxyException, IOException
    {
        return executeInternal((ProxyContext) context);
    }

    /**
     * <p>Execute a unit of processing work to be performed.  This
     * {@link AbstractProxyCommand} may either complete the required processing
     * and return <code>true</code>, or delegate remaining processing
     * to the next {@link AbstractProxyCommand} in a {@link ProxyProcessingChain} containing this
     * {@link AbstractProxyCommand} by returning <code>false</code>.
     * @param context
     * @return
     * @throws ReverseProxyException
     * @throws IOException
     */
    protected abstract boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException;

    /**
     * Invokes this method to give this proxy command a chance to dispose any resource
     * used in reverse proxy processing.
     */
    public void destroy()
    {
    }
}
