/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.Charset;

import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.portals.applications.webcontent2.rewriter.Source;

/**
 * Acts as an holder for a content rewriting Source in the form of an {@link HttpEntity}.
 */
public class HttpEntitySource implements Source
{

    /**
     * Underlying {@link HttpEntity} for this content source.
     */
    private final HttpEntity entity;

    /**
     * Character encoding to be used when {@link #getReader()} is invoked to create a
     * reader from the underlying gzipped byte stream.
     */
    private final String characterEncoding;

    /**
     * Constructs an <code>HttpEntitySource</code> for the underlying {@link HttpEntity}.
     * @param entity
     */
    public HttpEntitySource(final HttpEntity entity)
    {
        this(entity, null);
    }

    /**
     * Constructs a <code>HttpEntitySource</code> for the underlying {@link HttpEntity}
     * with the specific <code>characterEncoding</code> which can be used when {@link #getReader()}
     * is invoked from the underlying {@link HttpEntity} content stream.
     * @param entity
     * @param characterEncoding
     */
    public HttpEntitySource(final HttpEntity entity, final String characterEncoding)
    {
        this.entity = entity;
        this.characterEncoding = characterEncoding;
    }

    /**
     * {@inheritDoc}
     */
    public InputStream getInputStream() throws IOException
    {
        return entity.getContent();
    }

    /**
     * {@inheritDoc}
     */
    public Reader getReader() throws IOException
    {
        String charsetName = characterEncoding;

        if (charsetName == null)
        {
            ContentType contentType = ContentType.getOrDefault(entity);
            Charset charset = contentType.getCharset();
            charsetName = (charset != null ? charset.name() : null);
        }

        if (charsetName != null)
        {
            return new InputStreamReader(getInputStream(), charsetName);
        }
        else
        {
            return new InputStreamReader(getInputStream());
        }
    }
}
